/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package org.me.sealproject.customwidgets;

import android.content.Context;
import android.content.res.TypedArray;
import android.util.AttributeSet;
import android.util.Log;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;
import org.me.sealproject.R;

/**
 * A hashMap spinner. Functionality is to allow values and keys to be stored as separate things within spinner data. This is convenient
 * for such events as needing to display descriptive text but not wanting that value returned when retrieving such information from the display.
 * 
 * @author samuelgbeecher
 */
public class HashSpinner extends DefaultSpinner {
    
    public static final int DISPLAY_VALUES = 0;
    public static final int DISPLAY_KEYS = 1;
    public static final int DISPLAY_BOTH = 2;

    private int display;

    private ArrayList<String> values;
    private ArrayList<String> keys;

    public HashSpinner(Context context){
        this (context, null);
        setup(null);
    }

    public HashSpinner(Context context, AttributeSet attrs) {
        super (context, attrs);

        TypedArray myArray = context.obtainStyledAttributes(attrs, R.styleable.HashSpinner);

        setup(myArray);
    }

    public HashSpinner(Context context, AttributeSet attrs, int defStyle){
        super (context, attrs, defStyle);

        TypedArray myArray = context.obtainStyledAttributes(attrs, R.styleable.HashSpinner);

        setup(myArray);
    }
    
    private void setup(TypedArray attrs){
        values = new ArrayList<String>();
        keys = new ArrayList<String>();
        
        
        if(attrs != null){
            CharSequence[] keyEntries = attrs.getTextArray(R.styleable.HashSpinner_keyEntries);
            CharSequence[] valueEntries = attrs.getTextArray(R.styleable.HashSpinner_valueEntries);


            // Do not attempt to set the array if the given values are not included.
            if(keyEntries != null && valueEntries != null){
                for(int i = 0; i < keyEntries.length; i++){
                    put(keyEntries[i].toString(), valueEntries[i].toString());
                }

                refreshDisplay();
            }
            int spinnerD = attrs.getInt(R.styleable.HashSpinner_spinnerDisplay, 0);
            
            setDisplay(spinnerD);
        }
    }

    private ArrayList<String> concatStringArrayLists(ArrayList<String> a, ArrayList<String> b){
        ArrayList<String> conc = new ArrayList<String>(a.size());

        for(int i = 0; i < a.size(); i++){
            conc.add(a.get(i) + " - " + b.get(i));
        }

        return conc;
    }

     /**
     * Refresh the hash display. New values in the Hash Table will be output. New output will reflect the change of display type.
     */
    public void refreshDisplay(){
       // ArrayList<String> displayList = new ArrayList<String>();

        switch(display){
            case DISPLAY_VALUES:
                super.setSelectionList(values);
                break;
            case DISPLAY_KEYS:
                super.setSelectionList(keys);
                break;
            case DISPLAY_BOTH:
                super.setSelectionList(concatStringArrayLists(keys, values));
                break;
            default:
                super.setSelectionList(values);
                break;
        }

        //super.setSelectionList(displayList);
    }

    /**
     * Put the given value with the given key. They key is added if it does not exist. Old values are overwritten.
     * @param key
     * @param value
     */
    public void put(String key, String value){
        int index = keys.indexOf(key);
        
        if(index < 0){
            keys.add(key);
            values.add(value);
        }
        else{
            values.set(index, value);
        }
    }

    /**
     * Remove all elements from the list. Must call refreshDisplay() to update the display. 
     */
    public void removeAll(){
        keys = new ArrayList<String>();
        values = new ArrayList<String>();
    }

    /**
     * Set map to new map. Map must not be null.
     * @param map
     */
    public void setSelectionList(HashMap<String, String> map){
        removeAll();

        Set<String> keySet = map.keySet();

        for(Iterator<String> itr = keySet.iterator(); itr.hasNext(); ){
            String k = itr.next();
            put(k, map.get(k));
        }
    }

    /**
     * Set the map to the given keys and values.
     * Note: The new set is based on the given keys. Assumed matching is based on index. Key index 0 has a value at index 0.
     *          If there are duplicate keys, the old value will be overwritten.
     *
     * Input Args must not be null.
     * @param newKeys
     * @param newValues
     */
    public void setSelectionList(ArrayList<String> newKeys, ArrayList<String> newValues){
        removeAll();

        for(int i = 0; i < newKeys.size(); i++){
            put(newKeys.get(i), newValues.get(i));
        }

        refreshDisplay();
    }

    /**
     * Return the value of the current selection. 
     * @return String value
     */
    public String getSelectionValue(){
        return values.get(super.getSelectedItemPositionIgnoringDefault());
    }

    /**
     * Set the selection to the given value
     * @param value
     */
    public boolean setSelectionToValue(String value){
        int index = values.indexOf(value);

      

        switch(display){
            case DISPLAY_VALUES:
                return super.setSelection(value);

            case DISPLAY_KEYS:
                index = values.indexOf(value);
                return super.setSelection(keys.get(index));

            case DISPLAY_BOTH:
                index = values.indexOf(value);
                return super.setSelection(keys.get(index) + " - " + value);

            default:
                return super.setSelection(value);

        }
    }

    /**
     * Get the selection Key.
     * @return
     */
    public String getSelectionKey(){
        return keys.get(super.getSelectedItemPositionIgnoringDefault());
    }

    /**
     * Set the selection to the given Key
     * @param key
     */
    public boolean setSelectionToKey(String key){
        if(key == null){
            super.resetToDefaultText();
            return false;
        }
        int index = keys.indexOf(key);

//        super.setSelectionIgnoringDefault(index);
//        if(index >= 0){
//            return true;
//        }
//        else{
//            return false;
//        }
//
        switch(display){
            case DISPLAY_VALUES:
                index = keys.indexOf(key);
                return super.setSelection(values.get(index));
            case DISPLAY_KEYS:
                return super.setSelection(key);
            case DISPLAY_BOTH:
                index = keys.indexOf(key);
                return super.setSelection(key + " - " + values.get(index));
            default:
                index = keys.indexOf(key);
                return super.setSelection(values.get(index));
        }
    }

    /**
     * Set the display code to one of the codes:
     * DISPLAY_VALUES - display only the values of the hashmap
     * DISPLAY_KEYS - display only the keys of the hashmap
     * DISPLAY_BOTH - display key + " - " + value
     *
     * default is DISPLAY_VALUES.
     */
    public void setDisplay(int displayCode){
        display = displayCode;
        refreshDisplay();
    }

}
