/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package org.me.sealproject.customwidgets;

import android.content.Context;
import android.content.DialogInterface;
import android.content.res.TypedArray;

import android.util.AttributeSet;
import android.util.Log;

import android.widget.ArrayAdapter;
import android.widget.Spinner;
import android.widget.SpinnerAdapter;

import java.util.ArrayList;

import org.me.sealproject.*;

/**
 * This is merely a simple extension of the spinner control. Adds basic functionality that is in general convenient when using a spinner.
 * Initial text is always loaded at the beginning top of the spinner (index 0).
 * To be clear, all operations based on index are adjusted. I.e. index 0, when input, is treated as the first entry AFTER the default display text. Hence,
 * If no item is selected, the returned index of the selected item position will be -1 (the default text). The only way to redisplay this text after a selection
 * is with a manual call to resetDefaultTextDisplay().
 * @author samuelgbeecher
 */
public class DefaultSpinner extends Spinner {
    public static final String DEFAULT_INITIAL_DISPLAY_TEXT = "Select One";
    
    protected String initialText;
    protected ArrayAdapter<String> arrayAdapter;

    private boolean displayInitialText;
    private boolean oneOrFewerItems;

    public DefaultSpinner(Context context){
        this (context, null);
    }

    public DefaultSpinner(Context context, AttributeSet attrs) {
        super (context, attrs);
        
        TypedArray parsedAttr = null;
        
        if(attrs != null){
           parsedAttr = context.getResources().obtainAttributes(attrs, R.styleable.DefaultSpinner);
        }

        initialize(parsedAttr);
    }
    
    public DefaultSpinner(Context context, AttributeSet attrs, int defStyle){
        super (context, attrs, defStyle);

        TypedArray parsedAttr = null;

        if(attrs != null){
           parsedAttr = context.getResources().obtainAttributes(attrs, R.styleable.DefaultSpinner);
        }
        
        initialize(parsedAttr);
    }

    private void initialize(TypedArray parsedAttributes){
         String initialDisplayText = null;
         
         if(parsedAttributes != null){
             initialDisplayText = parsedAttributes.getString(R.styleable.DefaultSpinner_initialText);
        }

         if(initialDisplayText == null)
                initialText = DEFAULT_INITIAL_DISPLAY_TEXT;
         else
                initialText = initialDisplayText;

         arrayAdapter = new ArrayAdapter<String>(this.getContext(), android.R.layout.simple_spinner_item);
         arrayAdapter.setDropDownViewResource(android.R.layout.simple_spinner_dropdown_item);

         SpinnerAdapter spinnerAdapter = this.getAdapter();

         displayInitialText = parsedAttributes.getBoolean(R.styleable.DefaultSpinner_displayInitialText, true);
         
         if(spinnerAdapter == null || spinnerAdapter.isEmpty()){
             arrayAdapter.add("None");
             oneOrFewerItems = true;
         }
         else if(spinnerAdapter.getCount() == 1){
             arrayAdapter.add(spinnerAdapter.getItem(0).toString());
             oneOrFewerItems = true;
         }
         else{
             oneOrFewerItems = false;

             if(displayInitialText)
                arrayAdapter.add(initialText);
             
             for(int i = 0; i < spinnerAdapter.getCount(); i++){
                 arrayAdapter.add(spinnerAdapter.getItem(i).toString());
             }
         }

         setAdapter(arrayAdapter);
    }

    /**
     * Redisplay the default text as the current entry.
     */
    public void resetToDefaultText(){
        super.setSelection(0, true);
    }

    /**
     * Set the spinners current index to the selection name. Returns true if operation is successful. False otherwise. 
     * @param selectionName - String name of the spinner item being set to
     * @return
     */
    public boolean setSelection(String selectionName){
        int selectionPosition = arrayAdapter.getPosition(selectionName);

        if(selectionPosition >= 0){
            setSelection(selectionPosition, true);
            return true;
        }
        else{
            return false;
        }
    }

    /**
     * Get the current selection as a string value.
     * @return the selected item, or null if the selection is still on the default
     */
    @Override
    public String getSelectedItem(){
        int selectionPosition = super.getSelectedItemPosition();

        //Log.d("DefaultSpinner debug", selectionPosition + " is the selected position and list will display initial = " + displayInitial() + " and list is empty = " + isEmpty());


        if(isEmpty() || (selectionPosition <= 0 && displayInitial())){
            Log.d("DefaultSpinner debug", "returning null");
            return null;
        }
        else{
            return arrayAdapter.getItem(selectionPosition);
        }
    }

    /**
     * Set the selection of the spinner where the 0-based entry starts after the default text. 
     * @param selection
     */
    public void setSelectionIgnoringDefault(int selection){
        if(displayInitial())
            super.setSelection(selection+1);
        else
            super.setSelection(selection);
    }

    /**
     * Get the selected items position, ignoring the default value. I.e. Treat the first value after the default display text as the 0th entry. 
     * @return index of value ignoring default display text.
     */
    public int getSelectedItemPositionIgnoringDefault(){
        if(displayInitial()){
            return super.getSelectedItemPosition() - 1;
        }
        else{
            return super.getSelectedItemPosition();
        }
    }

    /**
     * Set the spinner list to the new list of strings. Old entries are replaced by new.
     * @param newList String[] of new list entries
     */
    public void setSelectionList(ArrayList<String> newList){
        if(newList.isEmpty()){
            newList.add("None");
            oneOrFewerItems = true;
        }
        else if(newList.size() == 1){
            oneOrFewerItems = true;
        }
        else{
            oneOrFewerItems = false;
        }

        // Display the initial text if the user specified to do so and the list has more than one item.
        if(displayInitial()){
            newList.add(0, initialText);
        }

        arrayAdapter = new ArrayAdapter<String>(this.getContext(), android.R.layout.simple_spinner_item, newList);
        arrayAdapter.setDropDownViewResource(android.R.layout.simple_spinner_dropdown_item);
        setAdapter(arrayAdapter);

        //Log.d("CONTROLLER DEBUG", "Selection List set for " + this);
    }

    public boolean isEmpty(){
        return arrayAdapter.getItem(0).equalsIgnoreCase("None");
    }

    /**
     * @return the displayInitialText
     */
    public boolean isDisplayInitialText() {
        return displayInitialText;
    }

    /**
     * @param displayInitialText the displayInitialText to set
     */
    public void setDisplayInitialText(boolean displayInitialTextIn) {
        displayInitialText = displayInitialTextIn;
    }

    private boolean displayInitial(){
        return displayInitialText && !oneOrFewerItems;
    }
}
