/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package org.me.sealproject.controllers.datacontrollers;

import android.content.Context;
import android.content.Intent;
import android.os.Environment;
import android.text.format.DateFormat;
import android.util.Log;

import java.io.*;
import java.util.*;

import org.me.sealproject.controllers.*;
import org.me.sealproject.sealdatatypes.*;

/**
 * @author samuelgbeecher
 */
public class SealDataController extends SealController{

    public static final String ROOT_DIRECTORY = "/Android/data/org.me.sealproject/";

    public static final String XML_FOLDER = "/sealXml";
    public static final String SEALS_XML_FILE_NAME = "seals.xml";
    public static final String ISLANDS_XML_FILE_NAME = "islands.xml";
    public static final String CODES_XML_FILE_NAME = "codes.xml";
    public static final String TAG_TYPES_XML_FILE_NAME = "tagTypes.xml";

    public static final String DATA_FOLDER = "/sealData";
    public static final String DATA_FILE = "seals.dat";

    public static final String CSV_FOLDER = "/csv";

    private File rootDirectory = null;

    private static SealDataController sharedInstance = null;

    private SealDatabaseController databaseController = null;
    private SealXmlController xmlController = null;

    private Intent customIntent;

    private ArrayList<SealIsland> islands;
    private HashMap<String, ArrayList> codes;
    private ArrayList<SealTagType> types;

    public static SealDataController getSharedInstance(Context context){
        if(sharedInstance == null){
            sharedInstance = new SealDataController(context);
        }

        return sharedInstance;
    }

    private SealDataController(Context context){
        super(context);

        LOG("Initializing Seal Data Controller");

        databaseController = new SealDatabaseController(context);
        xmlController = new SealXmlController(context);
        
        String state = Environment.getExternalStorageState();
        LOG(state);
        if (Environment.MEDIA_MOUNTED.equals(state)) {
            // Setup Routines
            setupFileSystem();
            setupInternalXml();
        }
        
        // Load internal xml
        loadXml();
        
    }

    private void setupFileSystem(){
        LOG(">>>>>> Setting up file system");
        //TOAST(directory.getAbsolutePath() + " Exists " + directory.exists() + "");
        rootDirectory = new File(Environment.getExternalStorageDirectory().getAbsolutePath() + ROOT_DIRECTORY);

        if(!rootDirectory.exists()){
            rootDirectory.mkdirs();
        }

        File xml = new File(rootDirectory.getAbsolutePath() + XML_FOLDER);
        if(!xml.exists()){
            xml.mkdir();
        }

        File data = new File(rootDirectory.getAbsolutePath() + DATA_FOLDER);
        if(!data.exists()){
            data.mkdir();
        }

        File csv = new File(rootDirectory.getAbsolutePath() + CSV_FOLDER);
        if(!csv.exists()){
            csv.mkdir();
        }
        LOG("<<<<<< File System Setup");
    }

    private void setupInternalXml(){
        LOG(">>>>>> Setting up internal xml");
        
        String xmlContent;
        File islandXmlFile = new File(rootDirectory.getAbsolutePath() + XML_FOLDER + "/" + ISLANDS_XML_FILE_NAME);

        xmlContent = null;
        if(islandXmlFile.exists()){
            LOG("Read Island Xml File");

            xmlContent = readFile(islandXmlFile, true);

            try{
                OutputStreamWriter output = new OutputStreamWriter(context.openFileOutput(ISLANDS_XML_FILE_NAME, Context.MODE_PRIVATE));
                output.write(xmlContent);
                output.close();
            }
            catch(IOException e){
                LOG("ERROR Writing file " + ISLANDS_XML_FILE_NAME);
            }
        }

        File codeXmlFile = new File(rootDirectory.getAbsolutePath() + XML_FOLDER + "/" +  CODES_XML_FILE_NAME);

        xmlContent = null;
        if(codeXmlFile.exists()){
            LOG("Read Code Xml File");
            
            xmlContent = readFile(codeXmlFile, true);

            try{
                OutputStreamWriter output = new OutputStreamWriter(context.openFileOutput(CODES_XML_FILE_NAME, Context.MODE_PRIVATE));
                output.write(xmlContent);
                output.close();
            }
            catch(IOException e){
                LOG("ERROR Writing file " + CODES_XML_FILE_NAME);
            }
        }

        File tagTypeXmlFile = new File(rootDirectory.getAbsolutePath() + XML_FOLDER + "/" +  TAG_TYPES_XML_FILE_NAME);

        xmlContent = null;
        if(tagTypeXmlFile.exists()){
            LOG("Read Tag Type Xml File");

            xmlContent = readFile(tagTypeXmlFile, true);

            try{
                OutputStreamWriter output = new OutputStreamWriter(context.openFileOutput(TAG_TYPES_XML_FILE_NAME, Context.MODE_PRIVATE));
                output.write(xmlContent);
                output.close();
            }
            catch(IOException e){
                LOG("ERROR Writing file " + TAG_TYPES_XML_FILE_NAME);
            }
        }

        LOG("<<<<<< Done setting up Internal Xml");
    }

    private String readFile(File file, boolean deleteAfterwards){
        String xml = null;
        try{
            InputStreamReader reader = new InputStreamReader(new FileInputStream(file));
            char readChar = 0;
            xml = "";
            
            while((readChar = (char)reader.read()) >= 0 && readChar < 65535){
               // LOG("Doing Stuff " + readChar + " int value = " + (int)readChar);

                xml += readChar;
            }

            LOG(xml);
            
            reader.close();

            if(deleteAfterwards){
                file.delete();
            }
        }
        catch(FileNotFoundException e){
           ERROR("File " + file.getName() + " not found", e);
            xml = null;
        }
        catch(IOException exception){
            ERROR("IOException in reading file " + file.getName(), exception);
            xml = null;
        }
        
        return xml;
    }

    private void loadXml(){
        islands = xmlController.parseIslandXml(ISLANDS_XML_FILE_NAME, true);
        codes = xmlController.parseCodeXml(CODES_XML_FILE_NAME, true);
        types = xmlController.parseTagTypeXml(TAG_TYPES_XML_FILE_NAME, true);
    }

    public SealDatabaseController getDatabaseController(){
       return databaseController;
    }

    public SealXmlController getSealXmlController(){
        return xmlController;
    }

    /**
     * This is a workaround. Custom intents cannot be passed between applications via Android's system (the original reference is not preserved).
     * Hence, this allows me to preserve the intent object for use in the next activity, if needed.
     * @param intent - the intent to be preserved
     */
    public void setCustomIntent(Intent intent){
        customIntent = intent;
    }

    public Intent getCustomIntent(){
        return customIntent;
    }

    /**
     * Load an object from the given folder and file. folder and file do not need qualifying backslashes, leave in normal format.
     * Throws IOException and ClassNotFoundException. 
     * @param folder
     * @param file
     * @return Object loaded or null if object does not exists or error occurs
     */
    public Object loadObject(String folder, String file) throws IOException, ClassNotFoundException{
        if(rootDirectory == null){
            LOG("Media Not Mounted");
            return null;
        }

        Object obj = null;
        ObjectInputStream input = null;

        LOG(rootDirectory.getAbsolutePath() + folder + "/"+ file);
        input = new ObjectInputStream(new FileInputStream(rootDirectory.getAbsolutePath() + folder + "/"+ file));
        obj = input.readObject();
        input.close();

        return obj;
        
    }

    public ArrayList loadXmlAsArrayList(String fileName){
        setupFileSystem();

        try{
            String state = Environment.getExternalStorageState();
            LOG(state);
            if (Environment.MEDIA_MOUNTED.equals(state)) {
                rootDirectory = new File(Environment.getExternalStorageDirectory().getAbsolutePath() + ROOT_DIRECTORY);
                
                File file = new File(rootDirectory.getAbsolutePath() + XML_FOLDER + "/"+ fileName);
                
                if(file.exists() && file.canRead()){
                    ArrayList list = xmlController.parseSealXml(rootDirectory.getAbsolutePath() + XML_FOLDER + "/"+ fileName, false);
                    return list;
                }
                else
                    return null;
                
            }else{
                LOG("Media Is Either Not Connected To Phone Correctly Or Is Mounted to a Computer");
                return null;
            }
        }
        catch(Exception e){
           // LOG(e + "");
            Log.e(DEBUG, "Exception IN Parsing", e);

            return null;
        }
    }

    public boolean deleteXmlFile(String file){
        try{
            String state = Environment.getExternalStorageState();
            if (Environment.MEDIA_MOUNTED.equals(state)) {
                rootDirectory = new File(Environment.getExternalStorageDirectory().getAbsolutePath() + ROOT_DIRECTORY);

                File xmlFile = new File(rootDirectory.getAbsolutePath() + XML_FOLDER + "/" + file);

                return xmlFile.delete();
            }
            else{
                LOG("Media not mounted. File Not deleted");
                return false;
            }
        }
        catch(Exception e){
            LOG("Error Accessing Memory Card. File not deleted");
            return false;
        }
    }

    public boolean exportCSV(long unixTimeStampInSeconds){
        databaseController.openOrCreateDatabase();

        SealObservation[] observations = databaseController.getObservationsOnDate(unixTimeStampInSeconds);
        SealCount[] counts = databaseController.getCountsOnDate(unixTimeStampInSeconds);

        databaseController.close();

        boolean successfulWriting = true;

        Date date = new Date(unixTimeStampInSeconds);
        if(observations != null){
            successfulWriting = writeObservationCSV(observations, date);
        }
        else{
            //TODO: Report to user if no observations are available to export
            successfulWriting = false;

            LOG("No observations to export for given date");
        }

        if(counts != null){
            if(!successfulWriting)
                writeCountCSV(counts, date);
            else
                successfulWriting = writeCountCSV(counts, date);
        }
        else{
            successfulWriting = false;
             //TODO: Report to user if no counts are available to export
            LOG("No Counts to export for given date");
        }

        return successfulWriting;
    }

    private boolean writeObservationCSV(SealObservation[] observations, Date date){
        String csvName = observations[0].getObserver() +"_"+ DateFormat.format("MM_dd_yy", date).toString() + "_observations.csv";
        try{
            String state = Environment.getExternalStorageState();
            if (Environment.MEDIA_MOUNTED.equals(state)) {
                rootDirectory = new File(Environment.getExternalStorageDirectory().getAbsolutePath() + ROOT_DIRECTORY);

                File csvFile = new File(rootDirectory.getAbsolutePath() + CSV_FOLDER + "/" + csvName);

                OutputStreamWriter writer = new OutputStreamWriter(new FileOutputStream(csvFile), "UTF-8");

                for(int i = 0; i < observations.length; i++){
                    writer.write(observations[i].toCSV());
                }

                writer.close();
                LOG("Observation CSV Exported");
                return true;

            }
            else{

                LOG("Media not mounted. CSV For Observations Not Exported");
                return false;

            }
        }
        catch(Exception e){
            ERROR("Error Accessing Environment. CSV For Observations Not exported", e);
            return false;
        }
    }

    private boolean writeCountCSV(SealCount[] counts, Date date){
        String csvName = counts[0].getObserver() +"_"+ DateFormat.format("MM_dd_yy", date).toString() + "_counts.csv";

         try{
            String state = Environment.getExternalStorageState();
            if (Environment.MEDIA_MOUNTED.equals(state)) {
                rootDirectory = new File(Environment.getExternalStorageDirectory().getAbsolutePath() + ROOT_DIRECTORY);

                File csvFile = new File(rootDirectory.getAbsolutePath() + CSV_FOLDER + "/" + csvName);

                OutputStreamWriter writer = new OutputStreamWriter(new FileOutputStream(csvFile), "UTF-8");

                for(int i = 0; i < counts.length; i++){
                    writer.write(counts[i].toCSV());
                }
                
                writer.close();


                CSVRecord record = getCountTotalsCSV(counts);
                
                csvName = counts[0].getObserver() +"_"+ DateFormat.format("MM_dd_yy", date).toString() + "_totalCounts.csv";
                csvFile = new File(rootDirectory.getAbsolutePath() + CSV_FOLDER + "/" + csvName);
                
                writer = new OutputStreamWriter(new FileOutputStream(csvFile), "UTF-8");
                writer.write(record.getCSV());
                writer.close();

                LOG("Count CSV Exported");

                return true;

            }
            else{
                LOG("Media not mounted. CSV For Observations Not Exported");
                return false;

            }
        }
        catch(Exception e){
            LOG("Error Accessing Environment. CSV For Observations Not exported");
            return false;
        }
    }

    private CSVRecord getCountTotalsCSV(SealCount[] counts){
        long startTime, endTime;
        int loneBull = 0, terrBull=0, haremBull=0, female=0, pup=0;
        int deadAdult=0, deadBull=0, deadFemale=0, deadPup=0;

        startTime = counts[0].getStartEntryTime();
        endTime = counts[0].getStartEntryTime();

        for(int i = 0; i < counts.length; i++){
            
            loneBull += counts[i].getLiveLoneBullCount();
            terrBull += counts[i].getLiveTerrBullCount();
            haremBull += counts[i].getLiveHaremBullCount();
            female += counts[i].getLiveFemaleCount();
            pup += counts[i].getLivePupCount();
            
            deadAdult += counts[i].getDeadNonAdolescentCount();
            deadBull += counts[i].getDeadBullCount();
            deadFemale += counts[i].getDeadFemaleCount();
            deadPup += counts[i].getDeadPupCount();

            if(startTime > counts[i].getStartEntryTime()){
                startTime = counts[i].getStartEntryTime();
            }

            if(endTime < counts[i].getEndEntryTime()){
                endTime = counts[i].getEndEntryTime();
            }
        }
        
        Date date = new Date(startTime*1000);

        CSVRecord record = new CSVRecord();
        record.addItem(DateFormat.format("MM/dd", date).toString());

        record.addItem(loneBull+"");
        record.addItem(terrBull+"");
        record.addItem(haremBull+"");
        record.addItem(female+"");
        record.addItem(pup+"");

        record.addBlank();
        record.addBlank();
        record.addBlank();
        record.addBlank();

        record.addItem(deadAdult+"");
        record.addItem(deadBull+"");
        record.addItem(deadFemale+"");
        record.addItem(deadPup+"");

        return record;
    }


    /**
     * @return the islands
     */
    public ArrayList<SealIsland> getIslands() {
        return islands;
    }

    /**
     * @return the codes
     */
    public HashMap<String, ArrayList> getCodes() {
        return codes;
    }

    /**
     * @return the types
     */
    public ArrayList<SealTagType> getTypes() {
        return types;
    }


    
}
