/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package org.me.sealproject.controllers.activitycontrollers;

import org.me.sealproject.activities.sealintents.SealValidationIntent;
import android.content.*;
import android.os.Bundle;
import android.text.*;
import android.view.View;
import android.view.View.OnClickListener;
import android.widget.*;
import android.widget.AdapterView.OnItemSelectedListener;
import android.widget.RadioGroup.OnCheckedChangeListener;

import java.util.*;

import org.me.sealproject.activities.*;
import org.me.sealproject.*;
import org.me.sealproject.activities.sealintents.SealIntent;
import org.me.sealproject.controllers.datacontrollers.*;
import org.me.sealproject.customwidgets.*;
import org.me.sealproject.sealdatatypes.*;

/**
 * Controls SealActivityObservation. 
 * @author samuelgbeecher
 */

public class SealObservationActivityController extends SealActivityController {

    public static final String PRIVATE_PREF_KEY_VISIBILITY = "visibility";
    public static final String PRIVATE_PREF_KEY_SEALNUMBER = "seal number";
    public static final String BUNDLE_NAME = "myBundle";

    public static final String CODE_VISIBILITY = "Visibility";
    public static final String CODE_PUP = "Pup";
    public static final String CODE_LOSS = "Loss";

    public static final int VALIDATION_ACTIVITY_REQUEST_CODE = 0;
    public static final int LOOKUP_ACTIVITY_REQUEST_CODE = 1;


    private SealObservation currentObs;
    private HashMap<Integer, SealObservation> observationsDictionary;

    private int highestEntry;

    private SealObservationActivity activity;
    private SealDatabaseController database;

    // Setting formChanged to true will force a revalidation of the form.
    private boolean formChanged, dataValidated;

    public SealObservationActivityController(SealObservationActivity act){
        super(act);
        activity = act;
        observationsDictionary = new HashMap<Integer, SealObservation>();
        currentObs = new SealObservation();

        
        initObservation();
        setupListeners();
        setupRookerySections(currentObs.getIsland(), currentObs.getRookery());
        setupCodes();

        LOG("SEtting Types");
        setupTypes();

        database = dataController.getDatabaseController();

        loadPreviousData();
        
        formChanged = false;
        dataValidated = false;

        //demoVariables();
    }


    /****************************************************************************************
                                            SETUP METHODS
    ****************************************************************************************/

    private void initObservation(){
         if(preferences.contains(activity.getResources().getString(R.string.preference_key_observer))){
            currentObs = getObservationWithDefaults();

            activity.setObserver(currentObs.getObserver());
            activity.setIsland(currentObs.getIsland());
            activity.setRookery(currentObs.getRookery());

        }
        else{
            TOAST_USER("Must Login", activity);
            activity.startActivity(new Intent(activity, SealLoginActivity.class));
        }
    }

    private void setupRookerySections(String islandName, String rookeryName){
        ArrayList<SealIsland> islands = dataController.getIslands();

        if(islands != null){
            SealIsland currentIsland = null;
            for(SealIsland island : islands){

                if(island.getName().equalsIgnoreCase(islandName)){
                    currentIsland = island;
                    break;
                }
            }

            ArrayList<String> sections = currentIsland.getSections(rookeryName);


            DefaultSpinner sectionSpinner = (DefaultSpinner)activity.findViewById(R.id.observation_section_spinner);
            sectionSpinner.setSelectionList(new ArrayList<String>(sections));
        }
    }

    private  void setupCodes(){
        HashMap<String, ArrayList> codes = dataController.getCodes();

        LOG("Setting up codes");
        if(codes != null && !codes.isEmpty()){
            LOG("Loading Codes");

            ArrayList<String> keys = new ArrayList<String>();
            ArrayList<String> values = new ArrayList<String>();

            // Visibility Codes
            ArrayList<SealCode> visibilityCodes = (ArrayList<SealCode>)codes.get(CODE_VISIBILITY);

            for(SealCode code : visibilityCodes){
                keys.add(code.getValue());
                values.add(code.getDescription());
            }

            HashSpinner visSpinner = ((HashSpinner)activity.findViewById(R.id.observation_visibility_spinner));
            visSpinner.setSelectionList(keys, values);
            visSpinner.refreshDisplay();

            // Pup Codes
            ArrayList<SealCode> pupCodes = (ArrayList<SealCode>)codes.get(CODE_PUP);

            keys = new ArrayList<String>();
            values = new ArrayList<String>();
            for(SealCode code : pupCodes){
                keys.add(code.getValue());
                values.add(code.getDescription());
            }
            
            HashSpinner pupSpinner = ((HashSpinner)activity.findViewById(R.id.observation_seal_pup_code_spinner));
            pupSpinner.setSelectionList(keys, values);


            // Loss Codes
            ArrayList<SealCode> lossCodes = (ArrayList<SealCode>)codes.get(CODE_LOSS);

            keys = new ArrayList<String>();
            values = new ArrayList<String>();
            for(SealCode code : lossCodes){
                keys.add(code.getValue());
                values.add(code.getDescription());
            }

            ComboSpinner topLossSpinner = ((ComboSpinner)activity.findViewById(R.id.observation_tag_type_spinner_top));
            topLossSpinner.setBottomSelectionList(keys, values);

            ComboSpinner bottomLossSpinner = ((ComboSpinner)activity.findViewById(R.id.observation_tag_type_spinner_bottom));
            bottomLossSpinner.setBottomSelectionList(keys, values);
            
        }
    }

    private void setupTypes(){
         ArrayList<SealTagType> types = dataController.getTypes();

         if(types != null){
            ArrayList<String> keys = new ArrayList<String>();
            ArrayList<String> values = new ArrayList<String>();

            for(SealTagType type : types){
                keys.add(type.getName());
                values.add(type.getDescription());
            }

            ComboSpinner topTypes = ((ComboSpinner)activity.findViewById(R.id.observation_tag_type_spinner_top));
            topTypes.setTopSelectionList(keys, values);

            ComboSpinner bottomTypes = ((ComboSpinner)activity.findViewById(R.id.observation_tag_type_spinner_bottom));
            bottomTypes.setTopSelectionList(keys, values);

            activity.setTypes(types);
         }
    }

    private void setupListeners(){
        ((EditText)activity.findViewById(R.id.observation_seal_number_autocomplete)).addTextChangedListener(new SealNumberTextWatcher());

        activity.findViewById(R.id.observation_tag_other_seen_checkbox).setOnClickListener(new OnOtherFlipperClick());
        activity.findViewById(R.id.observation_tag_flipper_seen_checkbox).setOnClickListener(new OnOtherFlipperClick());
        
        ((RadioGroup)activity.findViewById(R.id.observation_seal_sex_group)).setOnCheckedChangeListener(new OnSexChanged());
    }
    
    /**
     * Get previous observations from database and upload it to the database lookup list
     */
    private void loadPreviousData(){
        database.openOrCreateDatabase();

        SealObservation[] observations = database.getObservationsOnDate(TIME());

        // Retrieve previously entered observations
        Calendar cal = Calendar.getInstance();
        String today = cal.get(Calendar.YEAR) + "/" + cal.get(Calendar.MONTH) + "/" + cal.get(Calendar.DAY_OF_MONTH);
        if(observations != null){
            LOG("Observations Loaded");
            for(int i = 0; i < observations.length; i++){
                observationsDictionary.put(new Integer(observations[i].getSealNum()), observations[i]);
                if(observations[i].getSealNum() >= highestEntry){
                    highestEntry = observations[i].getSealNum() + 1;
                }

                LOG("Today's Date is " + today);

                cal.setTimeInMillis(observations[i].getDatetime()*1000);

                LOG("Database Observation Date is " + cal.get(Calendar.YEAR) + "/" + cal.get(Calendar.MONTH) + "/" + cal.get(Calendar.DAY_OF_MONTH));

            }
            activity.setSealNumber(highestEntry);
        }
        else{
            LOG("Observations Not Loaded");
        }
        

//        String[] values = database.getAllTagNumbers();
//        ArrayList<String> list = null;
//        if(values != null){
//            list = new ArrayList<String>(values.length);
//
//            for(int i = 0; i < values.length; i++){
//                list.add(values[i]);
//            }
//        }
//
//        ((TextBox)activity.findViewById(R.id.observation_tag_number_autocomplete_top)).setAdapter(list);
//        ((TextBox)activity.findViewById(R.id.observation_tag_number_autocomplete_bottom)).setAdapter(list);

        database.close();
    }

    /****************************************************************************************
                                    CONVENIENCE
    ****************************************************************************************/

    /**
     * Return SealObservation with defaults of observer, island, and rookery set.
     * SealObservation visibility and sealNumber are set based on previous data in the form.
     */
    private SealObservation getObservationWithDefaults(){
        SealObservation obs = new SealObservation();

        obs.setObserver(preferences.getString(activity.getResources().getString(R.string.preference_key_observer), ""));
        obs.setIsland(preferences.getString(activity.getResources().getString(R.string.preference_key_island), ""));
        obs.setRookery(preferences.getString(activity.getResources().getString(R.string.preference_key_rookery), ""));

        return obs;
    }


    /****************************************************************************************
    *                                SUBMISSION                                             *
    ****************************************************************************************/

    /**
     * Handles the on submit click for the activity
     */
    public void onSubmit(){
        SealObservation obs = activity.getObservation();
        obs.setId(currentObs.getId()); // Retain the observation id

        // after submit, data must be declared invalid to be changed. 
        dataValidated = true;
        
        // If it is a different observation, revalidate tag numbers
        if(!currentObs.equals(obs)){
            formChanged = true;
            LOG("Observation Form Changed");
            currentObs = obs;
        }

        currentObs.setDatetime(TIME());

        // Check the observation form input. This to make sure all the appropriate fields have been entered.
        if(formChanged){

            if(!validateObservationForm(currentObs)){
                dataValidated = false;
                LOG("Form is not  valid");
            }
            else if(!validateObservationData(currentObs)){
                dataValidated = false;
                LOG("Form and Data are valid");
            }
            else{
                dataValidated = true;
            }

            formChanged = false;
        }
        
        if(dataValidated && !formChanged)
            insertObservationIntoDatabase();
    }

    /**
     * Attempt to insert the data into the database. Displays appropriate success/error message.
     */
    public void insertObservationIntoDatabase(){
        database.openOrCreateDatabase();

        String successMessage;
        
        boolean insertGood;
        
        if(observationsDictionary.containsKey(currentObs.getSealNum())){
            insertGood = dataController.getDatabaseController().updateObservation(currentObs);
            successMessage = "Observation Modified";
        }
        else{
            insertGood = dataController.getDatabaseController().insertObservation(currentObs);
            successMessage = "Observation Submitted";
        }

        if(insertGood){

            LOG("Inserted Into Database");

            //Update lastseen
            if(currentObs.getSeal().getId() != null){
                LOG("Updating seal last seen");
                database.updateSeal(currentObs.getSeal());
            }

            if(highestEntry <= currentObs.getSealNum()){
                highestEntry = currentObs.getSealNum()+1;
            }

            observationsDictionary.remove(currentObs.getSealNum());
            observationsDictionary.put(currentObs.getSealNum(), currentObs);

            currentObs = new SealObservation();

            activity.resetForm();
            activity.setSealNumber(highestEntry);

            TOAST_USER(successMessage, activity);

            View v = activity.findViewById(R.id.observation_seal_number_autocomplete);
            v.setFocusableInTouchMode(true);
            v.requestFocus();

            // Reset the state variables
            formChanged = true;
            dataValidated = false;
        }
        else{
          TOAST_USER("Observation Not Submitted. There was an Error", activity);
          formChanged = false;
        }

        database.close();
        
    }

    /****************************************************************************************
                                    VALIDATION
    ****************************************************************************************/

    /**
     * Validate the observation form. Certain data must not be null for it to be inserted into the database. Requires observer to enter all valid data, ensuring
     * data entry is complete
     * @param obs
     * @return
     */
    private boolean validateObservationForm(SealObservation obs){
        boolean valid = false;

        Seal seal = obs.getSeal();
        SealTag firstTag = seal.getFirstTag();
        SealTag secondTag = seal.getSecondTag();

        if(obs.getRookerySection() == null){
            displayErrorMessage("Must Select Rookery Section", R.id.observation_section_spinner, true);
        }
        else if(obs.getVisibilityCode() < 0){
            displayErrorMessage("Must Select Visibility", R.id.observation_visibility_spinner, true);
        }
        else if(obs.getSealNum() < 0){
            displayErrorMessage("Must Enter Seal Number", R.id.observation_seal_number_autocomplete, true);
        }
        else if(firstTag.getSide() == null){
            displayErrorMessage("Must Select First Tag Side", R.id.observation_tag_side_group_top, true);
        }
        else if(firstTag.getLossCode() < 0 && firstTag.getType() == null) {
            displayErrorMessage("Must Select First Tag Type or Loss Code", R.id.observation_tag_type_spinner_top, true);
        }
        else if(firstTag.getLossCode() < 0 && firstTag.getColor() == null && !activity.firstColorIsEmpty()){
            displayErrorMessage("Must Select First Tag Color", R.id.observation_tag_color_spinner_top, true);
        }
        else if((obs.isOtherFlipperSeen() || obs.isOtherTagSeen()) && secondTag.getLossCode() < 0 && secondTag.getType() == null) {
            displayErrorMessage("Must Select Second Tag Type or Loss Code", R.id.observation_tag_type_spinner_bottom, true);
        }
        else if((obs.isOtherFlipperSeen() || obs.isOtherTagSeen()) && secondTag.getLossCode() < 0 && secondTag.getColor() == null && !activity.secondColorIsEmpty()){
            displayErrorMessage("Must Select Second Tag Color", R.id.observation_tag_color_spinner_bottom, true);
        }
        else{
            // All's good.
            valid = true;
        }

        return valid;
    }

    /**
     * Perform validation on the observation data as per the database.
     * @param observation - Observation to be validated
     * @return code for validation
     */
    private boolean validateObservationData(SealObservation observation){
        LOG("Validating Observation Data");

        boolean specialCase = false;

        String confirmationMessage = "Please Reconfirm:\n\n";

        ArrayList<SealValidationRow> comparisonRows = new ArrayList<SealValidationRow>();

        // Retrieve database data regarding seal
        Seal seal = observation.getSeal();
        SealTag first = seal.getFirstTag();
        SealTag second = seal.getSecondTag();

        Seal dbSeal = null;
        SealTag dbFirst = null;
        SealTag dbSecond = null;

        dbSeal = getSealWithTag(first);
        if(dbSeal != null) // First tag exists in the database
        {
            seal.setId(dbSeal.getId());

            dbFirst = dbSeal.getFirstTag();
            dbSecond = dbSeal.getSecondTag();

            // Match the first tag with the appropriate first tag
            // Handles two cases, the first is where both tags on the sael are not the same
            //                    the second is where both tags are the same and so the sides need to be matched
            if(!dbFirst.getNumber().equalsIgnoreCase(first.getNumber()) ||
                    (dbSecond != null && dbSecond.getNumber().equalsIgnoreCase(dbFirst.getNumber()) && dbSecond.getSide().equalsIgnoreCase(first.getSide())))
            {
                // Swap
                SealTag fTmp = dbSeal.getFirstTag();
                SealTag sTmp = dbSeal.getSecondTag();

                dbSeal.setFirstTag(sTmp);
                dbSeal.setSecondTag(fTmp);

                dbFirst = dbSeal.getFirstTag();
                dbSecond = dbSeal.getSecondTag();
            }

            comparisonRows.addAll(compareSealTags(first, dbFirst, "1st"));

            if(second != null) // Second tag was entered
            {
                if(dbSecond != null)
                    comparisonRows.addAll(compareSealTags(second, dbSecond, "2nd"));
            }
            else
            {
                // Do nothing
            }
            
        }
        else if(second != null) // Second tag was entered into the form as well
        {
            dbSeal = getSealWithTag(second);
            if(dbSeal != null) // Second tag is in the database while the first wasn't
            {
                dbSecond = dbSeal.getSecondTag();
                dbFirst = dbSeal.getFirstTag();

                if(!dbSecond.getNumber().equalsIgnoreCase(second.getNumber()) ||
                        (dbFirst.getNumber().equalsIgnoreCase(dbSecond.getNumber()) && dbFirst.getSide().equalsIgnoreCase(second.getSide())))
                {
                    // dbSecond and second are mismatched by either tagNumber or tagSide.

                    // Swap
                    SealTag fTmp = dbSeal.getFirstTag();
                    SealTag sTmp = dbSeal.getSecondTag();

                    dbSeal.setFirstTag(sTmp);
                    dbSeal.setSecondTag(fTmp);

                    dbSecond = dbSeal.getSecondTag();
                    dbFirst = dbSeal.getFirstTag();
                }

                comparisonRows.addAll(compareSealTags(first, dbFirst, "1st"));
                comparisonRows.addAll(compareSealTags(second, dbSecond, "2nd"));
            }
            else
            {
                 confirmationMessage += "Both tags do not exist in database.";
                 specialCase = true;
                // Special case
            }
        }
        else
        {
            // Special case
            confirmationMessage += "First tag does not exist in database.";
            specialCase = true;
        }

        if(dbSeal != null)
        {
            long timeInSeconds  = dbSeal.getLastSeen();
            Calendar dbLastSeen = Calendar.getInstance();
            dbLastSeen.setTimeInMillis(timeInSeconds * 1000);

            int dbYear = dbLastSeen.get(Calendar.YEAR);

            Calendar calendar = Calendar.getInstance();
            long currYear = calendar.get(Calendar.YEAR);

            if(dbYear < currYear)
            {
                seal.setLastSeen(timeInSeconds);
                confirmationMessage += "First sighting this year";
                specialCase = true;
            }

            // Only check seal sex if the value is not set.
            if(seal.getSex().toUpperCase().charAt(0) != 'U'){
                SealValidationRow row = validate(seal.getSex().toUpperCase().charAt(0) + "",
                                                   dbSeal.getSex().toUpperCase().charAt(0) + "",
                                                   "Seal sex does NOT match database",
                                                   SealValidationActivityController.SEAL_SEX_MISMATCH);

                if(row != null){
                    comparisonRows.add(row);
                }
            }
        }

        // Would be nice to farm out this to the calling method. But not clear on how to do it.
        if(!comparisonRows.isEmpty())
        {
            this.startValidationActivity(seal, dbSeal, comparisonRows);
        }
        else if(specialCase)
        {
            super.displayConfirmationDialog(confirmationMessage, -1, false);
        }

        boolean dataGood = !specialCase && comparisonRows.isEmpty();
        
        return dataGood;
    }

    private ArrayList<SealValidationRow> compareSealTags(SealTag target, SealTag comparison, String whichTag){
        ArrayList<SealValidationRow> validationRows = new ArrayList<SealValidationRow>();
        SealValidationRow row;
        String prefix;
        int offset;

        if(target == null || comparison == null){
            return null;
        }

        // TODO: This is ugly code, but it does work.
        // Offset is based on the SealValidationActivityController, the difference between 2nd tag error codes and 1st tag is 10. Hence, the offset if it is first or second.
        // Could create a second method, but I would rather find a better way to pass this information to my validation activity.
        // If time permits, will need to work more on this design. 
        if(whichTag.equalsIgnoreCase("1st")){
            prefix = "1st";
            offset = 0;
        }
        else{
            prefix = "2nd";
            offset = 10;
        }


        if(target.isLost() && !comparison.isLost())
        {
            
           row = new SealValidationRow(context,
                   prefix+" tag does NOT show as lost in database",
                   target.getLossCode()+"",
                   comparison.getType());

           row.setValidationType(SealValidationActivityController.FIRST_NOT_LOST + offset);

           validationRows.add(row);
        }
        else if(!target.isLost())
        {   
            if(comparison.isLost())// If the database tag is marked as lost, then no reason to check all other data
            {
                 row = new SealValidationRow(context,
                         prefix+" tag shows as lost in database",
                         target.getType()+"",
                         comparison.getLossCode() + "");

                 row.setValidationType(SealValidationActivityController.FIRST_LOST + offset);

                 validationRows.add(row);
            }
            else
            { 

                row = validate(target.getSide().toUpperCase()+"",
                        comparison.getSide().toUpperCase(), 
                        prefix+" tag side does NOT match database",
                        SealValidationActivityController.FIRST_SIDE_MISMATCH + offset);
                
                if(row != null)
                {
                    validationRows.add(row);
                }

                row = validate((target.getType() != null ? target.getType().toUpperCase() : null),
                        (comparison.getType() != null ? comparison.getType().toUpperCase() : null),
                        prefix+" tag type does NOT match database",
                        SealValidationActivityController.FIRST_TYPE_MISMATCH + offset);
                
                if(row != null)
                {
                    validationRows.add(row);
                }

                row = validate((target.getColor() != null ? target.getColor().toUpperCase() : null),
                        (comparison.getColor() != null ? comparison.getColor().toUpperCase() : null),
                        prefix+" tag color does NOT match database",
                        SealValidationActivityController.FIRST_COLOR_MISMATCH + offset);

                if(row != null)
                {
                    validationRows.add(row);
                }

                // Do not check the tag number if it was not entered
                if(target.getNumber() != null && !target.getNumber().trim().equals("")){
                    row = validate( target.getNumber(),
                                    comparison.getNumber(),
                                    prefix+" tag number does NOT match database",
                                    SealValidationActivityController.FIRST_NUMBER_MISMATCH + offset);

                    if(row != null)
                    {
                        validationRows.add(row);
                    }
                }
            }
        }

        return validationRows;
    }

    private Seal getSealWithTag(SealTag tag){
        database.openOrCreateDatabase();
        Seal seal = null;

        if(database.isOpen())
            seal = database.getSealWithTag(tag);
        
        database.close();

        return seal;
    }

    /**
     * Start the seal validation activity.
     * @param target - the target seal, or seal that is going to be modified.
     * @param comparison - the comparison seal that will be checked against the target.
     * @param rows - the number SealValidationRows that will be included. 
     */
    private void startValidationActivity(Seal target, Seal comparison, ArrayList<SealValidationRow> rows){
        SealValidationIntent intent = new SealValidationIntent();

        LOG(rows.size() + "");
        intent.setList(rows);
//        intent.putExtra(SealActivityValidationController.INPUT_VALIDATION_ROWS_KEY, rows.toArray());
        intent.setTargetSeal(target);
        intent.setCompSeal(comparison);

        Bundle bundle = new Bundle();

        // Preserve the intent for the future activity
        dataController.setCustomIntent(intent);


        activity.startActivityForResult(new Intent(context, SealValidationActivity.class), VALIDATION_ACTIVITY_REQUEST_CODE);
        
    }

     /**
     * Create and return a validation row based on comparison of the two given objects.
     * This is a convenience method
     * @param a
     * @param b
     * @param text
     * @param validationType
     * @return
     */
    private SealValidationRow validate(Object a, Object b, String text, int validationType){
        if(a == b){
            return null;
        }else if(a != null && a.equals(b)){
            return null;
        }

        SealValidationRow row = new SealValidationRow(context, text, ( a != null ? a.toString() : null ), (b != null ? b.toString() : null));
        row.setValidationType(validationType);

        return row;
    }


    private void setVisible(boolean visible, int resourceId){
        View tmpView = activity.findViewById(resourceId);

        if(!visible)
            tmpView.setVisibility(View.GONE);
        else
            tmpView.setVisibility(View.VISIBLE);

       // tmpView.refreshDrawableState();
    }
    
    
     
    /****************************************************************************************
                                    LISTENERS AND RESPONDERS
    ****************************************************************************************/


    @Override
    public void onActivityResult(int requestCode, int resultCode, Intent data){
        LOG("Result of activity : " + resultCode + " with request Code " + requestCode);

        if(requestCode == VALIDATION_ACTIVITY_REQUEST_CODE){
            onValidationActivityResult(requestCode, resultCode, data);
        }
        else if(requestCode == LOOKUP_ACTIVITY_REQUEST_CODE){
            LOG("Result is for Lookup");

            if(resultCode == SealActivity.RESULT_OK){
                LOG("Result OK");
                
                SealIntent sealIntent = (SealIntent)dataController.getCustomIntent();

                SealObservation obs = (SealObservation)sealIntent.getList().get(0);

                currentObs = obs;
                //activity.setObservation(obs);
                activity.setRookerySection(obs.getRookerySection());
                activity.setVisibility(obs.getVisibilityCode());
                activity.setSealNumber(obs.getSealNum());

                formChanged = true;
            }
        }
        // Regardless of outcome, data was either submitted or canceled, revalidation is necessary of the form
        //dataValidated = false;
    }

    public void onValidationActivityResult(int requestCode, int resultCode, Intent data){
        LOG("Result is for Validation");
        SealValidationIntent sealIntent = (SealValidationIntent)dataController.getCustomIntent();
        
        Bundle bundle = sealIntent.getExtras();
        ArrayList<String> errorComments = bundle.getStringArrayList(SealValidationActivityController.OUTPUT_VALIDATION_ERROR_COMMENTS_KEY);
        
        String currentComments = currentObs.getComments();
        currentComments += "\n";
        for(String err : errorComments){
            // Add only new error comments.
            if(!currentComments.contains(err)){
                currentComments += err + "\n";
            }
        }
        currentObs.setComments(currentComments);

        switch(resultCode){
            case SealValidationActivityController.VALIDATION_RESULT_OK:
                 LOG("Result: OK");
                currentObs.setSeal(sealIntent.getTargetSeal());

                insertObservationIntoDatabase();
                
                break;
            case SealValidationActivityController.VALIDATION_RESULT_NEED_FIRST_TAG_COLOR:
                LOG("Result: Need first tag color");

                currentObs.setSeal(sealIntent.getTargetSeal());
                activity.setSeal(sealIntent.getTargetSeal());

                super.displayErrorMessage("Please select a valid tag color", R.id.observation_tag_color_spinner_top, true);

                break;
            case SealValidationActivityController.VALIDATION_RESULT_NEED_FIRST_TAG_TYPE:
                LOG("Result: Need first tag type");

                // Set tag type to null so that tag type is required to be selected to match the color.
                sealIntent.getTargetSeal().getFirstTag().setType(null);

                currentObs.setSeal(sealIntent.getTargetSeal());
                activity.setSeal(sealIntent.getTargetSeal());

                super.displayErrorMessage("Please select a valid tag type for your color", R.id.observation_tag_type_spinner_top, true);

                break;
            case SealValidationActivityController.VALIDATION_RESULT_NEED_SECOND_TAG_COLOR:
                LOG("Result: Need second tag color");

                currentObs.setSeal(sealIntent.getTargetSeal());
                activity.setSeal(sealIntent.getTargetSeal());

                super.displayErrorMessage("Please select a valid tag color", R.id.observation_tag_color_spinner_bottom, true);
                
                break;
            case SealValidationActivityController.VALIDATION_RESULT_NEED_SECOND_TAG_TYPE:
                LOG("Result: Need second tag type");

                // Set tag type to null so that tag type is required to be selected to match the color.
                sealIntent.getTargetSeal().getSecondTag().setType(null);
                
                currentObs.setSeal(sealIntent.getTargetSeal());
                activity.setSeal(sealIntent.getTargetSeal());

                super.displayErrorMessage("Please select a valid tag type", R.id.observation_tag_color_spinner_bottom, true);

                break;
            default: // Cancel
                LOG("Result: Canceled");
                formChanged = true;
                break;
        }
    }

    
    /*
     * Handles Text input from the seal number
     * Currently, takes that input and puts it into a HashMap.
     */
    class SealNumberTextWatcher implements TextWatcher{
        public void beforeTextChanged(CharSequence arg0, int arg1, int arg2, int arg3) {
        }

        public void onTextChanged(CharSequence arg0, int arg1, int arg2, int arg3) {
        }

        public void afterTextChanged(Editable arg0) {
            if(!arg0.toString().equals("")){
                Integer sealNum = Integer.parseInt(arg0.toString());
                
                if(observationsDictionary.containsKey(sealNum)){
                    if(currentObs == observationsDictionary.get(sealNum))
                        return;
                    
                    currentObs = observationsDictionary.get(sealNum);

//                    activity.resetBottom();
//                    activity.resetFirstTag();
//                    activity.resetSecondTag();
                    
                    activity.setObservation(currentObs);
                    
                    formChanged = true;

                    TOAST_USER("Modifying Observation Seal # "+sealNum, activity);
                }
            }
        }
    }


    /**
     * Handle other flipper check box callbacks
     */
    class OnOtherFlipperClick implements OnClickListener{

        public void onClick(View v) {
            CheckBox box = (CheckBox)v;

            if(activity.getOtherTagSeen()  || activity.getOtherFlipperSeen()){
                activity.setOtherTagVisibility(View.VISIBLE);
            }
            else{
                activity.setOtherTagVisibility(View.GONE);
            }
        }
    }

    class OnSexChanged implements OnCheckedChangeListener{

        public void onCheckedChanged(RadioGroup group, int id) {
            
            if(id == R.id.observation_seal_sex_female_radiobutton) // the first, or Unknown sex selection
                setVisible(true, R.id.observation_seal_pup_code_row);
            else
                setVisible(false, R.id.observation_seal_pup_code_row);
        }
    }

}
