<?php if ( ! defined('BASEPATH')) exit('No direct script access allowed');

require_once(APPPATH . 'classes/LDAPDirectory.php');

class SunOne extends LDAPDirectory {
	private $defaultSearchAttributes = array("sn", "uid", "givenname", "middlename", "globalid", "mailalternateaddress", "mailforwardingaddress", "mailmessagestore", "mailhost", "mailQuota", "mailuserstatus", "uniqueidentifier", "cn", "mail", "fullname", "activated", "maildeliveryoption");
	private $allFilter = "(|(uid=SearchString)(uniqueidentifier=SearchString)(uniqueidentifier=xSearchString)(givenname=SearchString)(sn=SearchString))";
	private $nameFilter = "(|(givenname=SearchString)(sn=SearchString))";
	private $UAIDFilter = "(|(uniqueidentifier=SearchString)(uniqueidentifier=xSearchString))";
	private $usernameFilter = "(uid=SearchString)";
	
	const ACCOUNT_ENABLED = 31;
	const ACCOUNT_DISABLED = 0;
	
	
    function __construct($dbType = null) {
		if(is_null($dbType))
			parent::__construct('sunone');
		else parent::__construct($dbType);
    }
	
	public function get($uid, $searchAttributes = null) {
		if(is_null($searchAttributes)) $searchAttributes = $this->defaultSearchAttributes;
		
		if(is_a($uid, "ActiveObject")) {
			if(!is_a($uid, ActiveObject::USER)) throw new ActiveObjectRequiredException('Invalid Active Object for query.');
			$uid = $uid->id;
		}
		
		if(is_numeric($uid)) $filter = str_replace('SearchString', $uid, $this->UAIDFilter);
		else $filter = "(uid=$uid)";

		return parent::get($searchAttributes, $filter);
	}
	
	public function search($filter, $search) {
		if(is_numeric($search))			$filter = $this->UAIDFilter;
		elseif($filter == 'all')		$filter = $this->allFilter;
		elseif($filter == 'name')		$filter = $this->nameFilter;
		elseif($filter == 'uaid')		$filter = $this->UAIDFilter;
		elseif($filter == 'username')	$filter = $this->usernameFilter;
		
		return parent::search(array("uid", "globalid", "uniqueidentifier", "cn"), str_replace('SearchString', $search, $filter));
	}
	
	public function auth($user, $password) {
		$uid = $this->get($user, array('uid'));
		if(count($uid['uid']) > 0) $uid = $uid['uid'][0];
	
		$usernames = array();
		array_push($usernames, "uid=". $uid .",ou=Student,o=uaa.alaska.edu,o=isp");
		array_push($usernames, "uid=". $uid .",ou=Admin,o=uaa.alaska.edu,o=isp");
		
		foreach($usernames as $username) {
			if(parent::auth($username, $password) === true) return true;
		}
		
		return false;
	}
	
	/**
	 * SunOne LDAP does not support the ldap_rename function. This functionality has
	 * been replaced by remove/create logic instead.
	 **/
	public function rename(ActiveObject $ao, $newRDN, $newParent, $deleteOld = true) {
		//Open current entry
		$user = $this->get($ao, array('dn'));
		$dn = $user['dn'];
		
		$result = @ldap_search($this->conn, $dn, "uid=". $ao->uid);
		if(!$result || empty($result)) throw new Exception("Unable to retrieve records for user account '$dn'.");
		$data = ldap_get_entries($this->conn, $result);
		$attrs = array();
		
		//Retrieve all attributes and their values
		for($i = 0; $i < $data["count"]; $i++) {
			if($data[$i]) {
				for($j = 0; $j < $data[$i]["count"]; $j++) {
					if($data[$i][$j]) {
						$source = $data[$i][$j];
						if($data[$i][$j] != "objectclass") {
							$attrs[$source] = $data[$i][$data[$i][$j]][0];
						}
					}
				}
			}
		}
		
		//Remove current entry
		if($deleteOld) ldap_delete($this->conn, $dn);
		
		//Create new entry in new ou with stored attributes
		$attrs['objectclass'][0] = "top";
		$attrs['objectclass'][1] = "person";
		$attrs['objectclass'][2] = "organizationalPerson";
		$attrs['objectclass'][3] = "inetorgperson";
		$attrs['objectclass'][4] = "uaainetorgperson";
		$attrs['objectclass'][5] = "inetUser";
		$attrs['objectclass'][6] = "ipUser";
		$attrs['objectclass'][7] = "nsManagedPerson";
		$attrs['objectclass'][8] = "userPresenceProfile";
		$attrs['objectclass'][9] = "inetMailUser";
		$attrs['objectclass'][10] = "inetLocalMailRecipient";
		$mypath = $newRDN . "," . $newParent;
		return ldap_add($this->conn, $mypath, $attrs);
	}
	
	/**
	 * Adds the account's "activated" attribute, enabling it in the directory.
	 **/
	public function enable(ActiveObject $ao) {
		$user = $this->get($ao, array('dn'));
		if($user) $this->modify($user['dn'], array('activated' => self::ACCOUNT_ENABLED));
		else throw new Exception("User account not found in SunOne directory.");
	}
	
	/**
	 * Removes the account's "activated" attribute, disabling it in the directory.
	 **/
	public function disable(ActiveObject $ao) {
		$user = $this->get($ao, array('dn'));
		if($user) $this->remove($user['dn'], array('activated' => array()));
		else throw new Exception("User account not found in SunOne directory.");
	}
	
	/**
	 * Checks to see if the account is enabled in the directory by looking at it's "activated" attribute.
	 **/
	public function isEnabled(ActiveObject $ao) {
		$user = $this->get($ao, array('activated'));
		if(!$user || !isset($user['activated']) || count($user['activated']) == 0) return false;
		else return true;
	}
	
	public function membership($uid) {
		if(is_numeric($uid)) $filter = str_replace('SearchString', $uid, $this->UAIDFilter);
		else $filter = str_replace('SearchString', $uid, $this->usernameFilter);
		
		return parent::membership($filter);
	}
}


?>