<?php

class ModuleConfig {
	const DEFAULT_CATEGORY = 'Uncategorized';

	public $className;
	
	public $name;
	public $version;
	public $description;
	public $enabled;
	public $leftnav;
	public $pages = array();
	public $userButtons;
	public $userAttributes;
	public $classButtons;
	public $classAttributes;
	public $instance;
	
	function __construct($moduleName, $moduleDir) {
		$this->className = $moduleName;
		
		//Check for plugin architecture
		if(!in_array('config.xml', $moduleDir) || !in_array($moduleName . ".php", $moduleDir))
			throw new Exception("Invalid plugin architecture.");

		$config = simplexml_load_file(MODULE_PATH . $moduleName . "/config.xml");
		if(count(libxml_get_errors()) > 0) {
			libxml_clear_errors();
			throw new Exception("Configuration file is not properly formatted.");
		}
		
		$attr = $config->attributes();
		
		// Validate the XML
		if(!isset($attr['name'])) 		throw new Exception("Module must have 'name' attribute set.");
		if(!isset($attr['version']))	throw new Exception("Module must have 'version' attribute set.");
		if(!isset($attr['enabled'])) 	throw new Exception("Module must have 'enabled' attribute set.");
		if(count($config->description) != 1) throw new Exception("Module must have a description tag.");
		
		//Parse the config into the module array
		$this->name = trim($attr['name']);
		$this->version = trim($attr['version']);
		$this->enabled = $attr['enabled'] == 'true';
		$this->description = (string)$config->description;
		
		if(count($config->category) != 1) $this->category = self::DEFAULT_CATEGORY;
		else $this->category = (string)$config->category;
		
		if(count($config->leftnav) != 1) $this->leftnav = false;
		else $this->leftnav = $config->leftnav == 'true';
		
		$this->pages = $this->parseUIElement($config->page);
		$this->userButtons = $this->parseUIElement($config->userButton);
		$this->userAttributes = $this->parseUIElement($config->userAttribute);
		$this->classButtons = $this->parseUIElement($config->classButton);
		$this->classAttributes = $this->parseUIElement($config->classAttribute);
	}
	
	private function parseUIElement($config) {
		$ret = array();
	
		foreach($config as $element) {
			$ary = (array)$element->attributes();
			if(isset($ary['@attributes'])) $ary = $ary['@attributes'];
			if(isset($element->check)) {
				$ary['check'] = array();
				
				foreach($element->check as $check)
					array_push($ary['check'], (string)$check);
			}
			
			array_push($ret, $ary);
		}
		
		return $ret;
	}
	
	public function instantiate() {
		if(!isset($this->instance)) {
			include(MODULE_PATH . "/$this->className/$this->className.php");
			
			if(!class_exists($this->className)) throw new Exception("$this->className must be defined.");
			else {
				$this->instance = new $this->className;
				if(!is_a($this->instance, 'ModuleClass')) throw new Exception("$this->className class must inherit from ModuleClass base class.");
			}
		}
	}
	
	public function getValidItems($itemArray) {
		$validItems = array();
		if(!$this->enabled) return $validItems;
		
		foreach($itemArray as $item) {
			
			try {
				//Verify that the item's target method exists
				$this->instantiate();
				if(!is_callable(array($this->instance, "" . $item['function'] . ""))) throw new Exception("Target method '" .  $item['function'] . "' does not exist");
			
				//Run all check methods on the item
				if(isset($item['check'])) {
					$checks = $item['check'];
					if(!is_array($checks)) $checks = array($checks);
					
					$checkPassed = $this->callCheckFunctions($checks);
					if(!$checkPassed) continue;
				}
				
				$validItem = array();
				$validItem['name'] = $item['name'];
				$validItem['url'] = "/module/load/" . $this->className . "/" . $item['function'];
				if(isset($item['description'])) $validItem['description'] = $item['description'];
			
				array_push($validItems, $validItem);
			}
			catch(Exception $e) {
				log_error($this->className, 'Attribute/Button Validation', $e->getMessage());
			}
		}
		
		return $validItems;
	}
	
	private function callCheckFunctions($checks) {
		foreach($checks as $check) {
			$notResult = false;
			if(strpos($check, "!") === 0) {
				$notResult = true;
				$check = substr($check, 1);
			}

			if(!is_callable(array($this->instance, "$check"))) throw new Exception("Check failed - '$check' method does not exist");
			
			ob_start();
			$result = call_user_func(array($this->instance, "$check"));
			ob_end_clean();

			if($result === false && $notResult == false) return false;
			elseif($result === true && $notResult == true) return false;
			elseif($result !== false && $result !== true) throw new Exception("Check function must return a boolean value - '$result' returned.");
		}
		
		return true;
	}
}

?>