<?php

abstract class LDAPDirectory {
	const MAX_SEARCH_RESULTS = 500;
	protected $conn;
	private $authConn;
	protected $auth;
	protected $ldapOptions;
	
	public function auth($user, $password) {
		if(!isset($this->authConn)) $this->authConn = $this->connect();
		return @ldap_bind($this->authConn, $user, $password);
	}
	
	/**
	 * Caches authentication information and connects to the directory.
	 * Note that since CodeIgniter maintains one instance of each library this
	 * connection is essentially a Singleton.
	 **/
	protected function __construct($authKey, $ldapOptions = array(LDAP_OPT_PROTOCOL_VERSION => 3, LDAP_OPT_REFERRALS => 0)) {
		$this->ldapOptions = $ldapOptions;
		
		//Retrieve authentication information
		$CI =& get_instance();
		$CI->load->config('auth', TRUE);
		$this->auth = &$CI->config->item('auth');
		$this->auth = $this->auth[$authKey];
		
		$this->conn = $this->connect();
	}
	
	private function connect() {
		//Connect, configure & bind to directory
		$conn = ldap_connect($this->auth['SERVER'], $this->auth['PORT']);
		
		foreach($this->ldapOptions as $option => $value)
			ldap_set_option($conn, $option, $value);

		ldap_bind($conn, $this->auth['USER'], $this->auth['PASS']);
		
		return $conn;
	}

	/**
	 * Retrieves a single entry from the LDAP Directory. This method will clean the resulting
	 * entry so that count() functions and foreach loops will interpret it correctly.
	 **/
	public function get(array $searchAttributes, $filter) {
		$ureturn = @ldap_search($this->conn, $this->auth['BASE'], $filter, $searchAttributes, 0, 1);
		$entry = @ldap_first_entry($this->conn, $ureturn);
		if($entry === FALSE) return false;

		//Retrieve result attributes into array	
		$attributes = ldap_get_attributes($this->conn, $entry);
		
		//Clean array so that foreach loops will work on variable without encountering duplication
		for($x = 0; $x <= $attributes['count']; $x++) {
			unset($attributes[$x]);
		}
		
		//Remove 'count' entries as they are now not needed
		unset($attributes['count']);
		foreach($attributes as &$attribute) {
			unset($attribute['count']);
		}
		
		//Add DN string to attributes
		$attributes['dn'] = ldap_get_dn($this->conn, $entry);
		
		return $attributes;
	}
	
	
	/**
	 * Adds new records with provided attributes to the directory. This is fundamentally
	 * different from 'add' as it can create accounts.
	 **/
	public function addRecord($dn, array $entries) {
		return ldap_add($this->conn, $dn, $entries);
	}
	 
	/**
	 * Adds new attributes to a single record in the directory.
	 **/
	public function add($dn, array $entries) {
		return ldap_mod_add($this->conn, $dn, $entries);
	}
	
	/**
	 * Removes attributes from a record in the directory
	 **/
	public function del($dn, array $entries) {
		return ldap_mod_del($this->conn, $dn, $entries);
	}
	
	/**
	 * Alias method for $this->del
	 **/
	public function remove($dn, array $entries) {
		$this->del($dn, $entries);
	}
	
	/**
	 * Removes an entire record from the directory.
	 **/
	public function removeRecord($dn) {
		return ldap_delete($this->conn, $dn);
	}
	
	/**
	 * Modifies attributes for a single record in the directory
	 **/
	public function modify($dn, array $entries) {
		return ldap_mod_replace($this->conn, $dn, $entries);
	}
	
	/**
	 * Renames an account in the directory, copying it to a new location under
	 * the specified parent and RDN. It will perform a move operation if $deleteOld is
	 * true.
	 **/
	public function rename(ActiveObject $ao, $newRDN, $newParent, $deleteOld = true) {
		$user = $this->get($ao, array('dn'));
		
		return ldap_rename($this->conn, $user['dn'], $newRDN, $newParent, $deleteOld);
	}
	
	
	/**
	 * Enables the account in the directory. This is implemented differently for the various
	 * directory services and thus implementation is left to the child class.
	 **/
	public abstract function enable(ActiveObject $ao);
	
	/**
	 * Disables the account in the directory. This is implemented differently for the various
	 * directory services and thus implementation is left to the child class.
	 **/
	public abstract function disable(ActiveObject $ao);
	
	
	/**
	 * Checks to see if an account is enabled in the directory. This is implemented differently
	 * for the various directory services and thus implementation is left to the child class.
	 **/
	public abstract function isEnabled(ActiveObject $ao);
	
	
	
	/**
	 * Returns an array of all groups that this account record is a member of.
	 **/
	protected function membership($filter) {
		$ureturn = @ldap_search($this->conn, $this->auth['BASE'], $filter, array('memberof'), 0, 1);
		$entry = @ldap_first_entry($this->conn, $ureturn);
		if($entry === FALSE) return false;
		
		//Retrieve result attributes into array	
		$attributes = ldap_get_attributes($this->conn, $entry);
		if(!isset($attributes['memberOf'])) return array();
		
		unset($attributes['memberOf']['count']);
		return $attributes['memberOf'];
	}
	
	/**
	 * Retrieves multiple entries for the directory. Note that this method does no
	 * cleaning and returns the attributes as they are.
	 **/
	protected function search($searchAttributes, $filter) {
		$ureturn = @ldap_search($this->conn, $this->auth['BASE'], $filter, $searchAttributes, 0, self::MAX_SEARCH_RESULTS);
		return ldap_get_entries($this->conn, $ureturn);
	}
	
	/**
	 * Converts Active Directory formatted dates into unix timestamps.
	 * Code taken from: http://morecavalier.com/index.php?whom=Apps%2FLDAP+timestamp+converter
	 **/
	public static function parseDate($date) {
		$secsAfterADEpoch = $date / (10000000); // seconds since jan 1st 1601
		$ADToUnixConvertor = ((1970-1601) * 365 - 3 + round((1970-1601) / 4)) * 86400; // unix epoch - AD epoch * number of tropical days * seconds in a day
		// Why -3 ?
		// "If the year is the last year of a century, eg. 1700, 1800, 1900, 2000,
		// then it is only a leap year if it is exactly divisible by 400.
		// Therefore, 1900 wasn't a leap year but 2000 was."
	
		$date = intval($secsAfterADEpoch-$ADToUnixConvertor); // unix Timestamp version of AD timestamp
		return $date; 
	}
	
	/**
	 * Converts unix timestamps into Active Directory formatted dates.
	 * Modified directly from the 'parseDate' method.
	 **/
	public static function convertToAD($date) {
		$ADToUnixConvertor = ((1970-1601) * 365 - 3 + round((1970-1601) / 4)) * 86400; // unix epoch - AD epoch * number of tropical days * seconds in a day
		$date += $ADToUnixConvertor; // unix Timestamp version of AD timestamp
		$date *= 10000000; // seconds since jan 1st 1601
		
		return $date;
	}
	
	/**
	 * Closes the connection to the directory. Note that this is done automatically when the HTTP
	 * request finishes.
	 **/
	public function close() {
		ldap_close($this->conn);
	}
}

?>