﻿using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Audio;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.GamerServices;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.Input;
using Microsoft.Xna.Framework.Media;
using Microsoft.Xna.Framework.Net;
using Microsoft.Xna.Framework.Storage;
using XSIXNARuntime;

namespace Hook_FPS
{
    /// <summary>
    /// The ModelAsset Class stores information about the Crosswalk Model 
    /// content including animations and its World Parameters.
    /// </summary>
    public class ModelAsset
    {
        #region Private Fields

        // Model, Animation, and Animation Data
        private Model _CrosswalkModel;
        private List<XSIAnimationContent> _Animations;
        private XSIAnimationData _AnimationData;
        private String _FilePath;
        private int _AnimationIndex;
        private int _OldAnimationIndex;
        private int _NumberOfAnimations;
        private bool _PlaybackStatus;
        private TimeSpan _LastPlayCommand = TimeSpan.Parse("0");

        #endregion
        
        #region Public Fields

        // World Matrix Parameters and Bounding information for collision detection
        // These regions are public for performance reasons.
        public Vector3 Position;
        public Vector3 Forward; // If model is a camera, this is the camera intrest
        public Vector3 Up;
        public Vector3 Velocity;
        public float Scale;
        public float Yaw;
        public float Pitch;
        public float Roll;
        public Matrix World;

        #endregion

        #region Properties

        // Get and Set Asset Properties
        public Model CrosswalkModel { get { return _CrosswalkModel; } }
        public List<XSIAnimationContent> Animations { get { return _Animations; } set { _Animations = value; } }
        public XSIAnimationData AnimationData { get { return _AnimationData; } set { _AnimationData = value; } }
        public int AnimationIndex { get { return _AnimationIndex; } set { _AnimationIndex = value; } }
        public int OldAnimationIndex { get { return _OldAnimationIndex; } set { _OldAnimationIndex = value; } }
        public int NumberOfAnimations { get { return _NumberOfAnimations; } }
        public String FilePath { get { return _FilePath; } }
        public bool PlaybackStatus { get { return _PlaybackStatus; } set { _PlaybackStatus = value; } }
        public TimeSpan LastPlayCommand { get { return _LastPlayCommand; } set { _LastPlayCommand = value; } }

        #endregion
        
        #region Constructor(s)

        /// <summary>
        /// This is called to construct and initialize a new ModelAsset
        /// </summary>
        /// <param name="AssetPath">The path of the Model to load</param>
        /// <param name="content">The game content manager</param>
        public ModelAsset(String AssetPath, ContentManager content)
        {
            // Load the Model and Initialize Private Fields
            _CrosswalkModel = content.Load<Model>(AssetPath);
            _FilePath = AssetPath;
            _Animations = new List<XSIAnimationContent>();
            _AnimationIndex = 0;
            _OldAnimationIndex = 0;
            _NumberOfAnimations = 0;
            _PlaybackStatus = false;

            // Init public fields
            Position = Vector3.Zero;
            Forward = Vector3.Forward;
            Up = Vector3.Up;  //This Vector will be used for gravity orientation. The opposite of this vector is gravity.
            Velocity = Vector3.Zero;
            Scale = 1.0f;
            Yaw = 0.0f;
            Pitch = 0.0f;
            Roll = 0.0f;
            World = Matrix.Identity;

            // Init Bounding Sphere array

            // Load the Animations and set the World Matrix
            LoadAnimations();
            CalculateWorld();
        }

        #endregion

        #region Private Methods

        /// <summary>
        /// This is called to load the model animations
        /// </summary>
        private void LoadAnimations()
        {
            _AnimationData = _CrosswalkModel.Tag as XSIAnimationData;

            if (_AnimationData != null)
            {
                foreach (KeyValuePair<String, XSIAnimationContent> AnimationClip in _AnimationData.RuntimeAnimationContentDictionary)
                {
                    AnimationClip.Value.BindModelBones(_CrosswalkModel);
                    _Animations.Add(AnimationClip.Value);
                    _NumberOfAnimations++;
                }
                _NumberOfAnimations--;
                _AnimationData.ResolveBones(_CrosswalkModel);
            }
            
        }

        #endregion

        #region Public Methods

        /// <summary>
        /// This is used to set the Position vector for Translating
        /// </summary>
        /// <param name="vector">The vector to set the Model position</param>
        public void Translate(Vector3 vector)
        {
            Position += vector;
            
        }

        /// <summary>
        /// This is used to set the amount of rotation in radians
        /// </summary>
        /// <param name="radians">The amount of rotation in radians</param>
        public void RotateAboutY(float radians)
        {
            Yaw += radians;
        }

        /// <summary>
        /// This is used to set the amount of rotation in radians
        /// </summary>
        /// <param name="radians">The amount of rotation in radians</param>
        public void RotateAboutX(float radians)
        {
            Pitch += radians;
        }

        /// <summary>
        /// This is used to set the amount of rotation in radians
        /// </summary>
        /// <param name="radians">The amount of rotation in radians</param>
        public void RotateAboutZ(float radians)
        {
            Roll += radians;
        }

        /// <summary>
        /// This is called from Game.Update to update the state of the Model and Animtion sequence
        /// </summary>
        /// <param name="gameTime">The gameTime from Game1.cs</param>
        public void Update(GameTime gameTime)
        {
        
            // Determine whether or not to play an animation
            if (_Animations != null && _Animations.Count > 0)
            {
                if (_PlaybackStatus)
                {
                    _Animations[_AnimationIndex].PlayBack(gameTime.ElapsedGameTime, 0.5f);
                }
                else
                {
                    _Animations[_AnimationIndex].PlayBack(TimeSpan.Parse("00:00:00"), 0.5f);
                    
                    // The Cog for binding mouse look
                    //Models[CurrentModel].CrosswalkModel.Bones["Bip01_Spine"].Transform = Matrix.CreateRotationX(MathHelper.ToRadians(90.0f));
                }
            }
            // Recalculate the world matrix
            CalculateWorld();
        }

        /// <summary>
        /// This tells the animation to start playing
        /// </summary>
        /// <param name="gameTime">The gameTime from Game1.cs</param>
        public void Play(GameTime gameTime)
        {
            if (!_PlaybackStatus)
            {
                _PlaybackStatus = true;
                _Animations[_AnimationIndex].Reset();
                _LastPlayCommand = gameTime.TotalGameTime;
            }
            else
            {
                _Animations[_AnimationIndex].Reset();
                //_LastPlayCommand = gameTime.TotalGameTime;
            }
            
        }
        
        /// <summary>
        /// Calculates the forward vector based on the current gravity axis
        /// </summary>
        private void CalculateForward()
        {
           
            if (this.Up == Vector3.Up)
            {
                this.Forward = Matrix.Multiply(this.World, Matrix.CreateRotationY(this.Yaw)).Forward;
                this.Forward.Y = 0.0f;
            }
            else if (this.Up == Vector3.Down)
            {
                this.Forward = Matrix.Multiply(this.World, Matrix.CreateRotationY(-this.Yaw)).Forward;
                this.Forward.Y = 0.0f;
            }
            else if (this.Up == Vector3.Left)
            {
                this.Forward = Matrix.Multiply(this.World, Matrix.CreateRotationX(-this.Yaw)).Forward;
                this.Forward.X = 0.0f;
            }
            else if (this.Up == Vector3.Right)
            {
                this.Forward = Matrix.Multiply(this.World, Matrix.CreateRotationX(this.Yaw)).Forward;
                this.Forward.X = 0.0f;
            }
            else if (this.Up == Vector3.Forward)
            {
                this.Forward = Matrix.Multiply(this.World, Matrix.CreateRotationZ(-this.Yaw)).Forward;
                this.Forward.Z = 0.0f;
            }
            else if (this.Up == Vector3.Backward)
            {
                this.Forward = Matrix.Multiply(this.World, Matrix.CreateRotationZ(this.Yaw)).Forward;
                this.Forward.Z = 0.0f;
            }
            
            //reset the yaw
            this.Yaw = 0.0f;
            
        }



        /// <summary>
        /// This will calculate the world matrix based on the rotation and translation parameters
        /// </summary>
        public void CalculateWorld()
        {
            CalculateForward();
            this.World = Matrix.CreateWorld(this.Position, this.Forward, this.Up);
            this._CrosswalkModel.Root.Transform = this.World;
        }

        #endregion


    }
}