﻿using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Audio;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.GamerServices;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.Input;
using Microsoft.Xna.Framework.Media;
using Microsoft.Xna.Framework.Net;
using Microsoft.Xna.Framework.Storage;
using XSIXNARuntime;

namespace Hook_FPS
{
    class FirstPersonCamera
    {
        #region Private Fields

        private GraphicsDeviceManager _graphics;
        private Vector3 _position;
        private Matrix _projectionMatrix;
        private Matrix _viewMatrix;
        private float _leftrightRotation;
        private float _updownRotation;
        private const float _rotationSpeed = 0.1f;
        private const float _moveSpeed = 100.0f;
        private MouseState _originalMouseState;

        #endregion

        #region Properties

        public Vector3 position { get { return _position; } set { _position = value; } }
        public Matrix projectionMatrix { get { return _projectionMatrix; } set { _projectionMatrix = value; } }
        public Matrix viewMatrix { get { return _viewMatrix; } set { _viewMatrix = value; } }
        public float leftrightRotation { get; set; }
        public float updownRotation { get; set; }
        public float rotationSpeed { get; set; }
        public float moveSpeed { get; set; }
        public MouseState originalMouseState { get; set; }

        #endregion

        #region Constructor(s)

        /// <summary>
        /// This constructs a new FirstPersonCamera
        /// </summary>
        /// <param name="gameGraphicsManager">Passed from the Game1.cs GraphicsDeviceManager</param>
        public FirstPersonCamera(GraphicsDeviceManager gameGraphicsManager, MouseState newMouseState)
        {
            _graphics = gameGraphicsManager;
            _position = new Vector3(50.0f, 10.0f, 0.0f);
            _leftrightRotation = MathHelper.PiOver2;
            _updownRotation = -MathHelper.Pi / 10.0f;
            
            _projectionMatrix = Matrix.CreatePerspectiveFieldOfView(
                MathHelper.PiOver4,
                _graphics.GraphicsDevice.Viewport.AspectRatio,
                1.0f,
                10000.0f);

            _viewMatrix = Matrix.CreateLookAt(_position, Vector3.Zero, Vector3.Up);
            
            _originalMouseState = newMouseState;
        }

        #endregion

        #region Private Methods

        /// <summary>
        /// This adds a Vector to the current camera position and calls UpdateViewMatrix
        /// </summary>
        /// <param name="vectorToAdd">A vector3 to add to the current position vector</param>
        private void AddToCameraPosition(Vector3 vectorToAdd)
        {
            Matrix cameraRotation = Matrix.CreateRotationX(_updownRotation) *
                Matrix.CreateRotationY(_leftrightRotation);

            Vector3 rotatedVector = Vector3.Transform(vectorToAdd, cameraRotation);

            _position += _moveSpeed * rotatedVector;

            UpdateViewMatrix();
        }

        /// <summary>
        /// This updates the camera view matrix to handle all transformations
        /// </summary>
        private void UpdateViewMatrix()
        {
            Matrix cameraRotation = Matrix.CreateRotationX(_updownRotation) *
                Matrix.CreateRotationY(_leftrightRotation);

            Vector3 cameraOriginalTarget = new Vector3(0, 0, -1);
            Vector3 cameraOriginalUpVector = new Vector3(0, 1, 0);
            Vector3 cameraRotatedTarget = Vector3.Transform(cameraOriginalTarget, cameraRotation);
            Vector3 cameraFinalTarget = _position + cameraRotatedTarget;
            Vector3 cameraRotatedUpVector = Vector3.Transform(cameraOriginalUpVector, cameraRotation);

            
            _viewMatrix = Matrix.CreateLookAt(
                _position,
                cameraFinalTarget,
                cameraRotatedUpVector);
        }

        #endregion

        #region Public Methods

        /// <summary>
        /// This will process the mouse and keyboard input and apply changes to leftright rotation and updown rotation then update the camera view matrix.
        /// Also resets the mouse position to the center of the viewport.
        /// </summary>
        /// <param name="timeAmount">The amount of time elapsed</param>
        public void ProcessInput(float timeAmount, MouseState currentMouseState, Vector3 moveVector)
        {
            if (currentMouseState != _originalMouseState)
            {
                float xDifference = currentMouseState.X - _originalMouseState.X;
                float yDifference = currentMouseState.Y - _originalMouseState.Y;
                _leftrightRotation -= _rotationSpeed * xDifference * timeAmount;
                _updownRotation -= _rotationSpeed * yDifference * timeAmount;

                Mouse.SetPosition(_graphics.GraphicsDevice.Viewport.Width / 2, _graphics.GraphicsDevice.Viewport.Height / 2);

                UpdateViewMatrix();
            }

            AddToCameraPosition(moveVector * timeAmount);
        }

        #endregion
    }
}
