package main;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.GraphicsConfiguration;
import java.awt.Toolkit;
import javax.media.j3d.AmbientLight;
import javax.media.j3d.BoundingSphere;
import javax.media.j3d.BranchGroup;
import javax.media.j3d.Canvas3D;
import javax.media.j3d.Group;
import javax.media.j3d.Transform3D;
import javax.media.j3d.TransformGroup;
import javax.media.j3d.View;
import javax.swing.JPanel;
import javax.vecmath.Color3f;
import javax.vecmath.Point3d;
import javax.vecmath.Vector3d;
import utils.Logger;
import com.sun.j3d.utils.picking.PickTool;
import com.sun.j3d.utils.universe.PlatformGeometry;
import com.sun.j3d.utils.universe.SimpleUniverse;
import com.sun.j3d.utils.universe.ViewingPlatform;

public class Controller extends JPanel
{
	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;
	private static final int PWIDTH = Toolkit.getDefaultToolkit( ).getScreenSize( ).width;   // size of panel
	private static final int PHEIGHT = Toolkit.getDefaultToolkit( ).getScreenSize( ).height-100; 
	
	private static final int BOUNDSIZE = 100;  // larger than world
	
	private SimpleUniverse simpleUniverse;
	private BranchGroup sceneBranchGroup;
	public static final BoundingSphere UNIVERSE_BOUNDS = new BoundingSphere( new Point3d( 0, 0, 0 ), BOUNDSIZE );   
	// for environment nodes
	
	private TerrainManager terrain;	  // terrain manager
	private Logger log; // message logger
	
	public Controller(String fileName, Logger log) 
	{
		this.log = log;
		
		setLayout( new BorderLayout() );
		setOpaque( false );
		setPreferredSize( new Dimension(PWIDTH, PHEIGHT));
		
		GraphicsConfiguration config =
				SimpleUniverse.getPreferredConfiguration();
		Canvas3D canvas3D = new Canvas3D(config);
		add("Center", canvas3D);
		canvas3D.setFocusable(true); 
		canvas3D.requestFocus();
		simpleUniverse = new SimpleUniverse(canvas3D);
		
		sceneBranchGroup = new BranchGroup();
		
		lightScene();	 // add the lights
		
		PickTool pickTool = new PickTool(sceneBranchGroup);

	    terrain = new TerrainManager(fileName, pickTool, log);
		
		sceneBranchGroup.addChild( terrain.getTerrain() );  // add terrain, using TerrainManager
	    
		prepareViewPoint();
		
		simpleUniverse.addBranchGraph( sceneBranchGroup );
	}
	
	private void lightScene()
	{
		Color3f grey = new Color3f(0.5f, 0.5f, 0.5f);
		
		// Set up the ambient light 
		AmbientLight ambientLightNode = new AmbientLight(grey);
		ambientLightNode.setInfluencingBounds(UNIVERSE_BOUNDS);
		sceneBranchGroup.addChild(ambientLightNode);
	}
	
	private void initViewPosition(TransformGroup steerTransfromGroup)
	// rotate and move the viewpoint
	{
		Transform3D transformation3d = new Transform3D();
		steerTransfromGroup.getTransform(transformation3d);
		Transform3D toRotate = new Transform3D();
		toRotate.rotY(-Math.PI);   
		// rotate 180 degrees around Y-axis, so facing along positive z-axis
		
		transformation3d.mul(toRotate);
		transformation3d.setTranslation( terrain.getStartPosition() );  // place at terrain start
		steerTransfromGroup.setTransform(transformation3d); 
	}
	
	private void prepareViewPoint()
	{
		// adjust viewpoint parameters
		View userView = simpleUniverse.getViewer().getView();
		userView.setFieldOfView( Math.toRadians(90.0));  // wider FOV
		// 10 and 0.1; keep ratio between 100-1000
		userView.setBackClipDistance(20);	  // can see a long way
		userView.setFrontClipDistance(0.05);   // can see close things
		
		ViewingPlatform viewingPlatform = simpleUniverse.getViewingPlatform();
		
		// fix starting position and orientation of viewpoint
		TransformGroup steerTG = viewingPlatform.getViewPlatformTransform();
		initViewPosition(steerTG);
		
		// create ammo (beams and explosions)
		AmmoManager ammoManager = new AmmoManager(steerTG, sceneBranchGroup, terrain, "fire", log, terrain.getObjectManager() );
		
		// add a spotlight and avatar to viewpoint
		PlatformGeometry platformGeometry = new PlatformGeometry();
		Transform3D transform3d = new Transform3D(); // Need to translate the image back
		transform3d.setTranslation(new Vector3d(0.0, -0.35, -0.1)); // so the body sits correctly in the player's view.
		TransformGroup playerModel = new TransformGroup(transform3d);
		playerModel.addChild( terrain.getPlayerBody( platformGeometry, ammoManager ) );
		platformGeometry.addChild(playerModel);
		platformGeometry.setCapabilityIsFrequent(Group.ALLOW_COLLISION_BOUNDS_READ);
		platformGeometry.setCapability(Group.ALLOW_COLLIDABLE_READ);
		viewingPlatform.setPlatformGeometry(platformGeometry);
		
		// set up keyboard controls
		KeyAndMouseBehavior keyBehavior = new KeyAndMouseBehavior(terrain, ammoManager, log);
		keyBehavior.setSchedulingBounds(UNIVERSE_BOUNDS);
		viewingPlatform.setViewPlatformBehavior(keyBehavior);
	}
}
