package main;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;

import javax.media.j3d.BranchGroup;
import javax.media.j3d.Node;
import javax.vecmath.Vector3d;

import com.sun.j3d.utils.picking.PickTool;
import com.sun.j3d.utils.universe.PlatformGeometry;

import objects.DynamicObjectManager;
import objects.AI.EvolutionGenePool;
import objects.Factories.Factory;
import objects.Spells.FireSpell;
import objects.Spells.Spell;
import utils.Logger;

public class TerrainManager 
{
	// The established sizes are 1.0 = 10 feet tall, or 1 story tall.  A man's eyes are at about 5'3", or 0.53 units tall
	private static final int LENGTH = 40;    // max sides of terrain plan (should be even), the terrain can be smaller than the max sizes
	private static final int MAX_HEIGHT = 3; // the even heights are only 1.5 ft thick (celing/floor), or 0.3 units
	public static final double COORD_OFFSET = 0.5; // The graphics start at point 0,0, each grid square covers from point 0,0 to 1,1 so the offset is needed
											 // for simple collision detection, since object position 1,1 is now 1.5,1.5, in the middle of the grid
											 // square 1,1 instead of just point 1,1
	public static final double PLAYABLE_AREA = 1.0; // indicates that in the grid, the y-position of 1.0 is the place where people and things other than
	                                         // floors and roof tiles can exist.
	
	public static final char PLAYER_SPAWNER = 'S';
	public static final char DYNAMIC_BLOCK = 't';
	public static final char STATIC_BLOCK = 'b';
	public static final char CYLINDER_BLOCK = 'o';
	public static final char KEY_DYNAMIC_BLOCK = 'T';
	public static final char EMPTY_GROUND = ' ';
	public static final char STANDARD_FLOOR_TYLE = '.';
	public static final char STANDARD_ROOF_TYLE = '_';
	public static final char LIGHT = 'l';
	public static final char LIGHT_AND_NODE_INDICATOR = 'L';
	public static final char CASTER_SPAWNER = '\\';
	public static final char FIGHTER_SPAWNER = '/';
	public static final char TRAPPER_SPAWNER = '-';
	public static final char GENERAL_SPAWNER = '*';
	
	private char[][][] terrain;    // stores the input floor plan;
	/*  The coords are terrain[Z][X][Y] 
	    i.e. x-coords along row, z-coords along column
	    Remember that the terrain plan's y-coords have been mapped to
	    3D scene z-coords, representing the different floors.  */
	
	private double xStartPosn, zStartPosn, yStartPosn;    // the player's starting coordinate
	
	private BranchGroup terrainBranchGroup;              // scene subgraph for the terrain
	private Logger log;                         // contains an instance of the logger
	private DynamicObjectManager objectManager; // contains the list of 'killable' and 'consumable' and changing items
	private PickTool pickTool;
	private EvolutionGenePool genePool;
	private String[] split;
	private int aiNumber = 0;
	
	public TerrainManager(String file_name, PickTool pickTool, Logger log) 
	{ 
		this.log = log;
		this.pickTool = pickTool;
		objectManager = new DynamicObjectManager(log);
		genePool = new EvolutionGenePool();
		initialiseVars();
		readFile(file_name);
		buildTerrainRepresentation();
	} // end of TerrainManager()
	
	private void initialiseVars()
	{
		terrain = new char[LENGTH][LENGTH][MAX_HEIGHT];
	    /*for(int z=0; z<LENGTH; z++)    // an empty terrain data structure TODO this might be waste-full, but prevents 
	    {                                // out of bounds errors in the arrays.
	    	for(int x=0; x<LENGTH; x++)
	    	{
	    		for (int y = 0; y < MAX_HEIGHT; y++) {
					terrain[z][x][y] = ' ';
				} 
	    	}
	    }*/

	    xStartPosn = LENGTH/2;    // default position (top row, in the middle)
	    zStartPosn = 0;

	    // Initialize the terrain representation
	    terrainBranchGroup = new BranchGroup();
	}  // end of initialiseVars()
	
	private void readFile(String file_name)
	// Initialize terrain[][][] by reading the map from file_name
	{
		String map_name = null; 
	    log.println( "Reading terrain plan from " + map_name, Logger.TERRAIN_GENERATION|Logger.NOTE );
	    BufferedReader reader = null;
	    String line;
	    try  
	    {
	    	map_name = "./maps/"+file_name+".map";
	    	reader = new BufferedReader( new FileReader( map_name ) );
	    	int numRows = 0;
	    	int height = -1;
	    	while( ( height < MAX_HEIGHT ) && ( ( line = reader.readLine() ) != null ) ) 
	    	{
	    		//If the MOD is 0, then it is the beginning of a new floor
	    		if(numRows%LENGTH == 0)
	    		{
	    			numRows = 0;
	    			height++;
	    		}
	    		int x=0;
	    		while((x < LENGTH) && (x < line.length())) 
	    		{ // ignore any extra chars
	    			terrain[numRows][x][height] = line.charAt(x);
	    			x++;
	    		}
	    		numRows++;
	    	}
	    	reader.close();
	    } 
	    catch( IOException e ) 
	    { 
	    	System.out.println("Error reading terrain plan from " + map_name+e.getMessage());
	    	System.exit(0);
	    }
	    
	    try
	    {
	    	map_name = "./maps/"+file_name+".objects";
	    	reader = new BufferedReader( new FileReader( map_name ) );
	    	
	    	while( !( line = reader.readLine() ).equalsIgnoreCase( "[end spawner]" ) ) // search the file for the [node] section
	    	{
	    		log.println(line);
	    	}
	    	
	    	while( !( line = reader.readLine() ).equalsIgnoreCase( "[end node]" ) ) 
	    	{
	    		if( line.equalsIgnoreCase( "[node]" ) )
	    		{
	    			log.println( "Loading nodes" );
	    		}
	    		else
	    		{
	    			split = line.split( " " );
	    			log.println( split[0]+","+split[1], Logger.NOTE+Logger.TERRAIN_GENERATION );
	    			objectManager.addNodeToTree( split );
	    		}
	    	}
	    	
	    	while( !( line = reader.readLine() ).equalsIgnoreCase( "[end link]" ) ) 
	    	{
	    		if( line.equalsIgnoreCase( "[link]" ) )
	    		{
	    			log.println( "Loading the tree" );
	    		}
	    		else
	    		{
	    			objectManager.addLinksToNode( line.split( " " ) );
	    		}
	    	}
	    	objectManager.printTree();
	    	reader.close();
	    }
	    catch( IOException e ) 
	    { 
	    	System.out.println( "Error reading terrain plan from " + map_name + ": " + e.getMessage() );
	    	System.exit(0);
	    }
	} // end of readFile()
	
	private void buildTerrainRepresentation() 
	/* Create a BranchGroup and BufferedImage to represent the terrain.

	   The terrain plan array will contain spaces, a single 's' (starting
	   point), and many 'c's and 'b's (cylinders, blocks).
	*/
	{
		log.println("Building terrain representations... please wait");
	    char characterAtPosition;

	    for( int y = 0; y < MAX_HEIGHT; y++ ) 
	    {
			for( int z = 0; z < LENGTH; z++ )
			{
				for( int x = 0; x < LENGTH; x++ )  // can do this because the world is square
				{
					characterAtPosition = terrain[z][x][y];
					switch( characterAtPosition )
					{
					case PLAYER_SPAWNER: // player's starting position
						xStartPosn = x+COORD_OFFSET;
						zStartPosn = z+COORD_OFFSET;
						yStartPosn = y;
						terrain[z][x][y] = ' '; // clear cell
						break;
					case STATIC_BLOCK: // b = block, o = small pillar, . = standard floor allowed 
					case CYLINDER_BLOCK: // to do this since they are both static objects, they are 
					case STANDARD_FLOOR_TYLE: // created by the same factory.
						terrainBranchGroup.addChild( Factory.getStaticObject( characterAtPosition, z+COORD_OFFSET, x+COORD_OFFSET, y, pickTool ) );
						break;
					case STANDARD_ROOF_TYLE:
						terrainBranchGroup.addChild( Factory.getStaticObject( STANDARD_FLOOR_TYLE, z+COORD_OFFSET, x+COORD_OFFSET, y, pickTool ) );
						break;
					case CASTER_SPAWNER:
					case FIGHTER_SPAWNER:
					case TRAPPER_SPAWNER:
					case GENERAL_SPAWNER:
						log.println( new Vector3d( x+COORD_OFFSET, y, z+COORD_OFFSET ).toString() );
						terrainBranchGroup.addChild( Factory.getNormalAI( new Vector3d( x+COORD_OFFSET, y, z+COORD_OFFSET ), log, objectManager, 
								new AmmoManager(null, terrainBranchGroup, this, Spell.FIRE, log, objectManager), this, pickTool, aiNumber++, genePool ) );
						terrain[z][x][y] = ' '; // clear cell
						break;
					case EMPTY_GROUND: // This means there is nothing in this space, so do nothing
						      // with it.
						break;
					default:
						// This covers things like line breaks and the like.  Basically, just ignore any
						// characters it doesn't recognize.
						break;
					}
				}
			}
		}
	} // end of buildTerrainRepresentation()
	
	public BranchGroup getTerrain() // called by Controller to get a reference to the 3D terrain
	{
		return terrainBranchGroup;
	}  // end of 

	public void addDynamicObject(Spell spell) 
	{
		
	}

	public Node getPlayerBody( PlatformGeometry platformGeometry, AmmoManager ammoManager ) 
	{
		Node node = Factory.getPlayer( platformGeometry, log, objectManager, ammoManager, this );
		return node;
	}
	
	public Vector3d getStartPosition() // called by Controller
	{
		return new Vector3d( xStartPosn, yStartPosn, zStartPosn);
	}  // end of 
	
	public boolean canMoveTo( Vector3d nextLocation )
	// is (path) free of obstacles?
	// Called by the KeyBehavior object to test a possible move.
	{
	    //Check to see if outside the floor's boundaries
	    if ((nextLocation.x < 0) || (nextLocation.x >= LENGTH) ||
	    		(nextLocation.z < 0) || (nextLocation.z >= LENGTH) || 
	    		(nextLocation.y >= MAX_HEIGHT)|| (nextLocation.y < 0))
	    {
	    	return false; // since outside the possible terrain dimensions
	    }

		int direction = 0;
		char contains = ' ';
		boolean zCollide = false;
	    if ((direction = closeToBoundryZ(nextLocation.z)) != 999)
	    {
	    	contains = terrain[(int) ( direction )][(int) Math.floor(nextLocation.x)][(int) nextLocation.y];
	    	log.println("z-direction contains "+contains);
	    	if( isCollidable(contains) )
	    	{
	    		zCollide = true;
	    	}
	    }
	    
	    boolean xCollide = false;
	    if((direction = closeToBoundryX(nextLocation.x)) != 999)
	    {
	    	contains = terrain[(int) Math.floor(nextLocation.z)][(int) (direction)][(int) nextLocation.y];
	    	log.println("x-direction contains "+contains);
	    	if( isCollidable(contains) )
	    	{
	    		xCollide = true;
	    	}
	    }
	    
	    /*
	    if((direction = closeToBoundryY(nextLocation.y)) != 0)
	    {
	    	contains = terrain[(int) Math.floor(originalLocation.z)][(int) Math.floor(originalLocation.x)][(int) (direction)];
	    	log.println("y-direction contains "+contains);
	    	if( isCollidable(contains) )
	    	{
	    		slidingVector.setY(0.0); // Every vector is actually a combination of 3 vectors.  This takes out the y-vector.
	    		log.println("The y needs to be flattened.");
	    	}
	    }// No movement in the y direction is possible, so don't need to worry about this.*/
	    
	    log.println(zCollide +","+xCollide);
	    return !(zCollide || xCollide); // if either of these are true, then it can't move, so is false, else it can move
	}  // end of canMoveTo()

	private int closeToBoundryX(double number) 
	{
		return closeToBoundry( number, 0.35, 0.63 );
	}  // end of closeToBoundryX()
	
	@SuppressWarnings("unused")
	private int closeToBoundryY(double number) 
	{
		return closeToBoundry( number, 0.0, 1.0 );
	}  // end of closeToBoundryY()
	
	private int closeToBoundryZ(double number) 
	{
		return closeToBoundry( number, 0.35, 0.63 );
	}  // end of closeToBoundryZ()
	
	private int closeToBoundry(double number, double lowerBounds, double upperBounds) 
	{
		if(number%1 > upperBounds)
		{ // Indicates that the moving object is too close to the upper bounds of this dimension.
			return (int) Math.floor(number)+1;
		}
		else if(number%1 < lowerBounds)
		{ // Indicates that the moving object is too close to the lower bounds of this dimension.
			return (int) Math.floor(number)-1;
		}
		else 
		{
			return 999; // Indicates that the moving object is in no danger of collision.
		}
	}  // end of closeToBoundry()
	
	private boolean isCollidable(char contains) 
	{
		boolean collidable = false;
		switch(contains)
		{
		case TerrainManager.STATIC_BLOCK:// b = block
			collidable = true;
			break;
		case TerrainManager.CYLINDER_BLOCK:// o = cylinder
			collidable = true;
			break;
		case TerrainManager.STANDARD_FLOOR_TYLE:// created by the same factory.
			collidable = true;
			break;
		case TerrainManager.STANDARD_ROOF_TYLE:
			collidable = true;
			break;
		case TerrainManager.DYNAMIC_BLOCK:
			collidable = true;
		case TerrainManager.KEY_DYNAMIC_BLOCK:
			collidable = true;
			break;
		case TerrainManager.LIGHT:
			collidable = false;
			break;
		case TerrainManager.LIGHT_AND_NODE_INDICATOR:
			collidable = false;
			break;
		case TerrainManager.CASTER_SPAWNER:
			collidable = true;
			break;
		case TerrainManager.FIGHTER_SPAWNER:
			collidable = true;
			break;
		case TerrainManager.TRAPPER_SPAWNER:
			collidable = true;
			break;
		case TerrainManager.GENERAL_SPAWNER:
			collidable = true;
			break;
		case TerrainManager.EMPTY_GROUND:
			collidable = false;
			break;
		default:
			log.println("There was an error encountered in isCollidable() = "+contains, Logger.EXCEPTION);
			System.exit(0);
			break;
		}
		return collidable;
	}

	public void printMap()
	{
		for (int y = 0; y < MAX_HEIGHT; y++) {
	    	log.println("Floor "+y);
			for (int z = 0; z < LENGTH; z++) {
				for (int x = 0; x < LENGTH; x++) {
					log.print(terrain[z][x][y]+"",Logger.NOTE);
				}
				log.println("");
			}
			log.println("");
	    }
	}  // end of printMap()

	public DynamicObjectManager getObjectManager() 
	{
		return objectManager;
	}
}
