﻿using System.Collections.Generic;
using GuitarTrainer.Interfaces;

namespace GuitarTrainer.Sound
{
    class MasterPlayer : IRealtimeSoundPlayer, IEventSoundPlayer
    {

        private int timerFrequency = 4;
        private bool noteEventsEnabled;
        private ISoundPlayer soundPlayer;

        public int TimerFrequency
        {
            set { timerFrequency = value; }
        }

        public bool EnableNoteEvents
        {
            set { noteEventsEnabled = value; }
        }

        public void addEventListener(IPerformanceEventListener listener)
        {
            ((IEventSoundPlayer)soundPlayer).addEventListener(listener);
        }

        public void addTimerListener(IPerformanceTimerListener listener)
        {
            ((IRealtimeSoundPlayer)soundPlayer).addTimerListener(listener);
        }

        /// <summary>
        /// The goal is to arrange a performance for various events. This was never finalized
        /// since the program currently runs event-less. It currently relies on time spans and
        /// actual time deruation instead of being event driven.
        /// </summary>
        /// <param name="song">The song to arrange over</param>
        /// <param name="arr">The arrangement this will be linked to</param>
        /// <returns></returns>
        public IPerformance arrange(ISong song, IArrangement arr)
        {
            IArrangement arrangement;

            arrangement = arr ?? new DefaultArrangement(song);

            int tracks = song.TrackCount;
            List<int> arrangedMeasures = arrangement.MeasureList;
            int measures = arrangedMeasures.Count;

            if (soundPlayer is ITimerSettings)
                ((ITimerSettings)soundPlayer).TimerFrequency = timerFrequency;
            else if (soundPlayer is IEventSettings)
                ((IEventSettings)soundPlayer).EnableNoteEvents = noteEventsEnabled;

            IPerformance performance = soundPlayer.createPerformance(tracks, song.Tempo, song.Resolution);
            performance.TimerFrequency = timerFrequency;

            // Add timing events to the performance
            TimeSignature lastTimeSignature = null;
            int location = 0;
            for (int m = 0; m < measures; m++)
            {
                int measureIndex = arrangedMeasures[m];
                ISongMeasure measure = song.getPerformanceMeasure(measureIndex);

                TimeSignature currentTimeSignature = measure.getTimeSignature();
                if ((lastTimeSignature == null) || (!currentTimeSignature.Equals(lastTimeSignature)))
                {
                    performance.setTimeSignature(location, currentTimeSignature);
                    lastTimeSignature = currentTimeSignature;
                }

                performance.addTimerEvents(measure, location);
                location += measure.getLength();
            }

            // Loop for all tracks
            for (int t = 1; t <= tracks; t++)
            {
                ISongTrack songTrack = song.getTrack(t);
                performance.initializeTrack(songTrack);

                int virtualTracks = songTrack.VirtualTrackCount;

                // Loop for all virtual tracks
                for (int v = 0; v < virtualTracks; v++)
                {
                    EventStream es = new EventStream(songTrack, v);
                    location = 0;
                    for (int m = 0; m < measures; m++)
                    {
                        int measureIndex = arrangedMeasures[m];
                        ISongMeasure measure = song.getPerformanceMeasure(measureIndex);

                        ISongMeasureTrack smt = measure.getTrack(songTrack);
                        ISongVirtualTrack svt = smt.getVirtualTrack(v);
                        List<ISongEvent> events = svt.Events;
                        es.addEvents(events, location);

                        location += measure.getLength();
                    }
                    es.close();
                    performance.addMusicalEvents(es);
                }
            }

            soundPlayer.Performance = performance;
            return performance;
        }

        public void start()
        {
            soundPlayer.start();
        }

        public void stop()
        {
            ((IRealtimeSoundPlayer)soundPlayer).stop();
        }

        public void waitForCompletion()
        {
            ((IRealtimeSoundPlayer)soundPlayer).waitForCompletion();
        }

        public void close()
        {
            soundPlayer.close();
        }

        public IPerformance createPerformance(int tracks, Tempo tempo, int resolution)
        {
            // placeholder
            return null;
        }

        public IPerformance Performance
        {
            set { soundPlayer.Performance = value; }
        }
    }
}
