﻿using System;
using System.Collections.Generic;
using System.Windows.Forms;
using GuitarTrainer.GuitarProComponents;
using GuitarTrainer.InterfaceImpl;

namespace GuitarTrainer.Interfaces
{
    /// <summary>
    /// An adaptor to give Guitar Pro pieces the ISong interface
    /// </summary>
    static class GPAdaptor
    {
        /// <summary>
        /// The following definition of pulses per quarter note is for high audio resolution.
        /// It is to support dotted 128th notes and n-tuplet types up to 13. In "real" music,
        /// no one would ever write (or be able to play) something like this, though.
        /// </summary>
        private const int PPQ_HIGH_RESOLUTION = 2882880;

        /// <summary>
        /// Instead a compromise is made. Initially the adaptor is run in high precision but 
        /// a score will be generated at a reasonable scale.
        /// </summary>
        private const int PPQ_SCALE_FACTOR = 286;

        private static int measureIndex;
        private static int mtpIndex;
        private static List<GPMeasure> measures;
        private static List<GPMeasureTrackPair> measureTrackPairs;
        private static List<GPTrack> tracks;

        public static ISong makeSong(GPSong piece)
        {
            Tempo tempo = new Tempo {BPM = piece.getTempo()};

            ISong song = new SongImpl(PPQ_HIGH_RESOLUTION / PPQ_SCALE_FACTOR, tempo);

            measures = piece.measures;
            tracks = piece.tracks;

            int index = 1;
            foreach (GPTrack t in tracks)
            {
                int port = t.Port;
                int channel = t.Channel;
                int channelEffects = t.ChannelEffects;

                int channelIndex = (port - 1) * 16 + (channel - 1);
                GPMIDIChannel mc = piece.getChannels(channelIndex);

                int strings = t.NumberOfStrings;

                ISongTrack st = new SongTrackImpl(index, strings)
                                   {
                                       PrimaryDevice = new SongDeviceImpl(port, channelEffects)
                                   };

                if (channel != channelEffects)
                    st.SecondaryDevice = new SongDeviceImpl(port, channelEffects);

                st.BendSensitivity = 2;

                st.Volume = 8 * mc.Volume;
                st.Pan = 8 * mc.Balance;
                st.Chorus = 8 * mc.Reverb;
                st.Reverb = 8 * mc.Reverb;
                st.Tremolo = 8 * mc.Tremolo;
                st.Phaser = 8 * mc.Phaser;

                song.addTrack(st);

                index++;
            }

            measureTrackPairs = piece.measuresTracksPairs;

            measureIndex = 0;
            mtpIndex = 0;

            foreach (GPMeasure m in measures)
            {
                ISongPhrase sp = phraseFactory(song);
                song.addPhrase(sp);
            }

            return song;
        }

        private static ISongPhrase phraseFactory(ISong song)
        {
            ISongPhrase sp;
            GPMeasure measure;

            // "Peek" and get the next measure without advancing the index
            try
            {
                measure = measures[measureIndex];
            }
            catch (Exception)
            {
                throw;
            }

            // If the measure is a repeat measure then handle it accordingly
            if (measure.RepeatStart)
                sp = makeRepeatedSongPhrase(song);
            // It's not a repeat measure so just return a single, regular measure
            else
                sp = makeSongMeasure(song);

            return sp;
        }

        /// <summary>
        /// Makes a repeated song phrase for the song
        /// </summary>
        /// <param name="song">The song to created the repeated phrase on</param>
        /// <returns>The repeated song phrase created</returns>
        private static IRepeatedSongPhrase makeRepeatedSongPhrase(ISong song)
        {
            ISongPhraseList spl = new SongPhraseListImpl();

            // Read the first measure without testing for the repeat marker again
            spl.addPhrase(makeSongMeasure(song));

            // Check to see if we're done
            GPMeasure measure = measures[measureIndex];
            //measureIndex++;
            int repeatCount = measure.NumberOfRepititions;

            // Keep calling phraseFactory until a measure is found that is marked with a repeat
            // count. If there are no more measures then assume the final measure has a repeat of 1
            while (repeatCount == 0)
            {
                if (measureIndex > measures.Count)
                {
                    MessageBox.Show("Got to the end before closing repeat mark. Assuming 1");
                    repeatCount = 1;
                    break;
                }

                ISongPhrase phrase = phraseFactory(song);
                spl.addPhrase(phrase);
                measure = measures[measureIndex];
                //measureIndex++;
                repeatCount = measure.NumberOfRepititions;
            }

            return new RepeatedSongPhraseImpl(spl, repeatCount);
        }

        /// <summary>
        /// Makes a measure for the song
        /// </summary>
        /// <param name="song">The song to create the measure for</param>
        /// <returns>The measure created</returns>
        private static ISongMeasure makeSongMeasure(ISong song)
        {
            GPMeasure measure = measures[measureIndex++];

            int numerator = measure.Numerator;
            int denominator = measure.Denominator;

            int measureLength = PPQ_HIGH_RESOLUTION * 4 * numerator / denominator;

            TimeSignature ts = new TimeSignature(numerator, denominator);
            ISongMeasure sm = new SongMeasureImpl(measure.MeasureNumber, measureLength / PPQ_SCALE_FACTOR, ts);

            for (int trackIndex = 1; trackIndex <= tracks.Count; trackIndex++)
            {
                GPTrack track = tracks[trackIndex - 1];
                int capo = track.Capo;
                int strings = track.NumberOfStrings;
                int[] stringTuning = new int[strings];

                for (int i = 0; i < strings; i++)
                    stringTuning[i] = track.getStringsTuning(i);

                GPMeasureTrackPair mtp = measureTrackPairs[mtpIndex++];
                ISongMeasureTrack smt = new SongMeasureTrackImpl(song.getTrack(trackIndex));
                sm.addTrack(smt);

                List<GPBeat> beats = mtp.Beats;
                int beatOffset = 0;

                foreach (GPBeat beat in beats)
                {
                    int beatDuration = calculateDuration(beat.Duration, beat.DottedNotes, beat.NTuplet);

                    if (beat.IsNoteBeat)
                    {
                        List<GPNote> notes = beat.Notes;
                        List<int> stringsPlayed = new List<int>();

                        for (int i = 6; i >= 0; i--)
                            if (beat.isStringPlayed(i))
                                stringsPlayed.Add(i);

                        int stringIndex = 0;

                        foreach (GPNote note in notes)
                        {
                            if (stringsPlayed.Count < stringIndex)
                                break;

                            int stringNum = stringsPlayed[stringIndex];
                            int stringID = 6 - stringNum;
                            int noteDuration = beatDuration;

                            GPDuration tiDuration = note.Duration;

                            if (tiDuration != null)
                            {
                                bool tiDotted = note.IsDotted;
                                int tiTriplet = note.NTuplet;
                                noteDuration = calculateDuration(tiDuration, tiDotted, tiTriplet);
                            }

                            int virtualTrackID = stringID;
                            ISongVirtualTrack svt = smt.getVirtualTrack(virtualTrackID);
                            int noteStart = beatOffset/PPQ_SCALE_FACTOR;
                            int noteEnd = (beatOffset + noteDuration)/PPQ_SCALE_FACTOR;

                            if (note.IsTieNote)
                                svt.addEvent(new SongEventImpl(noteStart, new SongTieMessage(noteEnd - noteStart)));
                            else
                            {
                                int baseFret = note.FretNumber;
                                int fret = baseFret + capo;
                                GPDynamic dynamic = note.Dynamic;
                                int velocity = DynamicsHash.velocityOf(dynamic);
                                int pitch = stringTuning[stringID] + fret;

                                svt.addEvent(new SongEventImpl(noteStart,
                                                               new SongNoteOnMessage(pitch, velocity,
                                                                                     noteEnd - noteStart, baseFret)));
                            }

                            stringIndex++;
                        }
                    }
                    beatOffset += beatDuration;
                }

                if (beatOffset == measureLength) continue;

                // An under-fill is not bad, but an over-fill is.
                // An under-fill usually means a track hasn't had any notes yet or it has played all its notes.
                // If that is the case, GuitarPro has only one beat in the measure (a quarter rest) which will,
                // obviously, not add up to a full rest.
                if (beatOffset > measureLength)
                {
                    Environment.Exit(0);
                }
            }
            return sm;
        }

        /// <summary>
        /// Calculates the duration of a note
        /// </summary>
        /// <param name="duration">The duration of the note</param>
        /// <param name="dotted">If the note is dotted or not</param>
        /// <param name="nTuplet">If the note is a n-tuplet</param>
        /// <returns>The duration of the note</returns>
        private static int calculateDuration(GPDuration duration, bool dotted, int nTuplet)
        {
            int durationIndex = duration.getIndex();
            int beatDuration = (PPQ_HIGH_RESOLUTION * 4) >> durationIndex;
            if (dotted)
                beatDuration += (beatDuration >> 1);
            if (nTuplet != 0)
            {
                TripletValue tv = TripletHash.getTriplet(nTuplet);
                beatDuration *= tv.TimeOf;
                beatDuration /= tv.Notes;
            }
            return beatDuration;
        }
    }
}
