﻿using System;

namespace GuitarTrainer.GuitarProComponents
{
    // The class that describes a bend's type read from a Guitar Pro file.
    class GPBendType
    {
        // The bend type is represented by an integer
        private readonly int type;

        // The different bend types this class can handle
        public static readonly GPBendType NONE = new GPBendType(0);
        public static readonly GPBendType BEND = new GPBendType(1);
        public static readonly GPBendType BEND_RELEASE = new GPBendType(2);
        public static readonly GPBendType BEND_RELEASE_BEND = new GPBendType(3);
        public static readonly GPBendType PREBEND = new GPBendType(4);
        public static readonly GPBendType PREBEND_RELEASE = new GPBendType(5);
        
        // The following are tremolo bar specific type of bends
        public static readonly GPBendType DIP = new GPBendType(6);
        public static readonly GPBendType DIVE = new GPBendType(7);
        public static readonly GPBendType RELEASE_UP = new GPBendType(8);
        public static readonly GPBendType INVERTED_DIP = new GPBendType(9);
        public static readonly GPBendType RETURN_BEND = new GPBendType(10);
        public static readonly GPBendType RELEASE_DOWN = new GPBendType(11);

        // The bend types
        private static readonly GPBendType[] bendTypes = { NONE, BEND, BEND_RELEASE,
            BEND_RELEASE_BEND, PREBEND, PREBEND_RELEASE, DIP, DIVE, RELEASE_UP,
            INVERTED_DIP, RETURN_BEND, RELEASE_DOWN };

        // The bend types' names
        private static readonly string[] bendTypesNames = { "None", "Bend",
            "Bend and Release", "Bend and Release and Bend", "Prebend",
            "Prebend and Release", "Dip", "Dive", "Release (up)",
            "Inverted dip", "Return", "Release (down)" };

        /// <summary>
        /// Create a new bend type whose type is the paramenter 'type'
        /// </summary>
        /// <param name="type">The type of this bend</param>
        private GPBendType(int type)
        {
            this.type = type;
        }

        /// <summary>
        /// Returns true if the parameter object equals this
        /// </summary>
        /// <param name="o">The object to test</param>
        /// <returns>True if the object equals this, false otherwise</returns>
        public override bool Equals(Object o)
        {
            return (o is GPBendType ? type == ((GPBendType)o).type : false);
        }

        /// <summary>
        /// Returns the gpbend type assocated with the value type
        /// </summary>
        /// <param name="type">The type of bend to look for</param>
        /// <returns>The BendType that the type represents</returns>
        public static GPBendType valueOf(int type)
        {
            return bendTypes[type];
        }
        
        /// <summary>
        /// Returns a string representation of this bend type
        /// </summary>
        /// <returns>The string representation of type type this class is associated with</returns>
        public string toString()
        {
            return bendTypesNames[type];
        }
    }
}
