﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Windows.Forms;
using GuitarTrainer.GuitarProComponents;

namespace GuitarTrainer.GUI
{
    sealed class TrackPanel : Panel
    {
        // How far from the left or right is the start of the track
        private int leftMargin;
        private const int RIGHT_MARGIN = 15;

        private readonly Label trebelLabel;

        private readonly BarPanel tablaturePanel;

        private int numberOfBars;

        private Size clefSize;
        private const int SPACE_BETWEEN_CLEF_AND_BAR = 5;

        private readonly DisplayOptions displayOptions;
        private readonly MusicCursor tablatureMusicCursor;
        private GPTrack track;

        /// <summary>
        /// Default constructor
        /// </summary>
        public TrackPanel()
        {
            displayOptions = new DisplayOptions();
            track = null;

            BackColor = Color.White;

            trebelLabel = new Label {Image = Form1.clefs[0]};

            tablaturePanel = new BarPanel(displayOptions);
            numberOfBars = 0;

            arrange();

            tablaturePanel.ExtendTop = false;
            tablaturePanel.ExtendBottom = false;

            tablaturePanel.BackgroundColor = Color.White;
            tablaturePanel.ForegroundColor = Color.Black;

            Controls.Add(tablaturePanel);
            tablaturePanel.Visible = true;
        }

        /// <summary>
        /// Constructor with an established display options
        /// </summary>
        /// <param name="displayOptions">The display options to be used</param>
        public TrackPanel(DisplayOptions displayOptions)
        {
            if (displayOptions != null)
            {
                this.displayOptions = displayOptions;
                track = null;

                BackColor = Color.White;

                trebelLabel = new Label {Image = Form1.clefs[0]};

                tablaturePanel = new BarPanel(displayOptions);
                numberOfBars = 0;
                arrange();

                tablaturePanel.ExtendTop = false;
                tablaturePanel.ExtendBottom = false;

                tablaturePanel.ForegroundColor = Color.Black;

                // Music cursor must be added to the controls BEFORE tablaturePanel to 
                // ensure the music cursor gets drawn above the tablature panel.
                tablatureMusicCursor = new MusicCursor(tablaturePanel) {Visible = displayOptions.DisplayMusicCursor};


                Controls.Add(tablaturePanel);
                tablaturePanel.Size = new Size(680, 90);
                tablaturePanel.Visible = true;


                Controls.Add(tablatureMusicCursor);
            }
            else
            {
                MessageBox.Show("DisplayOptions passed is null. TrackPanel() (TrackPanel.cs)");
                Environment.Exit(0);
            }
        }

        /// <summary>
        /// Arranges this track panel to look nice
        /// </summary>
        private void arrange()
        {
            Size s = tablaturePanel.MinimumSize;

            if (clefSize.Height == 0 && clefSize.Width == 0)
                clefSize = trebelLabel.PreferredSize;

            int y = (s.Height - clefSize.Height) / 2;
            trebelLabel.SetBounds(leftMargin, y, clefSize.Width, clefSize.Height);

            int x = leftMargin + trebelLabel.Width + SPACE_BETWEEN_CLEF_AND_BAR;

            tablaturePanel.SetBounds(x, 1, s.Width, s.Height);
        }

        /// <summary>
        /// Adds a tablature bar to this track panel
        /// </summary>
        /// <param name="n"></param>
        private void addTablatureBar(int n)
        {
            if (n < 1) return;

            BarTablature bar = new BarTablature(displayOptions, tablaturePanel)
                                   {
                                       IsFirstBar = true,
                                       IsLastBar = (n == 1)
                                   };
            tablaturePanel.addBar(bar);

            if (n < 2) return;

            for (int i = 1; i < n - 1; i++)
            {
                bar = new BarTablature(displayOptions, tablaturePanel);
                tablaturePanel.addBar(bar);

                Width += 680;
            }

            bar = new BarTablature(displayOptions, tablaturePanel) {IsLastBar = true};
            tablaturePanel.addBar(bar);
        }

        /// <summary>
        /// Adds a new bar and increases the bar count by one.
        /// </summary>
        /// <param name="n"></param>
        public void addBar(int n)
        {
            addTablatureBar(n);
            numberOfBars += n;
        }

        #region Getters and setters

        public override Size MinimumSize
        {
            get
            {
                Size s2 = tablaturePanel.MinimumSize;
                int w = leftMargin + clefSize.Width + SPACE_BETWEEN_CLEF_AND_BAR + s2.Width + RIGHT_MARGIN;
                int h = s2.Height;

                Size s = new Size(w, h);

                return s;
            }
        }

        public GPTrack Track
        {
            set 
            { 
                track = value;
                
                setMusicCursorSingleColor(displayOptions.SingleMusicCursorColor);
                tablaturePanel.Track = value;
            }
        }

        private void setMusicCursorSingleColor(bool singleColor)
        {
            Color c;

            if (singleColor) return;

            GPColor gpc = track.Color;

            try
            {
                c = Color.FromArgb(gpc.red, gpc.green, gpc.blue);
            }
            catch (Exception)
            {
                c = Color.White;
            }

            if (c.Equals(Color.White))
                displayOptions.MusicCursorColor = c;
        }

        public List<GPMeasure> Measures
        {
            set { tablaturePanel.Measures = value; }
        }
        public void setMeasureTrackPair(int measureNum, GPMeasureTrackPair mtp)
        {
            tablaturePanel.setMeasureTrackPair(measureNum, mtp);
        }
        public Rectangle setMusicCursor(int measureNum, int prevMeasure, bool scroll)
        {
            MusicCursorPosition mcp = tablaturePanel.getMusicCursorPosition(measureNum, tablatureMusicCursor);
            Point p = mcp.Point;
            Rectangle rect = tablatureMusicCursor.Bounds;
            bool wasVisible = tablatureMusicCursor.Visible;

            p.X += leftMargin + trebelLabel.Width + SPACE_BETWEEN_CLEF_AND_BAR + 100;
            mcp.Point = p;

            if(wasVisible)
                tablatureMusicCursor.Visible = false;

            if (scroll)
            {
                if (measureNum > prevMeasure)
                    rect.X += rect.Width + 250;
                else
                    rect.Width = 1;
            }

            tablatureMusicCursor.MusicCursorPosition = mcp;

            if (wasVisible)
                tablatureMusicCursor.Visible = true;

            return rect;
        }
        public BarMtp getTablatureBar(int measure)
        {
            Bar bar = tablaturePanel.getBar(measure);
            BarMtp bmtp = null;

            try
            {
                bmtp = (BarMtp)bar;
            }
            catch (Exception)
            {
                MessageBox.Show("Error casting bar to barmtp getTablatureBar() (TrackPanel.cs)");
                Environment.Exit(0);
            }
            return bmtp;
        }
        #endregion
    }
}
