﻿using System;
using System.IO;
using System.Windows.Forms;
using GuitarTrainer.GuitarProComponents;

namespace GuitarTrainer.GP4File
{
    class GPInputStream
    {
        // The filename (equals "" if not available).
        private readonly string fileName;

        // File version
        protected string Version;

        // File filter
        private readonly CodecFileFilter codecFileFilter;

        // The input stream from which the file is read
        private readonly BinaryReader fIn;

        // The offset where we have gotten so far in the file
        private int offset;

        /// <summary>
        /// Creates a new GPInputStream from an existing stream
        /// </summary>
        /// <param name="fIn">The stream to copy</param>
        protected GPInputStream(BinaryReader fIn): this("", fIn)
        {}

        /// <summary>
        /// Creates a new GPInputStream from an existing stream
        /// </summary>
        /// <param name="fileName">The name of the file the BinaryReader is reading</param>
        /// <param name="fIn">The binary reader</param>
        private GPInputStream(string fileName, BinaryReader fIn)
        {
            this.fileName = fileName;
            Version = "";

            codecFileFilter = null;
            this.fIn = fIn;
            offset = 0;
        }

        /// <summary>
        /// Creates a new GPInputStream by cloning an existing GPInputStream. This is
        /// very useful when the method readPiece() is called. It starts by reading the 
        /// version of GuitarPro that the file was encoded with. It then starts decoding
        /// it accordingly. Since there are more file formats than GP4 (like GP3 or 2), 
        /// it could use different versions
        /// </summary>
        /// <param name="gpIn">The input stream to clone</param>
        protected GPInputStream(GPInputStream gpIn)
        {
            fileName = gpIn.fileName;
            Version = gpIn.Version;
            codecFileFilter = gpIn.codecFileFilter;
            fIn = gpIn.fIn;
            offset = gpIn.offset;
        }

        /// <summary>
        /// Closes the binary reader stream
        /// </summary>
        public void close()
        {
            fIn.Close();
        }

        /// <summary>
        /// Reads the next byte of data from the stream reader
        /// </summary>
        /// <returns>The byte read in</returns>
        protected int read()
        {
            offset++;
            try
            {
                return fIn.ReadByte();
            }
            catch (IOException e)
            {
                MessageBox.Show("Error reading file\n" + e);
                Environment.Exit(0);
            }

            // We should never hit this point
            return -1;
        }

        /// <summary>
        /// Reads some number of bytes from the input stream and stores them into
        /// the buffer array b
        /// </summary>
        /// <param name="b">The (passed by reference) byte array to place bytes into</param>
        protected void read(ref byte[] b)
        {
            try
            {
                offset += b.Length;
                b = fIn.ReadBytes(b.Length);
            }
            catch (IOException)
            {
                MessageBox.Show("Error reading byte stream");
                Environment.Exit(0);
            }
        }

        /// <summary>
        /// Skips over an discards n bytes of data from this input stream
        /// </summary>
        /// <param name="n">How many bytes we should skip</param>
        protected void skip(long n)
        {
            try
            {
                for (int i = 0; i < n; i++)
                    fIn.ReadByte();
            }
            catch (IOException)
            {
                MessageBox.Show("Error skipping bytes");
                Environment.Exit(0);
            }
        }

        /// <summary>
        /// Reads a boolean from the stream reader
        /// </summary>
        /// <returns>If a 1 was read in, returns true. If a 0 was read in, returns false</returns>
        protected bool readBoolean()
        {
            bool toReturn = false;
            try
            {
                toReturn = read() == 1;
            }
            catch (IOException)
            {
                MessageBox.Show("Error reading boolean");
                Environment.Exit(0);
            }

            return toReturn;
        }

        /// <summary>
        /// Reads a byte from the stream reader
        /// </summary>
        /// <returns>The signed byte read in</returns>
        protected sbyte readByte()
        {
            try
            {
                return (sbyte)read();
            }
            catch (IOException)
            {
                MessageBox.Show("Error reading byte");
                Environment.Exit(0);
            }
            return 1;
        }

        /// <summary>
        /// Reads a least significant bit integer from the stream reader
        /// </summary>
        /// <returns>The int read in</returns>
        protected int readInt()
        {
            byte[] b = { 0, 0, 0, 0 };

            try
            {
                // Could be something wrong here?
                read(ref b);
            }
            catch (IOException)
            {
                MessageBox.Show("Error reading int");
                Environment.Exit(0);
            }

            return ((b[3] & 0xff) << 24) | ((b[2] & 0xff) << 16)
                          | ((b[1] & 0xff) << 8) | (b[0] & 0xff);
        }

        /// <summary>
        /// Reads a string from the stream reader. Its length is given by a byte
        /// </summary>
        /// <param name="expectedLength">The expected length of the string</param>
        /// <returns>The string read in</returns>
        public string readStringByte(int expectedLength)
        {
            byte[] b;
            int realLength = 1;
            try
            {
                realLength = readUnsignedByte();

                b = expectedLength != 0 ? new byte[expectedLength] : new byte[realLength];
                read(ref b);
            }
            catch (IOException)
            {
                MessageBox.Show("Error reading string byte");
                Environment.Exit(0);

                // Added to get to compile
                b = new byte[expectedLength];
            }

            string str = System.Text.Encoding.ASCII.GetString(b);
            return str.Substring(0, realLength);
        }

        /// <summary>
        /// Return a string from the stream reader. Its length is given by an integer
        /// </summary>
        /// <returns>The numerical string read in</returns>
        protected string readStringInteger()
        {
            byte[] b;

            try
            {
                int length = readInt();
                b = new byte[length];
                read(ref b);
            }
            catch (IOException e)
            {
                MessageBox.Show("Error reading string integer\n" + e);
                Environment.Exit(0);

                // Added to get to compile
                b = new byte[1];
            }

            return System.Text.Encoding.ASCII.GetString(b);
        }

        /// <summary>
        /// Reads a string from the stream reader. Its length is obtained by
        /// subtracting 1 to an integer read
        /// </summary>
        /// <returns>The string read in</returns>
        protected string readStringIntegerPlusOne()
        {
            string str = "";

            try
            {
                int lengthPlusOne = readInt();
                int length = lengthPlusOne - 1;
                int r;
                if (lengthPlusOne > 0)
                {
                    // Reads the real length as a byte
                    r = read();
                    if (length != r)
                    {
                        throw new IOException("Wrong string length: should be " + length);
                    }

                    byte[] b = new byte[length];
                    read(ref b);

                    str = System.Text.Encoding.ASCII.GetString(b);
                }
                else
                {
                    read();
                    str = "";
                }
            }
            catch (IOException e)
            {
                MessageBox.Show("Error reading string integer plus one\n" + e);
                Environment.Exit(0);
            }

            return str;
        }

        /// <summary>
        /// Reads an unsigned byte from the stream
        /// </summary>
        /// <returns>The unsigned byte read in</returns>
        protected int readUnsignedByte()
        {
            int toReturn = 1;
            try
            {
                toReturn = read();
            }
            catch (IOException)
            {
                MessageBox.Show("Error reading unsigned byte");
                Environment.Exit(0);
            }

            return toReturn;
        }
    }
}
