﻿using System;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.IO;
using System.Windows.Forms;
using System.Threading;
using System.Text.RegularExpressions;
using GuitarTrainer.GP4File;
using GuitarTrainer.GUI;
using GuitarTrainer.Interfaces;
using WMPLib;

namespace GuitarTrainer
{
    public partial class Form1 : Form
    {
        #region Instance variables

        public const short NOT_PLAYING = 1;
        public const short PLAYING = 2;

        private static short currStatus;

        // Current launch path
        private string launchPath;

        // The clefs
        public static readonly Image[] clefs = new Image[2];
        // Rests
        public static readonly Image[] silences = new Image[7];
        // Notes
        public static readonly Image[] notes = new Image[8];
        // Dynamics
        public static readonly Image[] dynamics = new Image[8];
        // Effects on note
        public static Image[] eons = new Image[7];

        // The directory where the skins are saved
        private string skinDir, skinPath;

        // The stream for reading in guitar pro files
        BinaryReader br;
        GP4InputStream inStr;

        // Main menu
        MainMenu menu;

        // Play/pause button tool strip
        ToolStrip ts;

        // File open dialog for GP4 files
        private readonly OpenFileDialog gp4Diag;
        private readonly OpenFileDialog mp3Diag;

        public bool evaluate;

        //SongPanel sp;
        private SongInternalFrame sif;
        private ISong song;
        private Tempo tempo;
        private SongPanel sp;

        private DateTime start;

        private int wholeNoteInMilliSeconds;

        private WindowsMediaPlayer wplayer;

        #endregion

        public Form1()
        {
            inStr = new GP4InputStream(br);

            //loadImages();
            AutoScroll = true;

            // Creates the main menu (file, edit, etc)
            createMainMenu();
            Menu = menu;

            // Creates the tool strip (with play/pause buttons)
            createToolStrip();
            Controls.Add(ts);

            gp4Diag = new OpenFileDialog
                          {
                              Filter = "GuitarPro4 Files (*.gp4)|*.gp4",
                              Title = "Select a GuitarPro 4 files"
                          };

            mp3Diag = new OpenFileDialog
                          {
                              Filter = "MP3 Files (*.mp3)|*.mp3",
                              Title = "Select an MP3 file to play with your GP4 file"
                          };

            InitializeComponent();

            Text = "GuiarTrainer";

            backgroundWorker1.WorkerReportsProgress = true;
            backgroundWorker1.WorkerSupportsCancellation = true;

            Application.DoEvents();
        }

        public override sealed string Text
        {
            get { return base.Text; }
            set { base.Text = value; }
        }

        public override sealed bool AutoScroll
        {
            get { return base.AutoScroll; }
            set { base.AutoScroll = value; }
        }

        /// <summary>
        /// Creates the toolstrip for the main form
        /// </summary>
        private void createToolStrip()
        {
            ts = new ToolStrip {Dock = DockStyle.Top};

            var playButton = new ToolStripButton
            {
                Text = "Play",
                DisplayStyle = ToolStripItemDisplayStyle.Text,
                Visible = true
            };

            var stopButton = new ToolStripButton
            {
                Text = "Stop",
                DisplayStyle = ToolStripItemDisplayStyle.Text,
                Visible = true
            };

            var helpButton = new ToolStripButton
            {
                Text = "Find Lessons",
                DisplayStyle = ToolStripItemDisplayStyle.Text,
                Anchor = AnchorStyles.Right,
                Visible = true,
            };

            ts.Items.Add(playButton);
            ts.Items.Add(stopButton);
            ts.Items.Add(helpButton);

            playButton.Click += playButton_Click;
            stopButton.Click += stopButton_Click;
            helpButton.Click += helpButton_Click;
        }

        /// <summary>
        /// Creates the main menu for the main form
        /// </summary>
        private void createMainMenu()
        {
            menu = new MainMenu();
            

            // Top level menus
            MenuItem fileMenu = new MenuItem("File");
            menu.MenuItems.Add(fileMenu);
            MenuItem helpMenu = new MenuItem("Help");
            menu.MenuItems.Add(helpMenu);

            // Sub menus
            MenuItem openSubMenu = new MenuItem("Open");
            fileMenu.MenuItems.Add(openSubMenu);
            MenuItem exitSubMenu = new MenuItem("Exit");
            fileMenu.MenuItems.Add(exitSubMenu);
            MenuItem instructionsSubMenu = new MenuItem("Instructions");
            helpMenu.MenuItems.Add(instructionsSubMenu);
            MenuItem colorSubMenu = new MenuItem("Colors");
            helpMenu.MenuItems.Add(colorSubMenu);
            MenuItem aboutSubMenu = new MenuItem("About");
            helpMenu.MenuItems.Add(aboutSubMenu);

            // Add event handlers for the various actions
            exitSubMenu.Click += exitSubMenu_Click;
            openSubMenu.Click += openSubMenu_Click;
            aboutSubMenu.Click += aboutSubMenu_Click;
            instructionsSubMenu.Click += instructionsSubMenu_Click;
            colorSubMenu.Click += colorSubMenu_Click;
        }

        /// <summary>
        /// Opens the file stream and initiates the reading of the GuitarPro file
        /// </summary>
        /// <param name="binReader">The binary stream to read the GP file</param>
        private void readAndShow(BinaryReader binReader)
        {
            GP4InputStream stream = new GP4InputStream(binReader);
            sif = new SongInternalFrame(this, new DisplayOptions(), DateTime.Now);
            bool success = sif.readAndDisplay(stream);

            Size = new Size(1024, 786);

            if (success)
            {
                Controls.Add(sif);
                sif.Visible = true;
                /*
                 * This is how the panels are layed out:
                 * this form -> SongInteralFrame -> SongPanel -> TrackPanel -> BarPanel -> BarTablature
                 */
            }
            else
            {
                MessageBox.Show("Error reading file in");
                Environment.Exit(0);
            }
            try
            {
                stream.close();
            }
            catch (IOException)
            {
                MessageBox.Show("error closing file");
                Environment.Exit(0);
            }

            // Make sure we're not going to start playing immediately
            currStatus = NOT_PLAYING;
        }

        #region Menu event handlers

        private static void colorSubMenu_Click(object who, EventArgs e)
        {
            string message = "Note colors are as follows:";
            message += "\n\nCyan = whole note";
            message += "\nBlue = half note";
            message += "\nBlack = quarter note";
            message += "\nRed = eighth note";
            message += "\nGreen = 16th note";
            message += "\nOlive = 32nd note";
            message += "\nGold = 64th note";
            message += "\nYellow = 128th note";

            MessageBox.Show(message);
        }
        private static void instructionsSubMenu_Click(object who, EventArgs e)
        {
            string message = "This program accepts GuitarPro v4.x files. These can be identified by their *.gp4 extensions.";
            message += " In this program, a GuitarPro file is exptected to have an MP3 accompany it, supplied by you.";
            message += "\n\n1. To begin, click 'File', then 'Open'. You will be presented with a dialog to pick a file. Select any ";
            message += "gp4 file and click 'Ok'. \n2. Immediately following, a second dialog to choose a file will appear. Select ";
            message += "the mp3 file that goes along with the gp4 file you selected in step 1. \n3. If the file and mp3 loaded correctly";
            message += " then the song will display in the main application. You will be notified if something does not load";
            message += " properly. \n4. To begin playback, press the 'Play' button and the song will automatically begin.";
            message += "\n\nAt any time you may press the stop button to end playback of the song or load a new song.";

            MessageBox.Show(message);
        }
        private static void aboutSubMenu_Click(object who, EventArgs e)
        {
            string message = "Written by Adam Janke for his Computer Science capstone at the University of Alaska Anchorage";
            message += "\nAdamJanke@gmail.com";
            MessageBox.Show(message);
        }
        private static void exitSubMenu_Click(object who, EventArgs e)
        {
            Environment.Exit(0);
        }
        private void openSubMenu_Click(object who, EventArgs e)
        {
            if(gp4Diag.ShowDialog() == DialogResult.Cancel)
                return;

            try
            {
                br = new BinaryReader(File.Open(gp4Diag.FileName, FileMode.Open));
            }
            catch (Exception)
            {
                throw;
            }

            if(mp3Diag.ShowDialog() == DialogResult.Cancel)
                return;
            try
            {
                wplayer = new WindowsMediaPlayer {URL = mp3Diag.FileName};
                wplayer.controls.stop();
            }
            catch (Exception err)
            {
                MessageBox.Show(err.ToString());
                throw;
            }

            // If a song is playing, cancel it when someone wants to open a new song
            wplayer.controls.stop();
            backgroundWorker1.CancelAsync();

            readAndShow(br);
            Text = gp4Diag.FileName.Split('\\').Last();
        }

        private void playButton_Click(object who, EventArgs e)
        {
            // Do nothing if someone hits play twice.
            if (currStatus == PLAYING || sif == null) 
                return;
            
            currStatus = PLAYING;
            sif.setStatus(PLAYING);

            // Make sure we're looking at the origin of the track when play begins
            sif.setDisplayLocation(new Rectangle(0, 0, 1, 1));

            song = sif.Song;
            sp = sif.SongPanel;
            tempo = song.Tempo;

            wholeNoteInMilliSeconds = 240000/(int) tempo.BPM;

            wplayer.controls.play();

            for (int i = 0; i < sp.NumberOfTracks; i++)
            {
                sp.getTrackPanel(i).setMusicCursor(1, 0, false);
            }

            backgroundWorker1.RunWorkerAsync();
        }
        private void stopButton_Click(object who, EventArgs e)
        {
            // Do nothing if someone hits stop twice
            if(currStatus == NOT_PLAYING || sif == null)
                return;

            currStatus = NOT_PLAYING;
            sif.setStatus(NOT_PLAYING);

            wplayer.controls.stop();

            backgroundWorker1.CancelAsync();
        }
        private void helpButton_Click(object who, EventArgs e)
        {
            if(gp4Diag.FileName.Length != 0)
            {
                // Figure out the name of the song and replace spaces with +s so we can search
                string search = gp4Diag.FileName.Split('\\').Last().Replace(' ', '+');

                // Remove the file extension (and the period preceeding the extension)
                search = search.Substring(0, search.Length - 4) + " lesson";

                // Remove the version number (like v2, v3, v25, etc)
                search = Regex.Replace(search, "v\\d+", "");

                // Run the search at youtube
                System.Diagnostics.Process.Start("http://www.youtube.com/results?search_query=" + search + "&aq=f");
            }
            // Since no song has been opened yet, just open the youtube homepage
            else
            {
                System.Diagnostics.Process.Start("http://www.youtube.com/");
            }
        }

        private void backgroundWorker1_ProgressChanged(object sender, ProgressChangedEventArgs e)
        {
            bool multiTrack = sp.MultiTrackView;
            Rectangle rect = new Rectangle();

            for (int i = 0; i < sp.NumberOfTracks; i++)
            {
                bool scrollTrack = ((!multiTrack) || (i == 0));
                rect = sp.getTrackPanel(i).setMusicCursor(e.ProgressPercentage, e.ProgressPercentage - 1, scrollTrack);
            }

            sif.setDisplayLocation(rect);
        }

        private void backgroundWorker1_DoWork(object sender, DoWorkEventArgs e)
        {
            if (gp4Diag.FileName.Split('\\').Last().Equals("Creedence Clearwater Revival - Have You Ever Seen The Rain v2.gp4"))
            {
                wholeNoteInMilliSeconds += 40;
                Thread.Sleep(1000);
            }
            else if (gp4Diag.FileName.Split('\\').Last().Equals("Strokes (The) - 12 51 v2.gp4"))
            {
                wholeNoteInMilliSeconds -= 20;
                Thread.Sleep(2300);
            }
            else if (gp4Diag.FileName.Split('\\').Last().Equals("Strokes (The) - Juicebox v1.gp4"))
                Thread.Sleep(2300);

            start = DateTime.Now;
            int measureNum = 2;
            while (true)
            {
                if (backgroundWorker1.CancellationPending)
                {
                    e.Cancel = true;
                    break;
                }
                
                TimeSpan span = DateTime.Now.Subtract(start);
                if (span.TotalMilliseconds > wholeNoteInMilliSeconds && measureNum < sp.NumberOfMeasures)
                {
                    backgroundWorker1.ReportProgress(measureNum);
                    measureNum++;
                    start = DateTime.Now;
                }
            }
        }
        #endregion
    }
}
