/* Sprite.cpp
 * Created on: Apr 1, 2009
 * Author: Ian Roskam
 */
#include <nds.h>

#include "Sprite.h"
#include <string> //only used for debug print methods

/**
 * Default constructor is called when Character class is created.  All pointers
 * are set to NULL.  The pointers are set to real values when init is called.
 */
Sprite::Sprite()
{
	this->SPRITE_SIZE = 64*64;
	this->MAX_FRAMES = 10;
	for(int i=0; i<10; i++)
		this->spriteGfxMem[i] = NULL;
	this->spriteSheet = NULL;
	this->spritePalette = NULL;
	this->animFrame = 0;
	this->frameStrip = 0;
	this->framesPerAnimation = 10;
	this->cycleFrames = 0;
	this->divider = 0;
	this->loop = true;
}

/**
 * The constructor is called when Character is instantiated before
 * initilization.  Using init allows different Sprites to be created for player
 * one and player two.
 *
 * @param spriteSheet a pointer to the sprite sheet array
 * @param spriteSheetSize the length in pixels of spriteSheet
 * @param spritePalette a pointer to the sprite palette array
 * @param spritePaletteSize the length in pixels of spritePalette
 * @param oamID OAM ID number
 * @param posX the initial 'x' position of the sprite in screen coordinates
 * @param posY the initial 'y' position of the sprite in screen coordinates
 * @param direction which way the sprite is facing
 */
void Sprite::init(const short unsigned int* spriteSheet,
		unsigned int spriteSheetSize, const short unsigned int* spritePalette,
		unsigned int spritePaletteSize, int oamID, int posX, int posY,
		int direction)
{
	this->spriteSheet = spriteSheet;
	this->spriteSheetSize = spriteSheetSize;
	this->spritePalette = spritePalette;
	this->spritePaletteSize = spritePaletteSize;
	this->posX = posX;
	this->posY = posY;
	this->direction = direction;
	this->oamID = oamID;
	if(direction==RIGHT)
		this->vflip = true;
	else
		this->vflip = false;
	initSprites();
}
/**
 * Initilize sprite memory to handle a 1D mapping of bitmap sprites.
 * Also load the sprite palette.
 */
void Sprite::initSprites()
{
	oamInit(&oamMain, SpriteMapping_1D_256,	false);
	//iprintf("\nframes %d",framesPerAnimation);
	for(int i=0; i<framesPerAnimation; i++)
	{
		this->spriteGfxMem[i] = oamAllocateGfx(&oamMain, SpriteSize_64x64,
					SpriteColorFormat_256Color);
		//iprintf("\n spriteMem %#x",spriteGfxMem[i]);
	}
	//iprintf("\npalette %#x",&SPRITE_PALETTE[oamID*512]);
	dmaCopy(spritePalette,&SPRITE_PALETTE[oamID*spritePaletteSize],
			spritePaletteSize);
}
/**
 * Cycle through currently loaded frame strip.
 */
void Sprite::nextFrame()
{
	if(divider%4==0)
	{
		animFrame++;
		if(animFrame>(framesPerAnimation-1))
		{
			//if(loop)
				animFrame=cycleFrames;
			//else
				//animDone = true;
		}
	}
	divider++;
	if(divider>60)divider=0;
}
/**
 * Frames per animation varies between animations.  This method takes care of
 * setting framesPerAnimation before animation begins.
 * @param animationStrip - the current animation strip
 */
void Sprite::setFramesPerAnimation(strips animationStrip)
{
	switch(animationStrip)
	{
		case STANCE:
			loop = true;
			framesPerAnimation = 10;
			cycleFrames = 0;
			break;
		case FOWARD:
			loop = true;
			framesPerAnimation = 10;
			cycleFrames = 0;
			break;
		case REVERSE:
			loop = true;
			framesPerAnimation = 10;
			cycleFrames = 0;
			break;
		case JUMP:
			loop = false;
			framesPerAnimation = 10;
			cycleFrames = 0;
			break;
		case DUCK:
			loop = true;
			framesPerAnimation = 10;
			cycleFrames = 9;  //loop last frame of animation
			break;
		case BLOCK:
			loop = true;
			framesPerAnimation = 4;
			cycleFrames = 3; //loop last frame of animation
			break;
		case LOWPUNCH:
			loop = false;
			framesPerAnimation = 10;
			cycleFrames = 0;
			break;
		case HIGHPUNCH:
			loop = false;
			framesPerAnimation = 10;
			cycleFrames = 0;
			break;
		case LOWKICK:
			loop = false;
			framesPerAnimation = 7;
			cycleFrames = 0;
			break;
		case HIGHKICK:
			loop = false;
			framesPerAnimation = 10;
			cycleFrames = 0;
			break;
		default:
			return; //shouldnt get here
	}
}
/**
 * Load the current animation strip into VRAM
 * @param animationStrip - the index of the animation strip
 */
void Sprite::loadStrip(strips animationStrip)
{
	animFrame = 0;
	animDone = false;
	setFramesPerAnimation(animationStrip);
	const short unsigned int* offset = spriteSheet+(animationStrip*
							MAX_FRAMES*(SPRITE_SIZE/2));
	for(int i=0; i<MAX_FRAMES; i++)
	{
		//iprintf("\noffset %#x",offset);
		dmaCopy(offset,spriteGfxMem[i],SPRITE_SIZE);
		offset+=SPRITE_SIZE/2;
	}
}
/**
 * Load the current animation frame into OAM.
 */
void Sprite::animateSprite()
{

	oamSet(&oamMain,oamID,posX,posY,OBJPRIORITY_0,0,SpriteSize_64x64,
			SpriteColorFormat_256Color,spriteGfxMem[animFrame],0,false,
			false,false,false,false);
}

/**
 * Free up the memory used for sprites and delete pointers.
 */
Sprite::~Sprite() //Destructor
{
	oamClear(&oamMain, 0,0);
	for(int i=0; i<MAX_FRAMES; i++)
		oamFreeGfx(&oamMain, spriteGfxMem[i]); //clear VRAM of sprites
}
