/* Genuine Channels product.
 * 
 * Copyright (c) 2002-2007 Dmitry Belikov. All rights reserved.
 * 
 * This source code comes under and must be used and distributed according to the Genuine Channels license agreement.
 */

using System;
using System.Collections;
using System.Diagnostics;
using System.Net;
using System.Net.Sockets;
using System.Runtime.Remoting;
using System.Runtime.Remoting.Messaging;
using System.Runtime.Remoting.Channels;
using System.Threading;

using Shared.Channels;
using Shared.Channels.Connection;
using Shared.Channels.DotNetRemotingLayer;
using Shared.Channels.Logbook;
using Shared.Channels.Messaging;
using Shared.Channels.Security;
using Shared.Channels.TransportContext;
using Shared.Channels.Utilities;

namespace Shared.Channels.GenuineXHttp
{
	/// <summary>
	/// GenuineXHttpChannel implements a .NET Remoting channel wrapper over GXHTTP Transport Context.
	/// </summary>
	public class GenuineXHttpChannel : BasicChannelWithSecurity
	{
		/// <summary>
		/// Constructs an instance of the GenuineTcpChannel class.
		/// </summary>
		/// <param name="properties">An IDictionary of the channel properties which hold the configuration information for the current channel.</param>
		/// <param name="iClientChannelSinkProvider">The IClientChannelSinkProvider that creates the client channel sinks for the underlying channel through which remoting messages flow through.</param>
		/// <param name="iServerChannelSinkProvider">The IServerChannelSinkProvider that creates server channel sinks for the underlying channel through which remoting messages flow through.</param>
		public GenuineXHttpChannel(IDictionary properties, IClientChannelSinkProvider iClientChannelSinkProvider, IServerChannelSinkProvider iServerChannelSinkProvider)
			: base(iClientChannelSinkProvider, iServerChannelSinkProvider)
		{
			this.ITransportContext = TransportContextServices.CreateDefaultXHttpContext(properties, this);
			this.InitializeInstance(properties);

			if (this._channelName == null)
				this._channelName = "ghttp";
			if (this._urlPrefix == null)
				this._urlPrefix = "ghttp";
			this._possibleChannelPrefixes = new string[] { this.UrlPrefix, this.UriPrefix };

			// Start listening
			this.StartListening(null);
		}

		#region -- IChannelReceiver ----------------------------------------------------------------

		/// <summary>
		/// Returns an array of all the URLs for a URI.
		/// </summary>
		/// <param name="objectUri">The URI for which URLs are required.</param>
		/// <returns>An array of the URLs.</returns>
		public override string[] GetUrlsForUri(string objectUri)
		{
			return new string[1] { this.UriPrefix + @"://" + this.ITransportContext.HostIdentifier + "/" + objectUri };
		}

		/// <summary>
		/// Instructs the current channel to start listening for requests.
		/// </summary>
		/// <param name="data">Optional initialization information.</param>
		public override void StartListening(object data)
		{
			base.StartListening(data);

			if (this._localPort >= 0)
				this.ITransportContext.ConnectionManager.StartListening(this.ListeningEntry);
		}

		/// <summary>
		/// Instructs the current channel to stop listening for requests.
		/// </summary>
		/// <param name="data">Optional state information for the channel.</param>
		public override void StopListening(object data)
		{
			if (this._localPort >= 0)
			{
				this.ITransportContext.ConnectionManager.StopListening(this.ListeningEntry);
				base.StopListening(data);
			}
		}

		#endregion

		#region -- Channel Prefixes ----------------------------------------------------------------

		/// <summary>
		/// Gets the initial part of the URI prefix.
		/// </summary>
		public override string UriPrefix
		{ 
			get
			{
				return "_ghttp";
			}
		}

		#endregion

		#region -- Settings ------------------------------------------------------------------------

		/// <summary>
		/// The port number to bind to.
		/// </summary>
		private int _localPort = -1;

		/// <summary>
		/// Interface.
		/// </summary>
		private string _localInterface = "0.0.0.0";

		/// <summary>
		/// The listening entry the channel binds listening socket to.
		/// </summary>
		public string ListeningEntry
		{
			get
			{
				return this.UrlPrefix + @"://" + this._localInterface + ":" + this._localPort;
			}
		}

		/// <summary>
		/// Reads settings.
		/// </summary>
		/// <param name="properties">Settings container to read from.</param>
		protected override void ReadSettings(IDictionary properties)
		{
			base.ReadSettings(properties);

			// retrieve settings
			foreach (DictionaryEntry entry in properties)
			{
				if (string.Compare(entry.Key.ToString(), "interface", true) == 0)
					this._localInterface = entry.Value.ToString();

				if (string.Compare(entry.Key.ToString(), "port", true) == 0)
					this._localPort = GenuineUtility.SafeConvertToInt32(entry.Value, this._localPort);
			}
		}

		#endregion

	}
}
