// Much of this code is derived from poker.eval (look for it on sourceforge.net).
// This library is covered by the GPL Gnu license. See http://www.gnu.org/licenses/old-licenses/gpl-2.0.html 
// for more information on this license.

// This code is a very fast, native C# Texas Holdem mask evaluator (containing no interop or unsafe code). 
// This code can enumarate 35 million 5 card hands per second and 29 million 7 card hands per second on my desktop machine.
// That's not nearly as fast as the heavily macro-ed poker.eval C library. However, this implementation is
// in roughly the same ballpark for speed and is quite usable in C#.

// The speed ups are mostly table driven. That means that there are several very large tables included in this file. 
// The code is divided up into several files they are:
//      HandEvaluator.cs - base mask evaluator
//      HandIterator.cs - methods that support IEnumerable and methods that validate the mask evaluator
//      HandAnalysis.cs - methods to aid in analysis of Texas Holdem Hands.
//      PocketHands.cs - a class to manipulate pocket hands.
//      PocketQueryParser - a parser used to interprete pocket mask query language statements.
//                          The runtime portion of the query parser is copyrighted by Malcolm Crowe (but is freely distributable)

// Written (ported) by Keith Rule - Sept 2005, updated May 2007


using System;
using System.Text;
using System.Diagnostics;

namespace Private.Logic.Poker.Probability
{
    #region Hand Class
    /// <example>
    /// <code>
    /// using System;
    /// using System.Collections.Generic;
    /// using System.Text;
    /// using Private.Logic.Poker.Probability;
    /// 
    /// // Simple example of using the Holdem.Hand class
    /// class Program
    /// {
    ///     static void Main(string[] args)
    ///     {
    ///         // initialize board
    ///         string board = "2d kh qh 3h qc";
    ///         // Create a mask with AKs plus board
    ///         Hand h1 = new Hand("ad kd", board);
    ///         // Create a mask with 23 unsuited plus board
    ///         Hand h2 = new Hand("2h 3d", board);
    /// 
    ///         // Find stronger mask and print results
    ///         if (h1 > h2)
    ///         {
    ///             Console.WriteLine("{0} greater than \n\t{1}", h1.Description, h2.Description);
    ///         }
    ///         else
    ///         {
    ///             Console.WriteLine("{0} less than or equal \n\t{1}", h1.Description, h2.Description);
    ///         }
    ///     }
    /// }
    /// </code>
    /// </example>
    /// 
    /// <summary>
    /// Represents a Texas Holdem Hand
    /// </summary>
    public partial class Hand : IComparable
    {
        #region HandTypes Enumeration
        /// <summary>
        /// Possible types of hands in a texas holdem game.
        /// </summary>
        public enum HandTypes
        {
            /// <summary>
            /// Only a high card
            /// </summary>
            HighCard = 0,
            /// <summary>
            /// One Pair
            /// </summary>
            TwoOfAKind = 1,
            /// <summary>
            /// Two Pair
            /// </summary>
            TwoPair = 2,
            /// <summary>
            /// Three of a kind (Trips)
            /// </summary>
            ThreeOfAKind = 3,
            /// <summary>
            /// Straight
            /// </summary>
            Straight = 4,
            /// <summary>
            /// Flush
            /// </summary>
            Flush = 5,
            /// <summary>
            /// FullHouse
            /// </summary>
            FullHouse = 6,
            /// <summary>
            /// Four of a kind
            /// </summary>
            FourOfAKind = 7,
            /// <summary>
            /// Straight Flush
            /// </summary>
            StraightFlush = 8
        }
        #endregion

        #region Consts
        /// <summary>
        /// Represents the suit - Hearts
        /// </summary>
        public static readonly int Hearts = 2;
        /// <summary>
        /// Represents the suit - Diamonds
        /// </summary>
        public static readonly int Diamonds = 1;
        /// <summary>
        /// Represents the suit - Clubs
        /// </summary>
        public static readonly int Clubs = 0;
        /// <summary>
        /// Represents the suit - Spades
        /// </summary>
        public static readonly int Spades = 3;
        /// <summary>
        /// Rank of a card with a value of two.
        /// </summary>
        public static readonly int Rank2 = 0;
        /// <summary>
        /// Rank of a card with a value of three.
        /// </summary>
        public static readonly int Rank3 = 1;
        /// <summary>
        /// Rank of a card with a value of four.
        /// </summary>
        public static readonly int Rank4 = 2;
        /// <summary>
        /// Rank of a card with a value of five.
        /// </summary>
        public static readonly int Rank5 = 3;
        /// <summary>
        /// Rank of a card with a value of six.
        /// </summary>
        public static readonly int Rank6 = 4;
        /// <summary>
        /// Rank of a card with a value of seven.
        /// </summary>
        public static readonly int Rank7 = 5;
        /// <summary>
        /// Rank of a card with a value of eight.
        /// </summary>
        public static readonly int Rank8 = 6;
        /// <summary>
        /// Rank of a card with a value of nine.
        /// </summary>
        public static readonly int Rank9 = 7;
        /// <summary>
        /// Rank of a card with a value of ten.
        /// </summary>
        public static readonly int RankTen = 8;
        /// <summary>
        /// Rank of a card showing a Jack.
        /// </summary>
        public static readonly int RankJack = 9;
        /// <summary>
        /// Rank of a card showing a Queen.
        /// </summary>
        public static readonly int RankQueen = 10;
        /// <summary>
        /// Rank of a card showing a King.
        /// </summary>
        public static readonly int RankKing = 11;
        /// <summary>
        /// Rank of a card showing an Ace.
        /// </summary>
        public static readonly int RankAce = 12;
        ///<exclude/>
        public static readonly int CardJoker = 52;
        /// <summary>
        /// The total number of cards in a deck
        /// </summary>
        public static readonly int NumberOfCards = 52;
        /// <exclude/>
        public static readonly int NCardsWJoker = 53;
        /// <exclude/>
        private static readonly int HANDTYPE_SHIFT = 24;
        /// <exclude/>
        private static readonly int TOP_CARD_SHIFT = 16;
        /// <exclude/>
        private static readonly System.UInt32 TOP_CARD_MASK = 0x000F0000;
        /// <exclude/>
        private static readonly int SECOND_CARD_SHIFT = 12;
        /// <exclude/>
        private static readonly System.UInt32 SECOND_CARD_MASK = 0x0000F000;
        /// <exclude/>
        private static readonly int THIRD_CARD_SHIFT = 8;
        /// <exclude/>
        private static readonly int FOURTH_CARD_SHIFT = 4;
        /// <exclude/>
        private static readonly int FIFTH_CARD_SHIFT = 0;
        /// <exclude/>
        private static readonly System.UInt32 FIFTH_CARD_MASK = 0x0000000F;
        /// <exclude/>
        private static readonly int CARD_WIDTH = 4;
        /// <exclude/>
        private static readonly System.UInt32 CARD_MASK = 0x0F;
        #endregion

        #region Private Fields
        /// <summary>
        /// Hand mask for the current card set
        /// </summary>
        private System.UInt64 handmask;
        /// <summary>
        /// Contains string representing the pocket cards
        /// </summary>
        private string pocket;
        /// <summary>
        /// Contains a string representing the board (common cards)
        /// </summary>
        private string board;
        /// <summary>
        /// The value of the current had. This value allows hands to be 
        /// compared using a normal arithmitic compare function.
        /// </summary>
        private System.UInt32 handval;
        #endregion

        #region Constructor
        /// <summary>
        /// Default constructor
        /// </summary>
        public Hand()
        {
            pocket = board = "";
        }

        /// <example>
        /// <code>
        /// using System;
        /// using System.Collections.Generic;
        /// using System.Text;
        /// using Private.Logic.Poker.Probability;
        /// 
        /// // Simple example of using the Holdem.Hand class
        /// class Program
        /// {
        ///     static void Main(string[] args)
        ///     {
        ///         // initialize board
        ///         string board = "2d kh qh 3h qc";
        ///         // Create a mask with AKs plus board
        ///         Hand h1 = new Hand("ad kd", board);
        ///         // Create a mask with 23 unsuited plus board
        ///         Hand h2 = new Hand("2h 3d", board);
        /// 
        ///         // Find stronger mask and print results
        ///         if (h1 > h2)
        ///         {
        ///             Console.WriteLine("{0} greater than \n\t{1}", h1.Description, h2.Description);
        ///         }
        ///         else
        ///         {
        ///             Console.WriteLine("{0} less than or equal \n\t{1}", h1.Description, h2.Description);
        ///         }
        ///     }
        /// }
        /// </code>
        /// </example>
        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="pocket">Pocket Cards</param>
        /// <param name="board">Board</param>
        public Hand(string pocket, string board)
        {
#if DEBUG
            if (pocket == null) throw new ArgumentNullException("pocket");
            if (board == null) throw new ArgumentNullException("board");
#endif
            PocketCards = pocket;
            Board = board;
        }
        #endregion

        #region Card Parser/Formatter
        /// <summary>
        /// This function takes a string representing a full or partial holdem mask 
        /// and validates that the text represents valid cards and that no card is
        /// duplicated.
        /// </summary>
        /// <param name="hand">mask to validate</param>
        /// <returns>true of a valid mask, false otherwise</returns>
        public static bool ValidateHand(string hand)
        {
#if DEBUG
            if (hand == null) return false;
#endif

            int index = 0;
            ulong handmask = 0UL;
            int cards = 0;
            int card = 0;

            try
            {
                for (card = NextCard(hand, ref index); card >= 0; card = NextCard(hand, ref index))
                {
                    if ((handmask & (1UL << card)) != 0)
                        return false;
                    handmask |= (1UL << card);
                    cards++;
                }

                return card == -1 && cards > 0 && index >= hand.Length;
            }
            catch
            {
                return false;
            }
        }

        /// <summary>
        /// This function takes a string representing pocket cards and a board and then 
        /// validates that the text represents a valid mask.
        /// </summary>
        /// <param name="pocket">Pocket cards as a string</param>
        /// <param name="board">Board cards as a string</param>
        /// <returns></returns>
        public static bool ValidateHand(string pocket, string board)
        {
#if DEBUG
            if (pocket == null || pocket.Trim().Length == 0) throw new ArgumentNullException("pocket");
            if (board == null || board.Trim().Length == 0) throw new ArgumentNullException("board");
#endif
            return ValidateHand(pocket + " " + board);
        }

        /// <example>
        /// <code>
        ///  // Takes an ascii seven card mask and prints out a 
        ///  // value description. For example: "ad kd 2d kh qh 3h qc" would
        ///  // output "Two pair, King's and Queen's with a Ace for a kicker"
        ///  static void PrintDescription(string mask)
        ///  {
        ///      // Parse mask into a mask mask
        ///      ulong handmask = Hand.ParseHand(mask);
        ///  
        ///      // Convert mask mask into a compariable mask value.
        ///      uint handval = Hand.Evaluate(handmask, 7);
        ///  
        ///      // Print a description of the mask.
        ///      Console.WriteLine("Hand: {0}", Hand.DescriptionFromHandValue(handval));
        ///  }
        /// </code>
        /// </example>
        /// <summary>
        /// Parses an string description of a mask and returns a mask mask.
        /// </summary>
        /// <param name="mask">string descripton of a mask</param>
        /// <returns>a mask mask representing the mask</returns>
        public static ulong ParseHand(string mask)
        {
            int cards = 0;
            return ParseHand(mask, ref cards);
        }

        /// <summary>
        /// Parse Hand.
        /// </summary>
        /// <param name="hand">Hand string</param>
        /// <param name="cards">Cards found</param>
        /// <returns></returns>
        public static ulong ParseHand(string hand, ref int cards)
        {
            int index = 0;
            ulong handmask = 0UL;

#if DEBUG
            if (hand == null) throw new ArgumentNullException("hand");
#endif

            // A null mask is okay
            if (hand.Trim().Length == 0)
            {
                cards = 0;
                return 0UL;
            }

#if DEBUG
            // Hand contains either invalid strings or duplicate entries
            if (!Hand.ValidateHand(hand)) throw new ArgumentException("Bad hand definition");
#endif
            // Parse the mask
            cards = 0;
            for (int card = NextCard(hand, ref index); card >= 0; card = NextCard(hand, ref index))
            {
                handmask |= (1UL << card);
                cards++;
            }
            return handmask;
        }

        /// <summary>
        /// This static method parses the passed pocket cards and board and produces
        /// a card mask.
        /// </summary>
        /// <param name="pocket">ASCII string representing pocket cards</param>
        /// <param name="board">ASCII string representing board</param>
        /// <param name="cards">Number of cards represented in mask</param>
        /// <returns></returns>
        public static ulong ParseHand(string pocket, string board, ref int cards)
        {
            return ParseHand(pocket + " " + board, ref cards);
        }

        /// <summary>
        /// Reads an string definition of a card and returns the Card value.
        /// </summary>
        /// <param name="card">card string</param>
        /// <returns></returns>
        public static int ParseCard(string card)
        {
#if DEBUG
            if (card == null) 
                throw new ArgumentNullException("card");
#endif
            int index = 0;
            return NextCard(card, ref index);
        }

        /// <summary>
        /// Parses Card strings (internal)
        /// </summary>
        /// <param name="cards">string containing mask definition</param>
        /// <param name="index">iterator into card string</param>
        /// <returns></returns>
        private static int NextCard(string cards, ref int index)
        {
            int rank = 0, suit = 0;
#if DEBUG
            if (cards == null) throw new ArgumentNullException("cards");
#endif

            // Remove whitespace
            while (index < cards.Length && cards[index] == ' ')
                index++;

            if (index >= cards.Length)
                return -1;

            // Parse cards
            if (index < cards.Length)
            {
                switch (cards[index++])
                {
                    case '1':
                        try
                        {
                            if (cards[index] == '0')
                            {
                                index++;
                                rank = RankTen;
                            }
                            else
                            {
                                return -1;
                            }
                        }
                        catch
                        {
                            throw new ArgumentException("Bad hand string");
                        }
                        break;
                    case '2':
                        rank = Rank2;
                        break;
                    case '3':
                        rank = Rank3;
                        break;
                    case '4':
                        rank = Rank4;
                        break;
                    case '5':
                        rank = Rank5;
                        break;
                    case '6':
                        rank = Rank6;
                        break;
                    case '7':
                        rank = Rank7;
                        break;
                    case '8':
                        rank = Rank8;
                        break;
                    case '9':
                        rank = Rank9;
                        break;
                    case 'T':
                    case 't':
                        rank = RankTen;
                        break;
                    case 'J':
                    case 'j':
                        rank = RankJack;
                        break;
                    case 'Q':
                    case 'q':
                        rank = RankQueen;
                        break;
                    case 'K':
                    case 'k':
                        rank = RankKing;
                        break;
                    case 'A':
                    case 'a':
                        rank = RankAce;
                        break;
                    default:
                        return -2;
                }
            }
            else
            {
                return -2;
            }

            if (index < cards.Length)
            {
                switch (cards[index++])
                {
                    case 'H':
                    case 'h':
                        suit = Hearts;
                        break;
                    case 'D':
                    case 'd':
                        suit = Diamonds;
                        break;
                    case 'C':
                    case 'c':
                        suit = Clubs;
                        break;
                    case 'S':
                    case 's':
                        suit = Spades;
                        break;
                    default:
                        return -2;
                }
            }
            else
            {
                return -2;
            }

            return rank + (suit * 13);
        }

        /// <summary>
        /// Given a card value, returns it's rank
        /// </summary>
        /// <param name="card">card value</param>
        /// <returns>returns rank</returns>
        public static int CardRank(int card)
        {
#if DEBUG
            // Legal values are 0 - 52.
            if (card < 0 || card > 52)
                throw new ArgumentOutOfRangeException("card");
#endif
            return card % 13;
        }

        /// <summary>
        /// Given a card value, returns it's suit
        /// </summary>
        /// <param name="card">Card value</param>
        /// <returns>suit</returns>
        public static int CardSuit(int card)
        {
#if DEBUG
            // Legal values are 0 - 52.
            if (card < 0 || card > 52)
                throw new ArgumentOutOfRangeException("card");
#endif
            return card / 13;
        }

        /// <exclude/>
        public static string DescriptionFromHandValueInternal(uint handValue)
        {
            StringBuilder b = new StringBuilder();

            switch ((HandTypes)HandType(handValue))
            {
                case HandTypes.HighCard:
                    b.Append("High card: ");
                    b.Append(ranktbl[TopCard(handValue)]);
                    return b.ToString();
                case HandTypes.TwoOfAKind:
                    b.Append("One pair, ");
                    b.Append(ranktbl[TopCard(handValue)]);
                    return b.ToString();
                case HandTypes.TwoPair:
                    b.Append("Two pair, ");
                    b.Append(ranktbl[TopCard(handValue)]);
                    b.Append("'s and ");
                    b.Append(ranktbl[SECOND_CARD(handValue)]);
                    b.Append("'s with a ");
                    b.Append(ranktbl[THIRD_CARD(handValue)]);
                    b.Append(" for a kicker");
                    return b.ToString();
                case HandTypes.ThreeOfAKind:
                    b.Append("Three of a kind, ");
                    b.Append(ranktbl[TopCard(handValue)]);
                    b.Append("'s");
                    return b.ToString();
                case HandTypes.Straight:
                    b.Append("A straight, ");
                    b.Append(ranktbl[TopCard(handValue)]);
                    b.Append(" high");
                    return b.ToString();
                case HandTypes.Flush:
                    b.Append("A flush");
                    return b.ToString();
                case HandTypes.FullHouse:
                    b.Append("A fullhouse, ");
                    b.Append(ranktbl[TopCard(handValue)]);
                    b.Append("'s and ");
                    b.Append(ranktbl[SECOND_CARD(handValue)]);
                    b.Append("'s");
                    return b.ToString();
                case HandTypes.FourOfAKind:
                    b.Append("Four of a kind, ");
                    b.Append(ranktbl[TopCard(handValue)]);
                    b.Append("'s");
                    return b.ToString();
                case HandTypes.StraightFlush:
                    b.Append("A straight flush");
                    return b.ToString();
            }
            Debug.Assert(false); // Should never get here
            return "";
        }

       /// <summary>
       /// Evaluates a hand and returns a descriptive string.
       /// </summary>
       /// <param name="cards">Card mask</param>
       /// <returns></returns>
        public static string DescriptionFromMask(ulong cards)
        {
            int numberOfCards = BitCount(cards);

#if DEBUG
            // This functions supports 1-7 cards
            if (numberOfCards < 1 || numberOfCards > 7)
                throw new ArgumentOutOfRangeException("numberOfCards");
#endif
            // Seperate out by suit
            uint sc = (uint)((cards >> (CLUB_OFFSET)) & 0x1fffUL);
            uint sd = (uint)((cards >> (DIAMOND_OFFSET)) & 0x1fffUL);
            uint sh = (uint)((cards >> (HEART_OFFSET)) & 0x1fffUL);
            uint ss = (uint)((cards >> (SPADE_OFFSET)) & 0x1fffUL);

            uint handvalue = Evaluate(cards, numberOfCards);

            switch ((HandTypes)HandType(handvalue))
            {
                case HandTypes.HighCard:
                case HandTypes.TwoOfAKind:
                case HandTypes.TwoPair:
                case HandTypes.ThreeOfAKind:
                case HandTypes.Straight:
                case HandTypes.FullHouse:
                case HandTypes.FourOfAKind:
                    return DescriptionFromHandValueInternal(handvalue);
                case HandTypes.Flush:
                    if (nBitsTable[ss] >= 5)
                    {
                        return "Flush (Spades) with " + ranktbl[TopCard(handvalue)] + " high";
                    }
                    else if (nBitsTable[sc] >= 5)
                    {
                        return "Flush (Clubs) with " + ranktbl[TopCard(handvalue)] + " high";
                    }
                    else if (nBitsTable[sd] >= 5)
                    {
                        return "Flush (Diamonds) with " + ranktbl[TopCard(handvalue)] + " high";
                    }
                    else if (nBitsTable[sh] >= 5)
                    {
                        return "Flush (Hearts) with " + ranktbl[TopCard(handvalue)] + " high";
                    }
                    break;
                case HandTypes.StraightFlush:
                    if (nBitsTable[ss] >= 5)
                    {
                        return "Straight Flush (Spades) with " + ranktbl[TopCard(handvalue)] + " high";
                    }
                    else if (nBitsTable[sc] >= 5)
                    {
                        return "Straight (Clubs) with " + ranktbl[TopCard(handvalue)] + " high";
                    }
                    else if (nBitsTable[sd] >= 5)
                    {
                        return "Straight (Diamonds) with " + ranktbl[TopCard(handvalue)] + " high";
                    }
                    else if (nBitsTable[sh] >= 5)
                    {
                        return "Straight  (Hearts) with " + ranktbl[TopCard(handvalue)] + " high";
                    }
                    break;
            }
            Debug.Assert(false); // Should never get here
            return "";
        }

        /// <summary>
        /// Takes an string describing a mask and returns the description.
        /// </summary>
        /// <param name="mask">the string describing the mask</param>
        /// <returns>Returns a description string</returns>
        /// <example>
        /// <code>
        /// // Print a description of the mask.
        /// Console.WriteLine("Hand: {0}", Hand.DescriptionFromHand("ad kd 2d kh qh 3h qc"));
        /// </code>
        /// </example>
        public static string DescriptionFromHand(string mask)
        {
            int cards = 0;
#if DEBUG
            // Must not be null string
            if (mask == null)
                throw new ArgumentNullException("mask");
#endif
            return DescriptionFromMask(ParseHand(mask, ref cards));
        }

        /// <summary>
        /// Updates handmask and handval, called when card strings change
        /// </summary>
        private void UpdateHandMask()
        {
            int cards = 0;
            handmask = Hand.ParseHand(PocketCards, Board, ref cards);
            handval = Hand.Evaluate(handmask, cards);
        }

        #endregion

        #region Properties

        /// <summary>
        /// Returns the string representing the mask.
        /// </summary>
        /// <returns></returns>
        public override string ToString()
        {
            return PocketCards + " " + Board; ;
        }

        /// <summary>
        /// Returns mask mask value
        /// </summary>
        public System.UInt64 MaskValue
        {
            get { return handmask; }
        }

        /// <summary>
        /// Represents the Mask of the Pocket cards for this instance
        /// of Hand
        /// </summary>
        public System.UInt64 PocketMask
        {
            set
            {
                PocketCards = MaskToString(value);
            }

            get
            {
                return ParseHand(PocketCards);
            }
        }

        /// <summary>
        /// Represents the Mask of the Board cards for this instance
        /// of Hand
        /// </summary>
        public System.UInt64 BoardMask
        {
            set
            {
                Board = MaskToString(value);
            }

            get
            {
                return ParseHand(Board);
            }
        }

        /// <summary>
        /// Returns/Sets pocket card string
        /// </summary>
        public string PocketCards
        {
            get { return pocket; }
            set
            {
#if DEBUG
                if (value == null)
                    throw new ArgumentNullException("value");

                if (value.Trim().Length <= 0 || !Hand.ValidateHand(value))
                    throw new ArgumentNullException("value");
#endif

                pocket = value.Trim();
                UpdateHandMask();
            }
        }

        /// <summary>
        /// Returns/Sets board card string
        /// </summary>
        public string Board
        {
            get { return board; }
            set
            {
#if DEBUG
                if (value == null || value.Trim().Length <= 0 || !Hand.ValidateHand(value))
                    throw new ArgumentNullException("value");
#endif
                board = value;
                UpdateHandMask();
            }
        }

        /// <summary>
        /// Returns/Sets the mask value. This value may be used
        /// to compare one mask to another using standard numeric 
        /// compares.
        /// </summary>
        public uint HandValue
        {
            get { return handval; }
        }

        /// <summary>
        /// Returns a textual description of the current mask
        /// </summary>
        public string Description
        {
            get
            {
                return DescriptionFromMask(MaskValue);
            }
        }

        /// <summary>
        /// Returns the current mask type.
        /// </summary>
        public HandTypes HandTypeValue
        {
            get
            {
                return (HandTypes)HandType(HandValue);
            }
        }
        #endregion

        #region HandValue Functions

        /// <summary>
        /// This is a fast way to look up the index mask. 
        /// </summary>
        /// <param name="index">index of mask</param>
        /// <returns>mask</returns>
        public static ulong Mask(int index)
        {
            return CardMasksTable[index];
        }

        /// <exclude/>
        public static uint HandType(uint handValue)
        {
            return (handValue >> HANDTYPE_SHIFT);
        }

        // FXCop complained about this unused private method so I
        // commented it out for now.
        ///// <exclude/>
        //private static uint Cards(System.UInt32 handValue)
        //{
        //    return (handValue & CARD_MASK);
        //}

        /// <exclude/>
        public static uint TopCard(System.UInt32 hv)
        {
            return ((hv >> TOP_CARD_SHIFT) & CARD_MASK);
        }

        /// <exclude/>
        private static uint SECOND_CARD(System.UInt32 hv)
        {
            return (((hv) >> SECOND_CARD_SHIFT) & CARD_MASK);
        }

        /// <exclude/>
        private static uint THIRD_CARD(System.UInt32 hv)
        {
            return (((hv) >> THIRD_CARD_SHIFT) & CARD_MASK);
        }

        /// <exclude/>
        private static uint FOURTH_CARD(System.UInt32 hv)
        {
            return (((hv) >> FOURTH_CARD_SHIFT) & CARD_MASK);
        }

        /// <exclude/>
        private static uint FIFTH_CARD(System.UInt32 hv)
        {
            return (((hv) >> FIFTH_CARD_SHIFT) & CARD_MASK);
        }

        /// <exclude/>
        private static uint HANDTYPE_VALUE(HandTypes ht)
        {
            return (((uint)ht) << HANDTYPE_SHIFT);
        }

        ///// <exclude/>
        //private static uint HANDVALUE2HANDTYPE(ulong handval)
        //{
        //    return (((uint)handval) >> HANDTYPE_SHIFT) & 0xf;
        //}

        /// <exclude/>
        private static uint TOP_CARD_VALUE(System.UInt32 c)
        {
            return ((c) << TOP_CARD_SHIFT);
        }

        /// <exclude/>
        private static uint SECOND_CARD_VALUE(System.UInt32 c)
        {
            return ((c) << SECOND_CARD_SHIFT);
        }

        /// <exclude/>
        private static uint THIRD_CARD_VALUE(System.UInt32 c)
        {
            return ((c) << THIRD_CARD_SHIFT);
        }

        // FXCop complained about these unused private methods so I 
        // commented them out until I need them.
        ///// <exclude/>
        //private static uint FOURTH_CARD_VALUE(System.UInt32 c)
        //{
        //    return ((c) << FOURTH_CARD_SHIFT);
        //}

        ///// <exclude/>
        //private static uint FIFTH_CARD_VALUE(System.UInt32 c)
        //{
        //    return ((c) << FIFTH_CARD_SHIFT);
        //}
        /// <exclude/>
        private static uint CardMask(ulong cards, int suit)
        {
            return (uint)((cards >> (13 * suit)) & 0x1fffUL);
        }
        #endregion

        #region Evaluate Hand

        /// <summary>
        /// Turns a card mask into the equivalent human readable string.
        /// </summary>
        /// <param name="mask">mask to convert</param>
        /// <returns>human readable string that is equivalent to the mask represented by the mask</returns>
        public static string MaskToString(ulong mask)
        {
            StringBuilder builder = new StringBuilder();
            int count = 0;

            foreach (string s in Cards(mask))
            {
                if (count != 0)
                {
                    builder.Append(" ");
                }
                builder.Append(s);
                count++;
            }
           
            return builder.ToString();
        }

        /// <summary>
        /// Evaluates the card mask and returns the type of mask it is. This function is
        /// faster (but provides less information) than Evaluate or Evaluate.
        /// </summary>
        /// <param name="mask">card mask</param>
        /// <returns>A HandTypes value</returns>
        /// <example>
        /// <code>
        /// public static long ValidateEnumerate5()
        /// {
        ///     int[] handtypes = { 0, 0, 0, 0, 0, 0, 0, 0, 0 };
        ///     int count = 0;
        /// 
        ///     // Iterate through all possible 5 card hands
        ///     foreach (ulong mask in Hands(5))
        ///     {
        ///         handtypes[(int)Hand.EvaluateType(mask)]++;
        ///         count++;
        ///     }
        /// 
        ///     // Validate results.
        ///     System.Diagnostics.Debug.Assert(handtypes[(int)HandTypes.HighCard] == 1302540);
        ///     System.Diagnostics.Debug.Assert(handtypes[(int)HandTypes.TwoOfAKind] == 1098240);
        ///     System.Diagnostics.Debug.Assert(handtypes[(int)HandTypes.TwoPair] == 123552);
        ///     System.Diagnostics.Debug.Assert(handtypes[(int)HandTypes.ThreeOfAKind] == 54912);
        ///     System.Diagnostics.Debug.Assert(handtypes[(int)HandTypes.Straight] == 10200);
        ///     System.Diagnostics.Debug.Assert(handtypes[(int)HandTypes.Flush] == 5108);
        ///     System.Diagnostics.Debug.Assert(handtypes[(int)HandTypes.FullHouse] == 3744);
        ///     System.Diagnostics.Debug.Assert(handtypes[(int)HandTypes.FourOfAKind] == 624);
        ///     System.Diagnostics.Debug.Assert(handtypes[(int)HandTypes.StraightFlush] == 40);
        ///     System.Diagnostics.Debug.Assert(count == 2598960);
        ///     return count;
        /// }
        /// </code>
        /// </example>
        public static HandTypes EvaluateType(ulong mask)
        {
#if DEBUG
            int cards = BitCount(mask);
            if (cards <= 0 || cards > 7) throw new ArgumentException("mask");
            return EvaluateType(mask, cards);
#else
            return EvaluateType(mask, BitCount(mask));
#endif
        }

        /// <summary>
        ///  This function is faster (but provides less information) than Evaluate or Evaluate.
        /// </summary>
        /// <param name="mask">card mask</param>
        /// <param name="cards">number of cards in mask</param>
        /// <returns>HandType enum that describes the rank of the mask</returns>
        public static HandTypes EvaluateType(ulong mask, int cards)
        {
            HandTypes is_st_or_fl = HandTypes.HighCard;

            uint ss = (uint)((mask >> (SPADE_OFFSET)) & 0x1fffUL);
            uint sc = (uint)((mask >> (CLUB_OFFSET)) & 0x1fffUL);
            uint sd = (uint)((mask >> (DIAMOND_OFFSET)) & 0x1fffUL);
            uint sh = (uint)((mask >> (HEART_OFFSET)) & 0x1fffUL);

            uint ranks = sc | sd | sh | ss;
            uint rankinfo = nBitsAndStrTable[ranks];
            uint n_dups = (uint)(cards - (rankinfo >> 2));

            if ((rankinfo & 0x01U) != 0U)
            {
                if ((rankinfo & 0x02) != 0U)
                    is_st_or_fl = HandTypes.Straight;

                uint t = (uint)(nBitsAndStrTable[ss] | nBitsAndStrTable[sc] | nBitsAndStrTable[sd] | nBitsAndStrTable[sh]);

                if ((t & 0x01u) != 0U)
                {
                    if ((t & 0x02u) != 0U)
                        return HandTypes.StraightFlush;
                    else
                        is_st_or_fl = HandTypes.Flush;
                }

                if (is_st_or_fl != 0 && n_dups < 3)
                    return is_st_or_fl;
            }

            switch (n_dups)
            {
                case 0:
                    return HandTypes.HighCard;
                case 1:
                    return HandTypes.TwoOfAKind;
                case 2:
                    return ((ranks ^ (sc ^ sd ^ sh ^ ss)) != 0) ? HandTypes.TwoPair : HandTypes.ThreeOfAKind;
                default:
                    if (((sc & sd) & (sh & ss)) != 0) return HandTypes.FourOfAKind;
                    else if ((((sc & sd) | (sh & ss)) & ((sc & sh) | (sd & ss))) != 0) return HandTypes.FullHouse;
                    else if (is_st_or_fl != 0) return is_st_or_fl;
                    else return HandTypes.TwoPair;
            }
        }

        /// <summary>
        /// Evaluates a mask (passed as a mask mask) and returns a mask value.
        /// A mask value can be compared against another mask value to
        /// determine which has the higher value.
        /// </summary>
        /// <param name="cards">mask mask</param>
        /// <returns>Hand Value bit field</returns>
        public static uint Evaluate(ulong cards)
        {
            return Evaluate(cards, BitCount(cards));
        }

        /// <summary>
        /// Evaluates a mask (passed as a string) and returns a mask value.
        /// A mask value can be compared against another mask value to
        /// determine which has the higher value.
        /// </summary>
        /// <param name="mask">mask string</param>
        /// <returns>Hnad Value bit field</returns>
        public static uint Evaluate(string mask)
        {
            return Evaluate(Hand.ParseHand(mask));
        }

        /// <exclude/>
        private static readonly uint HANDTYPE_VALUE_STRAIGHTFLUSH = (((uint)HandTypes.StraightFlush) << HANDTYPE_SHIFT);
        /// <exclude/>
        private static readonly uint HANDTYPE_VALUE_STRAIGHT = (((uint)HandTypes.Straight) << HANDTYPE_SHIFT);
        /// <exclude/>
        private static readonly uint HANDTYPE_VALUE_FLUSH = (((uint)HandTypes.Flush) << HANDTYPE_SHIFT);
        /// <exclude/>
        private static readonly uint HANDTYPE_VALUE_FULLHOUSE = (((uint)HandTypes.FullHouse) << HANDTYPE_SHIFT);
        /// <exclude/>
        private static readonly uint HANDTYPE_VALUE_FOUR_OF_A_KIND = (((uint)HandTypes.FourOfAKind) << HANDTYPE_SHIFT);
        /// <exclude/>
        private static readonly uint HANDTYPE_VALUE_THREE_OF_A_KIND = (((uint)HandTypes.ThreeOfAKind) << HANDTYPE_SHIFT);
        /// <exclude/>
        private static readonly uint HANDTYPE_VALUE_TWOPAIR = (((uint)HandTypes.TwoPair) << HANDTYPE_SHIFT);
        /// <exclude/>
        private static readonly uint HANDTYPE_VALUE_TWO_OF_A_KIND = (((uint)HandTypes.TwoOfAKind) << HANDTYPE_SHIFT);
        /// <exclude/>
        private static readonly uint HANDTYPE_VALUE_HIGHCARD = (((uint)HandTypes.HighCard) << HANDTYPE_SHIFT);
        /// <exclude/>
        public static readonly int SPADE_OFFSET = 13 * Spades;
        /// <exclude/>
        public static readonly int CLUB_OFFSET = 13 * Clubs;
        /// <exclude/>
        public static readonly int DIAMOND_OFFSET = 13 * Diamonds;
        /// <exclude/>
        public static readonly int HEART_OFFSET = 13 * Hearts;

        /// <summary>
        /// Evaluates a mask (passed as a mask mask) and returns a mask value.
        /// A mask value can be compared against another mask value to
        /// determine which has the higher value.
        /// </summary>
        /// <param name="cards">mask mask</param>
        /// <param name="nCards">number of cards in the mask</param>
        /// <returns>mask value</returns>
        /// <example>
        /// <code>
        /// // Takes an ascii seven card mask and prints out a 
        /// // value description. For example: "ad kd 2d kh qh 3h qc" would
        /// // output "Two pair, King's and Queen's with a Ace for a kicker"
        /// static void PrintDescription(string mask)
        /// {
        ///     // Parse mask into a mask mask
        ///     ulong handmask = Hand.ParseHand(mask);
        /// 
        ///     // Convert mask mask into a compariable mask value.
        ///     uint handval = Hand.Evaluate(handmask, 7);
        /// 
        ///     // Print a description of the mask.
        ///     Console.WriteLine("Hand: {0}", Hand.DescriptionFromHandValue(handval));
        /// }
        /// </code>
        /// </example>
        public static uint Evaluate(ulong cards, int nCards)
        {
            uint retval = 0, four_mask, three_mask, two_mask;

#if DEBUG
            // This functions supports 1-7 cards
            if (nCards < 1 || nCards > 7) 
                throw new ArgumentOutOfRangeException("numberOfCards");
#endif
            // Seperate out by suit
            uint sc = (uint)((cards >> (CLUB_OFFSET)) & 0x1fffUL);
            uint sd = (uint)((cards >> (DIAMOND_OFFSET)) & 0x1fffUL);
            uint sh = (uint)((cards >> (HEART_OFFSET)) & 0x1fffUL);
            uint ss = (uint)((cards >> (SPADE_OFFSET)) & 0x1fffUL);

            uint ranks = sc | sd | sh | ss;
            uint n_ranks = nBitsTable[ranks];
            uint n_dups = ((uint)(nCards - n_ranks));

            /* Check for straight, flush, or straight flush, and return if we can
               determine immediately that this is the best possible mask 
            */
            if (n_ranks >= 5)
            {
                if (nBitsTable[ss] >= 5)
                {
                    if (straightTable[ss] != 0)
                        return HANDTYPE_VALUE_STRAIGHTFLUSH + (uint)(straightTable[ss] << TOP_CARD_SHIFT);
                    else
                        retval = HANDTYPE_VALUE_FLUSH + topFiveCardsTable[ss];
                }
                else if (nBitsTable[sc] >= 5)
                {
                    if (straightTable[sc] != 0)
                        return HANDTYPE_VALUE_STRAIGHTFLUSH + (uint)(straightTable[sc] << TOP_CARD_SHIFT);
                    else
                        retval = HANDTYPE_VALUE_FLUSH + topFiveCardsTable[sc];
                }
                else if (nBitsTable[sd] >= 5)
                {
                    if (straightTable[sd] != 0)
                        return HANDTYPE_VALUE_STRAIGHTFLUSH + (uint)(straightTable[sd] << TOP_CARD_SHIFT);
                    else
                        retval = HANDTYPE_VALUE_FLUSH + topFiveCardsTable[sd];
                }
                else if (nBitsTable[sh] >= 5)
                {
                    if (straightTable[sh] != 0)
                        return HANDTYPE_VALUE_STRAIGHTFLUSH + (uint)(straightTable[sh] << TOP_CARD_SHIFT);
                    else
                        retval = HANDTYPE_VALUE_FLUSH + topFiveCardsTable[sh];
                }
                else
                {
                    uint st = straightTable[ranks];
                    if (st != 0)
                        retval = HANDTYPE_VALUE_STRAIGHT + (st << TOP_CARD_SHIFT);
                }

                /* 
                   Another win -- if there can't be a FH/Quads (n_dups < 3), 
                   which is true most of the time when there is a made mask, then if we've
                   found a five card mask, just return.  This skips the whole process of
                   computing two_mask/three_mask/etc.
                */
                if (retval != 0 && n_dups < 3)
                    return retval;
            }

            /*
             * By the time we're here, either: 
               1) there's no five-card mask possible (flush or straight), or
               2) there's a flush or straight, but we know that there are enough
                  duplicates to make a full house / quads possible.  
             */
            switch (n_dups)
            {
                case 0:
                    /* It's a no-pair mask */
                    return HANDTYPE_VALUE_HIGHCARD + topFiveCardsTable[ranks];
                case 1:
                    {
                        /* It's a one-pair mask */
                        uint t, kickers;

                        two_mask = ranks ^ (sc ^ sd ^ sh ^ ss);

                        retval = (uint) (HANDTYPE_VALUE_TWO_OF_A_KIND + (topCardTable[two_mask] << TOP_CARD_SHIFT));
                        t = ranks ^ two_mask;      /* Only one bit set in two_mask */
                        /* Get the top five cards in what is left, drop all but the top three 
                         * cards, and shift them by one to get the three desired kickers */
                        kickers = (topFiveCardsTable[t] >> CARD_WIDTH) & ~FIFTH_CARD_MASK;
                        retval += kickers;
                        return retval;
                    }

                case 2:
                    /* Either two pair or trips */
                    two_mask = ranks ^ (sc ^ sd ^ sh ^ ss);
                    if (two_mask != 0)
                    {
                        uint t = ranks ^ two_mask; /* Exactly two bits set in two_mask */
                        retval = (uint) (HANDTYPE_VALUE_TWOPAIR
                            + (topFiveCardsTable[two_mask]
                            & (TOP_CARD_MASK | SECOND_CARD_MASK))
                            + (topCardTable[t] << THIRD_CARD_SHIFT));

                        return retval;
                    }
                    else
                    {
                        uint t, second;
                        three_mask = ((sc & sd) | (sh & ss)) & ((sc & sh) | (sd & ss));
                        retval = (uint) (HANDTYPE_VALUE_THREE_OF_A_KIND + (topCardTable[three_mask] << TOP_CARD_SHIFT));
                        t = ranks ^ three_mask; /* Only one bit set in three_mask */
                        second = topCardTable[t];
                        retval += (second << SECOND_CARD_SHIFT);
                        t ^= (1U << (int)second);
                        retval += (uint) (topCardTable[t] << THIRD_CARD_SHIFT);
                        return retval;
                    }

                default:
                    /* Possible quads, fullhouse, straight or flush, or two pair */
                    four_mask = sh & sd & sc & ss;
                    if (four_mask != 0)
                    {
                        uint tc = topCardTable[four_mask];
                        retval = (uint) (HANDTYPE_VALUE_FOUR_OF_A_KIND
                            + (tc << TOP_CARD_SHIFT)
                            + ((topCardTable[ranks ^ (1U << (int)tc)]) << SECOND_CARD_SHIFT));
                        return retval;
                    }

                    /* Technically, three_mask as defined below is really the set of
                       bits which are set in three or four of the suits, but since
                       we've already eliminated quads, this is OK */
                    /* Similarly, two_mask is really two_or_four_mask, but since we've
                       already eliminated quads, we can use this shortcut */

                    two_mask = ranks ^ (sc ^ sd ^ sh ^ ss);
                    if (nBitsTable[two_mask] != n_dups)
                    {
                        /* Must be some trips then, which really means there is a 
                           full house since n_dups >= 3 */
                        uint tc, t;
                        three_mask = ((sc & sd) | (sh & ss)) & ((sc & sh) | (sd & ss));
                        retval = HANDTYPE_VALUE_FULLHOUSE;
                        tc = topCardTable[three_mask];
                        retval += (tc << TOP_CARD_SHIFT);
                        t = (two_mask | three_mask) ^ (1U << (int)tc);
                        retval += (uint) (topCardTable[t] << SECOND_CARD_SHIFT);
                        return retval;
                    }

                    if (retval != 0) /* flush and straight */
                        return retval;
                    else
                    {
                        /* Must be two pair */
                        uint top, second;

                        retval = HANDTYPE_VALUE_TWOPAIR;
                        top = topCardTable[two_mask];
                        retval += (top << TOP_CARD_SHIFT);
                        second = topCardTable[two_mask ^ (1 << (int)top)];
                        retval += (second << SECOND_CARD_SHIFT);
                        retval += (uint) ((topCardTable[ranks ^ (1U << (int)top) ^ (1 << (int)second)]) << THIRD_CARD_SHIFT);
                        return retval;
                    }
            }
        }

        #endregion

        #region IComparable Members

        /// <summary>
        /// Used to compare one mask to another. This method allows
        /// normal compare functions to work as expected with a mask.
        /// </summary>
        /// <param name="obj">object to compare against</param>
        /// <returns></returns>
        public int CompareTo(object obj)
        {
            Hand h = obj as Hand;
            if (h == null) return -1;
            return ((int)HandValue) - ((int)h.HandValue);
        }

        /// <summary>
        /// Test for equality
        /// </summary>
        /// <param name="obj"></param>
        /// <returns>returns true if equal, false otherwise</returns>
        public override bool Equals(object obj)
        {
            return handval == ((Hand)obj).handval;
        }

        /// <summary>
        /// Returns hash code
        /// </summary>
        /// <returns>Hash code</returns>
        public override int GetHashCode()
        {
            return (int)handval;
        }

        /// <summary>
        /// Test for equality
        /// </summary>
        /// <param name="op1">left side object</param>
        /// <param name="op2">right side object</param>
        /// <returns>returns true if equal, false otherwise</returns>
        /// <example>
        /// <code>
        /// // These two hands are equal because the five best cards
        /// // in both hands are the same.
        /// Hand h1 = new Hand("ac as", "4d 5d 6c 7c 8d");
        /// Hand h2 = new Hand("td js", "4d 5d 6c 7c 8d");
        /// 
        /// if (h1 == h2)
        /// {
        ///     Console.WriteLine("Hands are equal");
        /// }
        /// </code>
        /// </example>
        static public bool operator ==(Hand op1, Hand op2)
        {
#if DEBUG
            if (object.ReferenceEquals(op1, null) || object.ReferenceEquals(op2, null))
                throw new ArgumentNullException();
#endif
            return op1.handval == op2.handval;
        }

        /// <summary>
        /// Test for inequality.
        /// </summary>
        /// <param name="op1">left side object</param>
        /// <param name="op2">right side object</param>
        /// <returns>returns true if not equal, false otherwise</returns>
        /// <example>
        /// <code>
        /// Hand h1 = new Hand("ac as", "4d 5d 6c 7c 8d");
        /// Hand h2 = new Hand("td 9s", "4d 5d 6c 7c 8d");
        /// 
        /// if (h1 != h2)
        /// {
        ///     Console.WriteLine("Hand h2 is a higher straight");
        /// }
        /// </code>
        /// </example>
        static public bool operator !=(Hand op1, Hand op2)
        {
#if DEBUG
            if (object.ReferenceEquals(op1, null) || object.ReferenceEquals(op2, null))
                throw new ArgumentNullException();
#endif
            return op1.handval != op2.handval;
        }

        /// <summary>
        /// Test that the left side is greater than the right side.
        /// </summary>
        /// <param name="op1">left side</param>
        /// <param name="op2">right side</param>
        /// <returns>returns true of the left item is greater than the right item</returns>
        /// <example>
        /// <code>
        /// Hand h1 = new Hand("ac as", "4d 5d 6c 7c 8d");
        /// Hand h2 = new Hand("td 9s", "4d 5d 6c 7c 8d");
        /// 
        /// if (h2 > h1)
        /// {
        ///     Console.WriteLine("Hand h2 is a higher straight");
        /// }
        /// </code>
        /// </example>
        static public bool operator >(Hand op1, Hand op2)
        {
#if DEBUG
            if (object.ReferenceEquals(op1, null) || object.ReferenceEquals(op2, null))
                throw new ArgumentNullException();
#endif
            return op1.handval > op2.handval;
        }

        /// <summary>
        /// Test that the left side is greater or equal than the right side.
        /// </summary>
        /// <param name="op1">left side</param>
        /// <param name="op2">right side</param>
        /// <returns>returns true of the left item is greater or equal than the right item</returns>
        /// <example>
        /// <code>
        /// Hand h1 = new Hand("ac as", "4d 5d 6c 7c 8d");
        /// Hand h2 = new Hand("td 9s", "4d 5d 6c 7c 8d");
        /// 
        /// if (h2 >= h1)
        /// {
        ///     Console.WriteLine("Hand h2 is a higher straight");
        /// }
        /// </code>
        /// </example>
        static public bool operator >=(Hand op1, Hand op2)
        {
#if DEBUG
            if (object.ReferenceEquals(op1, null) || object.ReferenceEquals(op2, null))
                throw new ArgumentNullException();
#endif
            return op1.handval >= op2.handval;
        }

        /// <summary>
        /// Test that the left side is less than the right side.
        /// </summary>
        /// <param name="op1">left side</param>
        /// <param name="op2">right side</param>
        /// <returns>returns true if the left item is less than the right item.</returns>
        static public bool operator <(Hand op1, Hand op2)
        {
#if DEBUG
            if (object.ReferenceEquals(op1, null) || object.ReferenceEquals(op2, null))
                throw new ArgumentNullException();
#endif
            return op1.handval < op2.handval;
        }

        /// <summary>
        /// Test that the left side is less than or equal to the right side.
        /// </summary>
        /// <param name="op1">left side</param>
        /// <param name="op2">right side</param>
        /// <returns>returns true if the left item is less than or equal to the right item.</returns>
        static public bool operator <=(Hand op1, Hand op2)
        {
#if DEBUG
            if (object.ReferenceEquals(op1, null) || object.ReferenceEquals(op2, null))
                throw new ArgumentNullException();
#endif
            return op1.handval <= op2.handval;
        }

        #endregion

        #region Lookup Tables

        #region Fast BitCounter

        /// <summary>
        /// Count Bits. Optimized for cards so only works with 52 bits.
        /// </summary>
        /// <param name="bitField"></param>
        /// <returns></returns>
        public static int BitCount(ulong bitField)
        {
            return  nBitsTable[bitField & 0x1fff] +
                    nBitsTable[(bitField >> 13) & 0x1fff] +
                    nBitsTable[(bitField >> 26) & 0x1fff] +
                    nBitsTable[(bitField >> 39) & 0x1fff];
        }
        #endregion

        #region nBitsAndStr Table
        /// <exclude/>
        public static readonly UInt16[] nBitsAndStrTable = 
		{ 
			0x0, 0x4, 0x4, 0x8, 0x4, 0x8, 0x8, 0xc, 0x4, 0x8, 0x8, 0xc, 0x8, 0xc, 0xc, 0x10, 0x4, 0x8, 0x8, 0xc, 
            0x8, 0xc, 0xc, 0x10, 0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x17, 0x4, 0x8, 0x8, 0xc, 0x8, 0xc, 0xc, 0x10, 
            0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 0xc, 0x10, 0x10, 0x15, 
            0x10, 0x15, 0x17, 0x1b, 0x4, 0x8, 0x8, 0xc, 0x8, 0xc, 0xc, 0x10, 0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 
            0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x1b, 0x8, 0xc, 0xc, 0x10, 
            0xc, 0x10, 0x10, 0x15, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 
            0x10, 0x15, 0x15, 0x19, 0x17, 0x1b, 0x1b, 0x1f, 0x4, 0x8, 0x8, 0xc, 0x8, 0xc, 0xc, 0x10, 0x8, 0xc, 0xc, 0x10, 
            0xc, 0x10, 0x10, 0x15, 0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x1b, 
            0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0xc, 0x10, 0x10, 0x15, 
            0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x1b, 0x1f, 0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 
            0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1f, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x17, 0x1b, 0x1b, 0x1f, 0x1b, 0x1f, 0x1f, 0x23, 0x4, 0x8, 0x8, 0xc, 
            0x8, 0xc, 0xc, 0x10, 0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 
            0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x1b, 0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 0xc, 0x10, 0x10, 0x15, 
            0x10, 0x15, 0x15, 0x19, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x1b, 0x1f, 
            0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0xc, 0x10, 0x10, 0x15, 
            0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 
            0x1b, 0x1f, 0x1f, 0x23, 0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 
            0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 0xc, 0x10, 0x10, 0x15, 
            0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1f, 0x23, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1d, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x17, 0x1b, 0x1b, 0x1f, 
            0x1b, 0x1f, 0x1f, 0x23, 0x1b, 0x1f, 0x1f, 0x23, 0x1f, 0x23, 0x23, 0x27, 0x4, 0x8, 0x8, 0xc, 0x8, 0xc, 0xc, 0x10, 
            0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 0xc, 0x10, 0x10, 0x15, 
            0x10, 0x15, 0x15, 0x1b, 0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 
            0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x1b, 0x1f, 0x8, 0xc, 0xc, 0x10, 
            0xc, 0x10, 0x10, 0x15, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1d, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x1b, 0x1f, 0x1f, 0x23, 
            0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0xc, 0x10, 0x10, 0x15, 
            0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1f, 0x23, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 
            0x1b, 0x1f, 0x1f, 0x23, 0x1f, 0x23, 0x23, 0x27, 0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 0xc, 0x10, 0x10, 0x15, 
            0x10, 0x15, 0x15, 0x19, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 
            0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1f, 0x23, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x23, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1f, 0x23, 0x23, 0x27, 0xc, 0x10, 0x10, 0x15, 
            0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x23, 0x27, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x17, 0x1b, 0x1b, 0x1f, 0x1b, 0x1f, 0x1f, 0x23, 
            0x1b, 0x1f, 0x1f, 0x23, 0x1f, 0x23, 0x23, 0x27, 0x1b, 0x1f, 0x1f, 0x23, 0x1f, 0x23, 0x23, 0x27, 0x1f, 0x23, 0x23, 0x27, 
            0x23, 0x27, 0x27, 0x2b, 0x4, 0x8, 0x8, 0xc, 0x8, 0xc, 0xc, 0x10, 0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 
            0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x1b, 0x8, 0xc, 0xc, 0x10, 
            0xc, 0x10, 0x10, 0x15, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x1b, 0x1f, 0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 0xc, 0x10, 0x10, 0x15, 
            0x10, 0x15, 0x15, 0x19, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 
            0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x1b, 0x1f, 0x1f, 0x23, 0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 
            0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1f, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1f, 0x23, 0xc, 0x10, 0x10, 0x15, 
            0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x1b, 0x1f, 0x1f, 0x23, 0x1f, 0x23, 0x23, 0x27, 
            0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0xc, 0x10, 0x10, 0x15, 
            0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1f, 0x23, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 
            0x19, 0x1d, 0x1d, 0x21, 0x1f, 0x23, 0x23, 0x27, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1d, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x23, 0x27, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x27, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 
            0x1b, 0x1f, 0x1f, 0x23, 0x1f, 0x23, 0x23, 0x27, 0x1f, 0x23, 0x23, 0x27, 0x23, 0x27, 0x27, 0x2b, 0x8, 0xc, 0xc, 0x10, 
            0xc, 0x10, 0x10, 0x15, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1d, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1f, 0x23, 
            0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 
            0x1f, 0x23, 0x23, 0x27, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x23, 0x27, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x25, 0x1f, 0x23, 0x23, 0x27, 0x23, 0x27, 0x27, 0x2b, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x23, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x23, 0x27, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x25, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x23, 0x27, 0x27, 0x2b, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 
            0x21, 0x25, 0x27, 0x2b, 0x17, 0x1b, 0x1b, 0x1f, 0x1b, 0x1f, 0x1f, 0x23, 0x1b, 0x1f, 0x1f, 0x23, 0x1f, 0x23, 0x23, 0x27, 
            0x1b, 0x1f, 0x1f, 0x23, 0x1f, 0x23, 0x23, 0x27, 0x1f, 0x23, 0x23, 0x27, 0x23, 0x27, 0x27, 0x2b, 0x1b, 0x1f, 0x1f, 0x23, 
            0x1f, 0x23, 0x23, 0x27, 0x1f, 0x23, 0x23, 0x27, 0x23, 0x27, 0x27, 0x2b, 0x1f, 0x23, 0x23, 0x27, 0x23, 0x27, 0x27, 0x2b, 
            0x23, 0x27, 0x27, 0x2b, 0x27, 0x2b, 0x2b, 0x2f, 0x4, 0x8, 0x8, 0xc, 0x8, 0xc, 0xc, 0x10, 0x8, 0xc, 0xc, 0x10, 
            0xc, 0x10, 0x10, 0x15, 0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x1b, 
            0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0xc, 0x10, 0x10, 0x15, 
            0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x1b, 0x1f, 0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 
            0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1f, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x1b, 0x1f, 0x1f, 0x23, 0x8, 0xc, 0xc, 0x10, 
            0xc, 0x10, 0x10, 0x15, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1d, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1f, 0x23, 
            0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x1b, 0x1f, 0x1f, 0x23, 
            0x1f, 0x23, 0x23, 0x27, 0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 
            0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 0xc, 0x10, 0x10, 0x15, 
            0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1f, 0x23, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1d, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1f, 0x23, 0x23, 0x27, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x23, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x23, 0x27, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x25, 0x1b, 0x1f, 0x1f, 0x23, 0x1f, 0x23, 0x23, 0x27, 0x1f, 0x23, 0x23, 0x27, 0x23, 0x27, 0x27, 0x2b, 
            0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0xc, 0x10, 0x10, 0x15, 
            0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1f, 0x23, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 
            0x19, 0x1d, 0x1d, 0x21, 0x1f, 0x23, 0x23, 0x27, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1d, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x23, 0x27, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x27, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1f, 0x23, 0x23, 0x27, 0x23, 0x27, 0x27, 0x2b, 0xc, 0x10, 0x10, 0x15, 
            0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x23, 0x27, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 
            0x23, 0x27, 0x27, 0x2b, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 
            0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x27, 0x2b, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x25, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x2b, 
            0x1b, 0x1f, 0x1f, 0x23, 0x1f, 0x23, 0x23, 0x27, 0x1f, 0x23, 0x23, 0x27, 0x23, 0x27, 0x27, 0x2b, 0x1f, 0x23, 0x23, 0x27, 
            0x23, 0x27, 0x27, 0x2b, 0x23, 0x27, 0x27, 0x2b, 0x27, 0x2b, 0x2b, 0x2f, 0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 
            0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1f, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1f, 0x23, 0xc, 0x10, 0x10, 0x15, 
            0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1f, 0x23, 0x23, 0x27, 
            0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x23, 0x27, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 
            0x1f, 0x23, 0x23, 0x27, 0x23, 0x27, 0x27, 0x2b, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1d, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x23, 0x27, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x27, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x23, 0x27, 0x27, 0x2b, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x25, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x27, 0x2b, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x2b, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 
            0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x29, 0x1f, 0x23, 0x23, 0x27, 0x23, 0x27, 0x27, 0x2b, 0x23, 0x27, 0x27, 0x2b, 
            0x27, 0x2b, 0x2b, 0x2f, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x23, 0x27, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x23, 0x27, 0x27, 0x2b, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x27, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x27, 0x2b, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 
            0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x2b, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 
            0x21, 0x25, 0x25, 0x29, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x29, 0x23, 0x27, 0x27, 0x2b, 0x27, 0x2b, 0x2b, 0x2f, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 
            0x21, 0x25, 0x27, 0x2b, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x2b, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x29, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x29, 
            0x21, 0x25, 0x25, 0x29, 0x27, 0x2b, 0x2b, 0x2f, 0x17, 0x1b, 0x1b, 0x1f, 0x1b, 0x1f, 0x1f, 0x23, 0x1b, 0x1f, 0x1f, 0x23, 
            0x1f, 0x23, 0x23, 0x27, 0x1b, 0x1f, 0x1f, 0x23, 0x1f, 0x23, 0x23, 0x27, 0x1f, 0x23, 0x23, 0x27, 0x23, 0x27, 0x27, 0x2b, 
            0x1b, 0x1f, 0x1f, 0x23, 0x1f, 0x23, 0x23, 0x27, 0x1f, 0x23, 0x23, 0x27, 0x23, 0x27, 0x27, 0x2b, 0x1f, 0x23, 0x23, 0x27, 
            0x23, 0x27, 0x27, 0x2b, 0x23, 0x27, 0x27, 0x2b, 0x27, 0x2b, 0x2b, 0x2f, 0x1b, 0x1f, 0x1f, 0x23, 0x1f, 0x23, 0x23, 0x27, 
            0x1f, 0x23, 0x23, 0x27, 0x23, 0x27, 0x27, 0x2b, 0x1f, 0x23, 0x23, 0x27, 0x23, 0x27, 0x27, 0x2b, 0x23, 0x27, 0x27, 0x2b, 
            0x27, 0x2b, 0x2b, 0x2f, 0x1f, 0x23, 0x23, 0x27, 0x23, 0x27, 0x27, 0x2b, 0x23, 0x27, 0x27, 0x2b, 0x27, 0x2b, 0x2b, 0x2f, 
            0x23, 0x27, 0x27, 0x2b, 0x27, 0x2b, 0x2b, 0x2f, 0x27, 0x2b, 0x2b, 0x2f, 0x2b, 0x2f, 0x2f, 0x33, 0x4, 0x8, 0x8, 0xc, 
            0x8, 0xc, 0xc, 0x10, 0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x17, 0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 
            0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x1b, 0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 0xc, 0x10, 0x10, 0x15, 
            0x10, 0x15, 0x15, 0x1b, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x1b, 0x1f, 
            0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x1b, 0xc, 0x10, 0x10, 0x15, 
            0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 
            0x1b, 0x1f, 0x1f, 0x23, 0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x1b, 
            0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 0xc, 0x10, 0x10, 0x15, 
            0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1f, 0x23, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1f, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x1b, 0x1f, 0x1f, 0x23, 0x1f, 0x23, 0x23, 0x27, 0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 
            0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x1b, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1f, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1f, 0x23, 0xc, 0x10, 0x10, 0x15, 
            0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x23, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1f, 0x23, 0x23, 0x27, 
            0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x23, 0x27, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x1b, 0x1f, 0x1f, 0x23, 0x1f, 0x23, 0x23, 0x27, 
            0x1f, 0x23, 0x23, 0x27, 0x23, 0x27, 0x27, 0x2b, 0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 0xc, 0x10, 0x10, 0x15, 
            0x10, 0x15, 0x15, 0x1b, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 
            0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1f, 0x23, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x23, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1f, 0x23, 0x23, 0x27, 0xc, 0x10, 0x10, 0x15, 
            0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x23, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x23, 0x27, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1f, 0x23, 0x23, 0x27, 
            0x23, 0x27, 0x27, 0x2b, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x23, 0x27, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x23, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x23, 0x27, 0x27, 0x2b, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x27, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x27, 0x2b, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 
            0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x2b, 0x1b, 0x1f, 0x1f, 0x23, 0x1f, 0x23, 0x23, 0x27, 0x1f, 0x23, 0x23, 0x27, 
            0x23, 0x27, 0x27, 0x2b, 0x1f, 0x23, 0x23, 0x27, 0x23, 0x27, 0x27, 0x2b, 0x23, 0x27, 0x27, 0x2b, 0x27, 0x2b, 0x2b, 0x2f, 
            0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x1b, 0xc, 0x10, 0x10, 0x15, 
            0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1f, 0x23, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 
            0x19, 0x1d, 0x1d, 0x21, 0x1f, 0x23, 0x23, 0x27, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1f, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x23, 0x27, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x27, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1f, 0x23, 0x23, 0x27, 0x23, 0x27, 0x27, 0x2b, 0xc, 0x10, 0x10, 0x15, 
            0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x23, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x23, 0x27, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 
            0x23, 0x27, 0x27, 0x2b, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 
            0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x27, 0x2b, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x27, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x2b, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x2b, 0x1f, 0x23, 0x23, 0x27, 
            0x23, 0x27, 0x27, 0x2b, 0x23, 0x27, 0x27, 0x2b, 0x27, 0x2b, 0x2b, 0x2f, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x23, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x23, 0x27, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x27, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x23, 0x27, 0x27, 0x2b, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 
            0x21, 0x25, 0x27, 0x2b, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x2b, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x2b, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x29, 
            0x23, 0x27, 0x27, 0x2b, 0x27, 0x2b, 0x2b, 0x2f, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x23, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x27, 0x2b, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 
            0x21, 0x25, 0x25, 0x2b, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x2b, 
            0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x29, 0x21, 0x25, 0x25, 0x29, 0x27, 0x2b, 0x2b, 0x2f, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 
            0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x2b, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 
            0x21, 0x25, 0x25, 0x2b, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x29, 0x21, 0x25, 0x25, 0x29, 0x25, 0x29, 0x2b, 0x2f, 
            0x1b, 0x1f, 0x1f, 0x23, 0x1f, 0x23, 0x23, 0x27, 0x1f, 0x23, 0x23, 0x27, 0x23, 0x27, 0x27, 0x2b, 0x1f, 0x23, 0x23, 0x27, 
            0x23, 0x27, 0x27, 0x2b, 0x23, 0x27, 0x27, 0x2b, 0x27, 0x2b, 0x2b, 0x2f, 0x1f, 0x23, 0x23, 0x27, 0x23, 0x27, 0x27, 0x2b, 
            0x23, 0x27, 0x27, 0x2b, 0x27, 0x2b, 0x2b, 0x2f, 0x23, 0x27, 0x27, 0x2b, 0x27, 0x2b, 0x2b, 0x2f, 0x27, 0x2b, 0x2b, 0x2f, 
            0x2b, 0x2f, 0x2f, 0x33, 0x8, 0xc, 0xc, 0x10, 0xc, 0x10, 0x10, 0x15, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x1b, 
            0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 0xc, 0x10, 0x10, 0x15, 
            0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1f, 0x23, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1f, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1f, 0x23, 0x23, 0x27, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x23, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x23, 0x27, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x27, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1f, 0x23, 0x23, 0x27, 0x23, 0x27, 0x27, 0x2b, 
            0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x23, 0x27, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 
            0x1d, 0x21, 0x21, 0x25, 0x23, 0x27, 0x27, 0x2b, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x23, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x27, 0x2b, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 
            0x21, 0x25, 0x25, 0x2b, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x2b, 
            0x1f, 0x23, 0x23, 0x27, 0x23, 0x27, 0x27, 0x2b, 0x23, 0x27, 0x27, 0x2b, 0x27, 0x2b, 0x2b, 0x2f, 0xc, 0x10, 0x10, 0x15, 
            0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1f, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x23, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x23, 0x27, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 
            0x23, 0x27, 0x27, 0x2b, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 
            0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x27, 0x2b, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x27, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x2b, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x2b, 0x1d, 0x21, 0x21, 0x25, 
            0x21, 0x25, 0x25, 0x29, 0x23, 0x27, 0x27, 0x2b, 0x27, 0x2b, 0x2b, 0x2f, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x27, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x27, 0x2b, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 
            0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x2b, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 
            0x21, 0x25, 0x25, 0x2b, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x29, 0x21, 0x25, 0x25, 0x29, 0x27, 0x2b, 0x2b, 0x2f, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x2b, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 
            0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x2b, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x29, 0x21, 0x25, 0x25, 0x29, 
            0x25, 0x29, 0x2b, 0x2f, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x2b, 
            0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x29, 0x21, 0x25, 0x25, 0x29, 0x25, 0x29, 0x29, 0x2f, 0x1f, 0x23, 0x23, 0x27, 
            0x23, 0x27, 0x27, 0x2b, 0x23, 0x27, 0x27, 0x2b, 0x27, 0x2b, 0x2b, 0x2f, 0x23, 0x27, 0x27, 0x2b, 0x27, 0x2b, 0x2b, 0x2f, 
            0x27, 0x2b, 0x2b, 0x2f, 0x2b, 0x2f, 0x2f, 0x33, 0xc, 0x10, 0x10, 0x15, 0x10, 0x15, 0x15, 0x19, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1f, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x23, 0x27, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x27, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x23, 0x27, 0x27, 0x2b, 0x10, 0x15, 0x15, 0x19, 
            0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x27, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x27, 0x2b, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x2b, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 
            0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x2b, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x29, 0x23, 0x27, 0x27, 0x2b, 
            0x27, 0x2b, 0x2b, 0x2f, 0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 
            0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 
            0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x27, 0x2b, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x27, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x2b, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x2b, 0x1d, 0x21, 0x21, 0x25, 
            0x21, 0x25, 0x25, 0x29, 0x21, 0x25, 0x25, 0x29, 0x27, 0x2b, 0x2b, 0x2f, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 
            0x21, 0x25, 0x25, 0x2b, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x2b, 
            0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x29, 0x21, 0x25, 0x25, 0x29, 0x25, 0x29, 0x2b, 0x2f, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x2b, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x29, 
            0x21, 0x25, 0x25, 0x29, 0x25, 0x29, 0x29, 0x2f, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x29, 0x21, 0x25, 0x25, 0x29, 
            0x25, 0x29, 0x29, 0x2f, 0x23, 0x27, 0x27, 0x2b, 0x27, 0x2b, 0x2b, 0x2f, 0x27, 0x2b, 0x2b, 0x2f, 0x2b, 0x2f, 0x2f, 0x33, 
            0x10, 0x15, 0x15, 0x19, 0x15, 0x19, 0x19, 0x1d, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x23, 0x15, 0x19, 0x19, 0x1d, 
            0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 
            0x21, 0x25, 0x27, 0x2b, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x27, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x2b, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x2b, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x29, 
            0x21, 0x25, 0x25, 0x29, 0x27, 0x2b, 0x2b, 0x2f, 0x15, 0x19, 0x19, 0x1d, 0x19, 0x1d, 0x1d, 0x21, 0x19, 0x1d, 0x1d, 0x21, 
            0x1d, 0x21, 0x21, 0x27, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x2b, 
            0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x2b, 0x1d, 0x21, 0x21, 0x25, 
            0x21, 0x25, 0x25, 0x29, 0x21, 0x25, 0x25, 0x29, 0x25, 0x29, 0x2b, 0x2f, 0x19, 0x1d, 0x1d, 0x21, 0x1d, 0x21, 0x21, 0x25, 
            0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x2b, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x29, 0x21, 0x25, 0x25, 0x29, 
            0x25, 0x29, 0x29, 0x2f, 0x1d, 0x21, 0x21, 0x25, 0x21, 0x25, 0x25, 0x29, 0x21, 0x25, 0x25, 0x29, 0x25, 0x29, 0x29, 0x2f, 
            0x21, 0x25, 0x25, 0x29, 0x25, 0x29, 0x29, 0x2d, 0x27, 0x2b, 0x2b, 0x2f, 0x2b, 0x2f, 0x2f, 0x33, 0x17, 0x1b, 0x1b, 0x1f, 
            0x1b, 0x1f, 0x1f, 0x23, 0x1b, 0x1f, 0x1f, 0x23, 0x1f, 0x23, 0x23, 0x27, 0x1b, 0x1f, 0x1f, 0x23, 0x1f, 0x23, 0x23, 0x27, 
            0x1f, 0x23, 0x23, 0x27, 0x23, 0x27, 0x27, 0x2b, 0x1b, 0x1f, 0x1f, 0x23, 0x1f, 0x23, 0x23, 0x27, 0x1f, 0x23, 0x23, 0x27, 
            0x23, 0x27, 0x27, 0x2b, 0x1f, 0x23, 0x23, 0x27, 0x23, 0x27, 0x27, 0x2b, 0x23, 0x27, 0x27, 0x2b, 0x27, 0x2b, 0x2b, 0x2f, 
            0x1b, 0x1f, 0x1f, 0x23, 0x1f, 0x23, 0x23, 0x27, 0x1f, 0x23, 0x23, 0x27, 0x23, 0x27, 0x27, 0x2b, 0x1f, 0x23, 0x23, 0x27, 
            0x23, 0x27, 0x27, 0x2b, 0x23, 0x27, 0x27, 0x2b, 0x27, 0x2b, 0x2b, 0x2f, 0x1f, 0x23, 0x23, 0x27, 0x23, 0x27, 0x27, 0x2b, 
            0x23, 0x27, 0x27, 0x2b, 0x27, 0x2b, 0x2b, 0x2f, 0x23, 0x27, 0x27, 0x2b, 0x27, 0x2b, 0x2b, 0x2f, 0x27, 0x2b, 0x2b, 0x2f, 
            0x2b, 0x2f, 0x2f, 0x33, 0x1b, 0x1f, 0x1f, 0x23, 0x1f, 0x23, 0x23, 0x27, 0x1f, 0x23, 0x23, 0x27, 0x23, 0x27, 0x27, 0x2b, 
            0x1f, 0x23, 0x23, 0x27, 0x23, 0x27, 0x27, 0x2b, 0x23, 0x27, 0x27, 0x2b, 0x27, 0x2b, 0x2b, 0x2f, 0x1f, 0x23, 0x23, 0x27, 
            0x23, 0x27, 0x27, 0x2b, 0x23, 0x27, 0x27, 0x2b, 0x27, 0x2b, 0x2b, 0x2f, 0x23, 0x27, 0x27, 0x2b, 0x27, 0x2b, 0x2b, 0x2f, 
            0x27, 0x2b, 0x2b, 0x2f, 0x2b, 0x2f, 0x2f, 0x33, 0x1f, 0x23, 0x23, 0x27, 0x23, 0x27, 0x27, 0x2b, 0x23, 0x27, 0x27, 0x2b, 
            0x27, 0x2b, 0x2b, 0x2f, 0x23, 0x27, 0x27, 0x2b, 0x27, 0x2b, 0x2b, 0x2f, 0x27, 0x2b, 0x2b, 0x2f, 0x2b, 0x2f, 0x2f, 0x33, 
            0x23, 0x27, 0x27, 0x2b, 0x27, 0x2b, 0x2b, 0x2f, 0x27, 0x2b, 0x2b, 0x2f, 0x2b, 0x2f, 0x2f, 0x33, 0x27, 0x2b, 0x2b, 0x2f, 
            0x2b, 0x2f, 0x2f, 0x33, 0x2b, 0x2f, 0x2f, 0x33, 0x2f, 0x33, 0x33, 0x37
		};
        #endregion

        #region nBitsTable
        // A table representing the bit count for a 13 bit integer.
        /// <exclude/>
        private static readonly System.UInt16[] nBitsTable = 
		{ 
            0, 1, 1, 2, 1, 2, 2, 3, 1, 2, 2, 3, 2, 3, 3,
            4, 1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4,
            4, 5, 1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3,
            4, 4, 5, 2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5,
            4, 5, 5, 6, 1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3,
            4, 3, 4, 4, 5, 2, 3, 3, 4, 3, 4, 4, 5, 3, 4,
            4, 5, 4, 5, 5, 6, 2, 3, 3, 4, 3, 4, 4, 5, 3,
            4, 4, 5, 4, 5, 5, 6, 3, 4, 4, 5, 4, 5, 5, 6,
            4, 5, 5, 6, 5, 6, 6, 7, 1, 2, 2, 3, 2, 3, 3,
            4, 2, 3, 3, 4, 3, 4, 4, 5, 2, 3, 3, 4, 3, 4,
            4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 2, 3, 3, 4, 3,
            4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 3, 4, 4, 5,
            4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 2, 3, 3,
            4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 3, 4,
            4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 3,
            4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
            4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7,
            8, 1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4,
            4, 5, 2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4,
            5, 5, 6, 2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5,
            4, 5, 5, 6, 3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5,
            6, 5, 6, 6, 7, 2, 3, 3, 4, 3, 4, 4, 5, 3, 4,
            4, 5, 4, 5, 5, 6, 3, 4, 4, 5, 4, 5, 5, 6, 4,
            5, 5, 6, 5, 6, 6, 7, 3, 4, 4, 5, 4, 5, 5, 6,
            4, 5, 5, 6, 5, 6, 6, 7, 4, 5, 5, 6, 5, 6, 6,
            7, 5, 6, 6, 7, 6, 7, 7, 8, 2, 3, 3, 4, 3, 4,
            4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 3, 4, 4, 5, 4,
            5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 3, 4, 4, 5,
            4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 4, 5, 5,
            6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 3, 4,
            4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 4,
            5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8,
            4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7,
            8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8,
            8, 9, 1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3,
            4, 4, 5, 2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5,
            4, 5, 5, 6, 2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4,
            5, 4, 5, 5, 6, 3, 4, 4, 5, 4, 5, 5, 6, 4, 5,
            5, 6, 5, 6, 6, 7, 2, 3, 3, 4, 3, 4, 4, 5, 3,
            4, 4, 5, 4, 5, 5, 6, 3, 4, 4, 5, 4, 5, 5, 6,
            4, 5, 5, 6, 5, 6, 6, 7, 3, 4, 4, 5, 4, 5, 5,
            6, 4, 5, 5, 6, 5, 6, 6, 7, 4, 5, 5, 6, 5, 6,
            6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 2, 3, 3, 4, 3,
            4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 3, 4, 4, 5,
            4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 3, 4, 4,
            5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 4, 5,
            5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 3,
            4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
            4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7,
            8, 4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7,
            7, 8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7,
            8, 8, 9, 2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5,
            4, 5, 5, 6, 3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5,
            6, 5, 6, 6, 7, 3, 4, 4, 5, 4, 5, 5, 6, 4, 5,
            5, 6, 5, 6, 6, 7, 4, 5, 5, 6, 5, 6, 6, 7, 5,
            6, 6, 7, 6, 7, 7, 8, 3, 4, 4, 5, 4, 5, 5, 6,
            4, 5, 5, 6, 5, 6, 6, 7, 4, 5, 5, 6, 5, 6, 6,
            7, 5, 6, 6, 7, 6, 7, 7, 8, 4, 5, 5, 6, 5, 6,
            6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 5, 6, 6, 7, 6,
            7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 3, 4, 4, 5,
            4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 4, 5, 5,
            6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 4, 5,
            5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 5,
            6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9,
            4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7,
            8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8,
            8, 9, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7,
            8, 8, 9, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8, 8, 9,
            8, 9, 9, 10, 1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3,
            4, 3, 4, 4, 5, 2, 3, 3, 4, 3, 4, 4, 5, 3, 4,
            4, 5, 4, 5, 5, 6, 2, 3, 3, 4, 3, 4, 4, 5, 3,
            4, 4, 5, 4, 5, 5, 6, 3, 4, 4, 5, 4, 5, 5, 6,
            4, 5, 5, 6, 5, 6, 6, 7, 2, 3, 3, 4, 3, 4, 4,
            5, 3, 4, 4, 5, 4, 5, 5, 6, 3, 4, 4, 5, 4, 5,
            5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 3, 4, 4, 5, 4,
            5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 4, 5, 5, 6,
            5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 2, 3, 3,
            4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 3, 4,
            4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 3,
            4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
            4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7,
            8, 3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6,
            6, 7, 4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6,
            7, 7, 8, 4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7,
            6, 7, 7, 8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7,
            8, 7, 8, 8, 9, 2, 3, 3, 4, 3, 4, 4, 5, 3, 4,
            4, 5, 4, 5, 5, 6, 3, 4, 4, 5, 4, 5, 5, 6, 4,
            5, 5, 6, 5, 6, 6, 7, 3, 4, 4, 5, 4, 5, 5, 6,
            4, 5, 5, 6, 5, 6, 6, 7, 4, 5, 5, 6, 5, 6, 6,
            7, 5, 6, 6, 7, 6, 7, 7, 8, 3, 4, 4, 5, 4, 5,
            5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 4, 5, 5, 6, 5,
            6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 4, 5, 5, 6,
            5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 5, 6, 6,
            7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 3, 4,
            4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 4,
            5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8,
            4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7,
            8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8,
            8, 9, 4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6,
            7, 7, 8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8,
            7, 8, 8, 9, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7,
            8, 7, 8, 8, 9, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8,
            8, 9, 8, 9, 9, 10, 2, 3, 3, 4, 3, 4, 4, 5, 3,
            4, 4, 5, 4, 5, 5, 6, 3, 4, 4, 5, 4, 5, 5, 6,
            4, 5, 5, 6, 5, 6, 6, 7, 3, 4, 4, 5, 4, 5, 5,
            6, 4, 5, 5, 6, 5, 6, 6, 7, 4, 5, 5, 6, 5, 6,
            6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 3, 4, 4, 5, 4,
            5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 4, 5, 5, 6,
            5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 4, 5, 5,
            6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 5, 6,
            6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 3,
            4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
            4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7,
            8, 4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7,
            7, 8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7,
            8, 8, 9, 4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7,
            6, 7, 7, 8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7,
            8, 7, 8, 8, 9, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7,
            7, 8, 7, 8, 8, 9, 6, 7, 7, 8, 7, 8, 8, 9, 7,
            8, 8, 9, 8, 9, 9, 10, 3, 4, 4, 5, 4, 5, 5, 6,
            4, 5, 5, 6, 5, 6, 6, 7, 4, 5, 5, 6, 5, 6, 6,
            7, 5, 6, 6, 7, 6, 7, 7, 8, 4, 5, 5, 6, 5, 6,
            6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 5, 6, 6, 7, 6,
            7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 4, 5, 5, 6,
            5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 5, 6, 6,
            7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 5, 6,
            6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 6,
            7, 7, 8, 7, 8, 8, 9, 7, 8, 8, 9, 8, 9, 9, 10,
            4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7,
            8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8,
            8, 9, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7,
            8, 8, 9, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8, 8, 9,
            8, 9, 9, 10, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7,
            8, 7, 8, 8, 9, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8,
            8, 9, 8, 9, 9, 10, 6, 7, 7, 8, 7, 8, 8, 9, 7,
            8, 8, 9, 8, 9, 9, 10, 7, 8, 8, 9, 8, 9, 9, 10,
            8, 9, 9, 10, 9, 10, 10, 11, 1, 2, 2, 3, 2, 3, 3,
            4, 2, 3, 3, 4, 3, 4, 4, 5, 2, 3, 3, 4, 3, 4,
            4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 2, 3, 3, 4, 3,
            4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 3, 4, 4, 5,
            4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 2, 3, 3,
            4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 3, 4,
            4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 3,
            4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
            4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7,
            8, 2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5,
            5, 6, 3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5,
            6, 6, 7, 3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6,
            5, 6, 6, 7, 4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6,
            7, 6, 7, 7, 8, 3, 4, 4, 5, 4, 5, 5, 6, 4, 5,
            5, 6, 5, 6, 6, 7, 4, 5, 5, 6, 5, 6, 6, 7, 5,
            6, 6, 7, 6, 7, 7, 8, 4, 5, 5, 6, 5, 6, 6, 7,
            5, 6, 6, 7, 6, 7, 7, 8, 5, 6, 6, 7, 6, 7, 7,
            8, 6, 7, 7, 8, 7, 8, 8, 9, 2, 3, 3, 4, 3, 4,
            4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 3, 4, 4, 5, 4,
            5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 3, 4, 4, 5,
            4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 4, 5, 5,
            6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 3, 4,
            4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 4,
            5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8,
            4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7,
            8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8,
            8, 9, 3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5,
            6, 6, 7, 4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7,
            6, 7, 7, 8, 4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6,
            7, 6, 7, 7, 8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7,
            7, 8, 7, 8, 8, 9, 4, 5, 5, 6, 5, 6, 6, 7, 5,
            6, 6, 7, 6, 7, 7, 8, 5, 6, 6, 7, 6, 7, 7, 8,
            6, 7, 7, 8, 7, 8, 8, 9, 5, 6, 6, 7, 6, 7, 7,
            8, 6, 7, 7, 8, 7, 8, 8, 9, 6, 7, 7, 8, 7, 8,
            8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 2, 3, 3, 4, 3,
            4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 3, 4, 4, 5,
            4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 3, 4, 4,
            5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 4, 5,
            5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 3,
            4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
            4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7,
            8, 4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7,
            7, 8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7,
            8, 8, 9, 3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6,
            5, 6, 6, 7, 4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6,
            7, 6, 7, 7, 8, 4, 5, 5, 6, 5, 6, 6, 7, 5, 6,
            6, 7, 6, 7, 7, 8, 5, 6, 6, 7, 6, 7, 7, 8, 6,
            7, 7, 8, 7, 8, 8, 9, 4, 5, 5, 6, 5, 6, 6, 7,
            5, 6, 6, 7, 6, 7, 7, 8, 5, 6, 6, 7, 6, 7, 7,
            8, 6, 7, 7, 8, 7, 8, 8, 9, 5, 6, 6, 7, 6, 7,
            7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 6, 7, 7, 8, 7,
            8, 8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 3, 4, 4, 5,
            4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 4, 5, 5,
            6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 4, 5,
            5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 5,
            6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9,
            4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7,
            8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8,
            8, 9, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7,
            8, 8, 9, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8, 8, 9,
            8, 9, 9, 10, 4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6,
            7, 6, 7, 7, 8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7,
            7, 8, 7, 8, 8, 9, 5, 6, 6, 7, 6, 7, 7, 8, 6,
            7, 7, 8, 7, 8, 8, 9, 6, 7, 7, 8, 7, 8, 8, 9,
            7, 8, 8, 9, 8, 9, 9, 10, 5, 6, 6, 7, 6, 7, 7,
            8, 6, 7, 7, 8, 7, 8, 8, 9, 6, 7, 7, 8, 7, 8,
            8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 6, 7, 7, 8, 7,
            8, 8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 7, 8, 8, 9,
            8, 9, 9, 10, 8, 9, 9, 10, 9, 10, 10, 11, 2, 3, 3,
            4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 3, 4,
            4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 3,
            4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
            4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7,
            8, 3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6,
            6, 7, 4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6,
            7, 7, 8, 4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7,
            6, 7, 7, 8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7,
            8, 7, 8, 8, 9, 3, 4, 4, 5, 4, 5, 5, 6, 4, 5,
            5, 6, 5, 6, 6, 7, 4, 5, 5, 6, 5, 6, 6, 7, 5,
            6, 6, 7, 6, 7, 7, 8, 4, 5, 5, 6, 5, 6, 6, 7,
            5, 6, 6, 7, 6, 7, 7, 8, 5, 6, 6, 7, 6, 7, 7,
            8, 6, 7, 7, 8, 7, 8, 8, 9, 4, 5, 5, 6, 5, 6,
            6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 5, 6, 6, 7, 6,
            7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 5, 6, 6, 7,
            6, 7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 6, 7, 7,
            8, 7, 8, 8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 3, 4,
            4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 4,
            5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8,
            4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7,
            8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8,
            8, 9, 4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6,
            7, 7, 8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8,
            7, 8, 8, 9, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7,
            8, 7, 8, 8, 9, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8,
            8, 9, 8, 9, 9, 10, 4, 5, 5, 6, 5, 6, 6, 7, 5,
            6, 6, 7, 6, 7, 7, 8, 5, 6, 6, 7, 6, 7, 7, 8,
            6, 7, 7, 8, 7, 8, 8, 9, 5, 6, 6, 7, 6, 7, 7,
            8, 6, 7, 7, 8, 7, 8, 8, 9, 6, 7, 7, 8, 7, 8,
            8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 5, 6, 6, 7, 6,
            7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 6, 7, 7, 8,
            7, 8, 8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 6, 7, 7,
            8, 7, 8, 8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 7, 8,
            8, 9, 8, 9, 9, 10, 8, 9, 9, 10, 9, 10, 10, 11, 3,
            4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
            4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7,
            8, 4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7,
            7, 8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7,
            8, 8, 9, 4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7,
            6, 7, 7, 8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7,
            8, 7, 8, 8, 9, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7,
            7, 8, 7, 8, 8, 9, 6, 7, 7, 8, 7, 8, 8, 9, 7,
            8, 8, 9, 8, 9, 9, 10, 4, 5, 5, 6, 5, 6, 6, 7,
            5, 6, 6, 7, 6, 7, 7, 8, 5, 6, 6, 7, 6, 7, 7,
            8, 6, 7, 7, 8, 7, 8, 8, 9, 5, 6, 6, 7, 6, 7,
            7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 6, 7, 7, 8, 7,
            8, 8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 5, 6, 6, 7,
            6, 7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 6, 7, 7,
            8, 7, 8, 8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 6, 7,
            7, 8, 7, 8, 8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 7,
            8, 8, 9, 8, 9, 9, 10, 8, 9, 9, 10, 9, 10, 10, 11,
            4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7,
            8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8,
            8, 9, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7,
            8, 8, 9, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8, 8, 9,
            8, 9, 9, 10, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7,
            8, 7, 8, 8, 9, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8,
            8, 9, 8, 9, 9, 10, 6, 7, 7, 8, 7, 8, 8, 9, 7,
            8, 8, 9, 8, 9, 9, 10, 7, 8, 8, 9, 8, 9, 9, 10,
            8, 9, 9, 10, 9, 10, 10, 11, 5, 6, 6, 7, 6, 7, 7,
            8, 6, 7, 7, 8, 7, 8, 8, 9, 6, 7, 7, 8, 7, 8,
            8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 6, 7, 7, 8, 7,
            8, 8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 7, 8, 8, 9,
            8, 9, 9, 10, 8, 9, 9, 10, 9, 10, 10, 11, 6, 7, 7,
            8, 7, 8, 8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 7, 8,
            8, 9, 8, 9, 9, 10, 8, 9, 9, 10, 9, 10, 10, 11, 7,
            8, 8, 9, 8, 9, 9, 10, 8, 9, 9, 10, 9, 10, 10, 11,
            8, 9, 9, 10, 9, 10, 10, 11, 9, 10, 10, 11, 10, 11, 11,
            12, 1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4,
            4, 5, 2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4,
            5, 5, 6, 2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5,
            4, 5, 5, 6, 3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5,
            6, 5, 6, 6, 7, 2, 3, 3, 4, 3, 4, 4, 5, 3, 4,
            4, 5, 4, 5, 5, 6, 3, 4, 4, 5, 4, 5, 5, 6, 4,
            5, 5, 6, 5, 6, 6, 7, 3, 4, 4, 5, 4, 5, 5, 6,
            4, 5, 5, 6, 5, 6, 6, 7, 4, 5, 5, 6, 5, 6, 6,
            7, 5, 6, 6, 7, 6, 7, 7, 8, 2, 3, 3, 4, 3, 4,
            4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 3, 4, 4, 5, 4,
            5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 3, 4, 4, 5,
            4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 4, 5, 5,
            6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 3, 4,
            4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 4,
            5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8,
            4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7,
            8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8,
            8, 9, 2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4,
            5, 5, 6, 3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6,
            5, 6, 6, 7, 3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5,
            6, 5, 6, 6, 7, 4, 5, 5, 6, 5, 6, 6, 7, 5, 6,
            6, 7, 6, 7, 7, 8, 3, 4, 4, 5, 4, 5, 5, 6, 4,
            5, 5, 6, 5, 6, 6, 7, 4, 5, 5, 6, 5, 6, 6, 7,
            5, 6, 6, 7, 6, 7, 7, 8, 4, 5, 5, 6, 5, 6, 6,
            7, 5, 6, 6, 7, 6, 7, 7, 8, 5, 6, 6, 7, 6, 7,
            7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 3, 4, 4, 5, 4,
            5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 4, 5, 5, 6,
            5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 4, 5, 5,
            6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 5, 6,
            6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 4,
            5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8,
            5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8, 8,
            9, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8,
            8, 9, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8, 8, 9, 8,
            9, 9, 10, 2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5,
            4, 5, 5, 6, 3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5,
            6, 5, 6, 6, 7, 3, 4, 4, 5, 4, 5, 5, 6, 4, 5,
            5, 6, 5, 6, 6, 7, 4, 5, 5, 6, 5, 6, 6, 7, 5,
            6, 6, 7, 6, 7, 7, 8, 3, 4, 4, 5, 4, 5, 5, 6,
            4, 5, 5, 6, 5, 6, 6, 7, 4, 5, 5, 6, 5, 6, 6,
            7, 5, 6, 6, 7, 6, 7, 7, 8, 4, 5, 5, 6, 5, 6,
            6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 5, 6, 6, 7, 6,
            7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 3, 4, 4, 5,
            4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 4, 5, 5,
            6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 4, 5,
            5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 5,
            6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9,
            4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7,
            8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8,
            8, 9, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7,
            8, 8, 9, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8, 8, 9,
            8, 9, 9, 10, 3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5,
            6, 5, 6, 6, 7, 4, 5, 5, 6, 5, 6, 6, 7, 5, 6,
            6, 7, 6, 7, 7, 8, 4, 5, 5, 6, 5, 6, 6, 7, 5,
            6, 6, 7, 6, 7, 7, 8, 5, 6, 6, 7, 6, 7, 7, 8,
            6, 7, 7, 8, 7, 8, 8, 9, 4, 5, 5, 6, 5, 6, 6,
            7, 5, 6, 6, 7, 6, 7, 7, 8, 5, 6, 6, 7, 6, 7,
            7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 5, 6, 6, 7, 6,
            7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 6, 7, 7, 8,
            7, 8, 8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 4, 5, 5,
            6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 5, 6,
            6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 5,
            6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9,
            6, 7, 7, 8, 7, 8, 8, 9, 7, 8, 8, 9, 8, 9, 9,
            10, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8,
            8, 9, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8, 8, 9, 8,
            9, 9, 10, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8, 8, 9,
            8, 9, 9, 10, 7, 8, 8, 9, 8, 9, 9, 10, 8, 9, 9,
            10, 9, 10, 10, 11, 2, 3, 3, 4, 3, 4, 4, 5, 3, 4,
            4, 5, 4, 5, 5, 6, 3, 4, 4, 5, 4, 5, 5, 6, 4,
            5, 5, 6, 5, 6, 6, 7, 3, 4, 4, 5, 4, 5, 5, 6,
            4, 5, 5, 6, 5, 6, 6, 7, 4, 5, 5, 6, 5, 6, 6,
            7, 5, 6, 6, 7, 6, 7, 7, 8, 3, 4, 4, 5, 4, 5,
            5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 4, 5, 5, 6, 5,
            6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 4, 5, 5, 6,
            5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 5, 6, 6,
            7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 3, 4,
            4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 4,
            5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8,
            4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7,
            8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8,
            8, 9, 4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6,
            7, 7, 8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8,
            7, 8, 8, 9, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7,
            8, 7, 8, 8, 9, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8,
            8, 9, 8, 9, 9, 10, 3, 4, 4, 5, 4, 5, 5, 6, 4,
            5, 5, 6, 5, 6, 6, 7, 4, 5, 5, 6, 5, 6, 6, 7,
            5, 6, 6, 7, 6, 7, 7, 8, 4, 5, 5, 6, 5, 6, 6,
            7, 5, 6, 6, 7, 6, 7, 7, 8, 5, 6, 6, 7, 6, 7,
            7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 4, 5, 5, 6, 5,
            6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 5, 6, 6, 7,
            6, 7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 5, 6, 6,
            7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 6, 7,
            7, 8, 7, 8, 8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 4,
            5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8,
            5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8, 8,
            9, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8,
            8, 9, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8, 8, 9, 8,
            9, 9, 10, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8,
            7, 8, 8, 9, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8, 8,
            9, 8, 9, 9, 10, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8,
            8, 9, 8, 9, 9, 10, 7, 8, 8, 9, 8, 9, 9, 10, 8,
            9, 9, 10, 9, 10, 10, 11, 3, 4, 4, 5, 4, 5, 5, 6,
            4, 5, 5, 6, 5, 6, 6, 7, 4, 5, 5, 6, 5, 6, 6,
            7, 5, 6, 6, 7, 6, 7, 7, 8, 4, 5, 5, 6, 5, 6,
            6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 5, 6, 6, 7, 6,
            7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 4, 5, 5, 6,
            5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 5, 6, 6,
            7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 5, 6,
            6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 6,
            7, 7, 8, 7, 8, 8, 9, 7, 8, 8, 9, 8, 9, 9, 10,
            4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7,
            8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8,
            8, 9, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7,
            8, 8, 9, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8, 8, 9,
            8, 9, 9, 10, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7,
            8, 7, 8, 8, 9, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8,
            8, 9, 8, 9, 9, 10, 6, 7, 7, 8, 7, 8, 8, 9, 7,
            8, 8, 9, 8, 9, 9, 10, 7, 8, 8, 9, 8, 9, 9, 10,
            8, 9, 9, 10, 9, 10, 10, 11, 4, 5, 5, 6, 5, 6, 6,
            7, 5, 6, 6, 7, 6, 7, 7, 8, 5, 6, 6, 7, 6, 7,
            7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 5, 6, 6, 7, 6,
            7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 6, 7, 7, 8,
            7, 8, 8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 5, 6, 6,
            7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 6, 7,
            7, 8, 7, 8, 8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 6,
            7, 7, 8, 7, 8, 8, 9, 7, 8, 8, 9, 8, 9, 9, 10,
            7, 8, 8, 9, 8, 9, 9, 10, 8, 9, 9, 10, 9, 10, 10,
            11, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8,
            8, 9, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8, 8, 9, 8,
            9, 9, 10, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8, 8, 9,
            8, 9, 9, 10, 7, 8, 8, 9, 8, 9, 9, 10, 8, 9, 9,
            10, 9, 10, 10, 11, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8,
            8, 9, 8, 9, 9, 10, 7, 8, 8, 9, 8, 9, 9, 10, 8,
            9, 9, 10, 9, 10, 10, 11, 7, 8, 8, 9, 8, 9, 9, 10,
            8, 9, 9, 10, 9, 10, 10, 11, 8, 9, 9, 10, 9, 10, 10,
            11, 9, 10, 10, 11, 10, 11, 11, 12, 2, 3, 3, 4, 3, 4,
            4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 3, 4, 4, 5, 4,
            5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 3, 4, 4, 5,
            4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 4, 5, 5,
            6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 3, 4,
            4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 4,
            5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8,
            4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7,
            8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8,
            8, 9, 3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5,
            6, 6, 7, 4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7,
            6, 7, 7, 8, 4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6,
            7, 6, 7, 7, 8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7,
            7, 8, 7, 8, 8, 9, 4, 5, 5, 6, 5, 6, 6, 7, 5,
            6, 6, 7, 6, 7, 7, 8, 5, 6, 6, 7, 6, 7, 7, 8,
            6, 7, 7, 8, 7, 8, 8, 9, 5, 6, 6, 7, 6, 7, 7,
            8, 6, 7, 7, 8, 7, 8, 8, 9, 6, 7, 7, 8, 7, 8,
            8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 3, 4, 4, 5, 4,
            5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 4, 5, 5, 6,
            5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 4, 5, 5,
            6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 5, 6,
            6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 4,
            5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8,
            5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8, 8,
            9, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8,
            8, 9, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8, 8, 9, 8,
            9, 9, 10, 4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7,
            6, 7, 7, 8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7,
            8, 7, 8, 8, 9, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7,
            7, 8, 7, 8, 8, 9, 6, 7, 7, 8, 7, 8, 8, 9, 7,
            8, 8, 9, 8, 9, 9, 10, 5, 6, 6, 7, 6, 7, 7, 8,
            6, 7, 7, 8, 7, 8, 8, 9, 6, 7, 7, 8, 7, 8, 8,
            9, 7, 8, 8, 9, 8, 9, 9, 10, 6, 7, 7, 8, 7, 8,
            8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 7, 8, 8, 9, 8,
            9, 9, 10, 8, 9, 9, 10, 9, 10, 10, 11, 3, 4, 4, 5,
            4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 4, 5, 5,
            6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 4, 5,
            5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 5,
            6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9,
            4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7,
            8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8,
            8, 9, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7,
            8, 8, 9, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8, 8, 9,
            8, 9, 9, 10, 4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6,
            7, 6, 7, 7, 8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7,
            7, 8, 7, 8, 8, 9, 5, 6, 6, 7, 6, 7, 7, 8, 6,
            7, 7, 8, 7, 8, 8, 9, 6, 7, 7, 8, 7, 8, 8, 9,
            7, 8, 8, 9, 8, 9, 9, 10, 5, 6, 6, 7, 6, 7, 7,
            8, 6, 7, 7, 8, 7, 8, 8, 9, 6, 7, 7, 8, 7, 8,
            8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 6, 7, 7, 8, 7,
            8, 8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 7, 8, 8, 9,
            8, 9, 9, 10, 8, 9, 9, 10, 9, 10, 10, 11, 4, 5, 5,
            6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8, 5, 6,
            6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 5,
            6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9,
            6, 7, 7, 8, 7, 8, 8, 9, 7, 8, 8, 9, 8, 9, 9,
            10, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8,
            8, 9, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8, 8, 9, 8,
            9, 9, 10, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8, 8, 9,
            8, 9, 9, 10, 7, 8, 8, 9, 8, 9, 9, 10, 8, 9, 9,
            10, 9, 10, 10, 11, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7,
            7, 8, 7, 8, 8, 9, 6, 7, 7, 8, 7, 8, 8, 9, 7,
            8, 8, 9, 8, 9, 9, 10, 6, 7, 7, 8, 7, 8, 8, 9,
            7, 8, 8, 9, 8, 9, 9, 10, 7, 8, 8, 9, 8, 9, 9,
            10, 8, 9, 9, 10, 9, 10, 10, 11, 6, 7, 7, 8, 7, 8,
            8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 7, 8, 8, 9, 8,
            9, 9, 10, 8, 9, 9, 10, 9, 10, 10, 11, 7, 8, 8, 9,
            8, 9, 9, 10, 8, 9, 9, 10, 9, 10, 10, 11, 8, 9, 9,
            10, 9, 10, 10, 11, 9, 10, 10, 11, 10, 11, 11, 12, 3, 4,
            4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 4,
            5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8,
            4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7,
            8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8,
            8, 9, 4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6,
            7, 7, 8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8,
            7, 8, 8, 9, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7,
            8, 7, 8, 8, 9, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8,
            8, 9, 8, 9, 9, 10, 4, 5, 5, 6, 5, 6, 6, 7, 5,
            6, 6, 7, 6, 7, 7, 8, 5, 6, 6, 7, 6, 7, 7, 8,
            6, 7, 7, 8, 7, 8, 8, 9, 5, 6, 6, 7, 6, 7, 7,
            8, 6, 7, 7, 8, 7, 8, 8, 9, 6, 7, 7, 8, 7, 8,
            8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 5, 6, 6, 7, 6,
            7, 7, 8, 6, 7, 7, 8, 7, 8, 8, 9, 6, 7, 7, 8,
            7, 8, 8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 6, 7, 7,
            8, 7, 8, 8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 7, 8,
            8, 9, 8, 9, 9, 10, 8, 9, 9, 10, 9, 10, 10, 11, 4,
            5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8,
            5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8, 8,
            9, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8,
            8, 9, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8, 8, 9, 8,
            9, 9, 10, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8,
            7, 8, 8, 9, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8, 8,
            9, 8, 9, 9, 10, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8,
            8, 9, 8, 9, 9, 10, 7, 8, 8, 9, 8, 9, 9, 10, 8,
            9, 9, 10, 9, 10, 10, 11, 5, 6, 6, 7, 6, 7, 7, 8,
            6, 7, 7, 8, 7, 8, 8, 9, 6, 7, 7, 8, 7, 8, 8,
            9, 7, 8, 8, 9, 8, 9, 9, 10, 6, 7, 7, 8, 7, 8,
            8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 7, 8, 8, 9, 8,
            9, 9, 10, 8, 9, 9, 10, 9, 10, 10, 11, 6, 7, 7, 8,
            7, 8, 8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 7, 8, 8,
            9, 8, 9, 9, 10, 8, 9, 9, 10, 9, 10, 10, 11, 7, 8,
            8, 9, 8, 9, 9, 10, 8, 9, 9, 10, 9, 10, 10, 11, 8,
            9, 9, 10, 9, 10, 10, 11, 9, 10, 10, 11, 10, 11, 11, 12,
            4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7,
            8, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8,
            8, 9, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7,
            8, 8, 9, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8, 8, 9,
            8, 9, 9, 10, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7,
            8, 7, 8, 8, 9, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8,
            8, 9, 8, 9, 9, 10, 6, 7, 7, 8, 7, 8, 8, 9, 7,
            8, 8, 9, 8, 9, 9, 10, 7, 8, 8, 9, 8, 9, 9, 10,
            8, 9, 9, 10, 9, 10, 10, 11, 5, 6, 6, 7, 6, 7, 7,
            8, 6, 7, 7, 8, 7, 8, 8, 9, 6, 7, 7, 8, 7, 8,
            8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 6, 7, 7, 8, 7,
            8, 8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 7, 8, 8, 9,
            8, 9, 9, 10, 8, 9, 9, 10, 9, 10, 10, 11, 6, 7, 7,
            8, 7, 8, 8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 7, 8,
            8, 9, 8, 9, 9, 10, 8, 9, 9, 10, 9, 10, 10, 11, 7,
            8, 8, 9, 8, 9, 9, 10, 8, 9, 9, 10, 9, 10, 10, 11,
            8, 9, 9, 10, 9, 10, 10, 11, 9, 10, 10, 11, 10, 11, 11,
            12, 5, 6, 6, 7, 6, 7, 7, 8, 6, 7, 7, 8, 7, 8,
            8, 9, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8, 8, 9, 8,
            9, 9, 10, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8, 8, 9,
            8, 9, 9, 10, 7, 8, 8, 9, 8, 9, 9, 10, 8, 9, 9,
            10, 9, 10, 10, 11, 6, 7, 7, 8, 7, 8, 8, 9, 7, 8,
            8, 9, 8, 9, 9, 10, 7, 8, 8, 9, 8, 9, 9, 10, 8,
            9, 9, 10, 9, 10, 10, 11, 7, 8, 8, 9, 8, 9, 9, 10,
            8, 9, 9, 10, 9, 10, 10, 11, 8, 9, 9, 10, 9, 10, 10,
            11, 9, 10, 10, 11, 10, 11, 11, 12, 6, 7, 7, 8, 7, 8,
            8, 9, 7, 8, 8, 9, 8, 9, 9, 10, 7, 8, 8, 9, 8,
            9, 9, 10, 8, 9, 9, 10, 9, 10, 10, 11, 7, 8, 8, 9,
            8, 9, 9, 10, 8, 9, 9, 10, 9, 10, 10, 11, 8, 9, 9,
            10, 9, 10, 10, 11, 9, 10, 10, 11, 10, 11, 11, 12, 7, 8,
            8, 9, 8, 9, 9, 10, 8, 9, 9, 10, 9, 10, 10, 11, 8,
            9, 9, 10, 9, 10, 10, 11, 9, 10, 10, 11, 10, 11, 11, 12,
            8, 9, 9, 10, 9, 10, 10, 11, 9, 10, 10, 11, 10, 11, 11,
            12, 9, 10, 10, 11, 10, 11, 11, 12, 10, 11, 11, 12, 11, 12,
            12, 13
		};
        #endregion

        #region StraightTable
        // This table returns a straights starting card (0 if not a straight)
        /// <exclude/>
        private static readonly System.UInt16[] straightTable = 
		{ 
			0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x6, 0x6, 0x6, 0x6, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x6, 0x6, 0x6, 0x6, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x8, 0x8, 0x8, 0x8, 
            0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x6, 0x6, 0x6, 0x6, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x6, 0x6, 0x6, 0x6, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x6, 0x6, 0x6, 0x6, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x6, 0x6, 0x6, 0x6, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x6, 0x6, 0x6, 0x6, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x6, 0x6, 0x6, 0x6, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x5, 0x5, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x6, 0x6, 0x6, 0x6, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x7, 0x7, 0x7, 0x7, 
            0x7, 0x7, 0x7, 0x7, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x6, 0x6, 0x6, 0x6, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x6, 0x6, 0x6, 0x6, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x6, 0x6, 0x6, 0x6, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x6, 0x6, 0x6, 0x6, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x6, 0x6, 0x6, 0x6, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 
            0x8, 0x8, 0x8, 0x8, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x6, 0x6, 0x6, 0x6, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x6, 0x6, 0x6, 0x6, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x6, 0x6, 0x6, 0x6, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x6, 0x6, 0x6, 0x6, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 
            0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x6, 0x6, 0x6, 0x6, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x7, 0x7, 0x7, 0x7, 
            0x7, 0x7, 0x7, 0x7, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x6, 0x6, 0x6, 0x6, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x6, 0x6, 0x6, 0x6, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x6, 0x6, 0x6, 0x6, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x8, 0x8, 0x8, 0x8, 
            0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x6, 0x6, 0x6, 0x6, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x6, 0x6, 0x6, 0x6, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x6, 0x6, 0x6, 0x6, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x6, 0x6, 0x6, 0x6, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 
            0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x6, 0x6, 0x6, 0x6, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x6, 0x6, 0x6, 0x6, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x6, 0x6, 0x6, 0x6, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x7, 0x7, 0x7, 0x7, 
            0x7, 0x7, 0x7, 0x7, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x6, 0x6, 0x6, 0x6, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x3, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x6, 0x6, 0x6, 0x6, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x4, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x5, 0x5, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 
            0x0, 0x0, 0x0, 0x4, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x3, 
            0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc
		};
        #endregion

        #region Top Five Card Table
        /// <exclude/>
        private static readonly System.UInt32[] topFiveCardsTable = 
		{ 
			0x0, 0x0, 0x10000, 0x10000, 0x20000, 0x20000, 0x21000, 0x21000, 0x30000, 0x30000, 
            0x31000, 0x31000, 0x32000, 0x32000, 0x32100, 0x32100, 0x40000, 0x40000, 0x41000, 0x41000, 
            0x42000, 0x42000, 0x42100, 0x42100, 0x43000, 0x43000, 0x43100, 0x43100, 0x43200, 0x43200, 
            0x43210, 0x43210, 0x50000, 0x50000, 0x51000, 0x51000, 0x52000, 0x52000, 0x52100, 0x52100, 
            0x53000, 0x53000, 0x53100, 0x53100, 0x53200, 0x53200, 0x53210, 0x53210, 0x54000, 0x54000, 
            0x54100, 0x54100, 0x54200, 0x54200, 0x54210, 0x54210, 0x54300, 0x54300, 0x54310, 0x54310, 
            0x54320, 0x54320, 0x54321, 0x54321, 0x60000, 0x60000, 0x61000, 0x61000, 0x62000, 0x62000, 
            0x62100, 0x62100, 0x63000, 0x63000, 0x63100, 0x63100, 0x63200, 0x63200, 0x63210, 0x63210, 
            0x64000, 0x64000, 0x64100, 0x64100, 0x64200, 0x64200, 0x64210, 0x64210, 0x64300, 0x64300, 
            0x64310, 0x64310, 0x64320, 0x64320, 0x64321, 0x64321, 0x65000, 0x65000, 0x65100, 0x65100, 
            0x65200, 0x65200, 0x65210, 0x65210, 0x65300, 0x65300, 0x65310, 0x65310, 0x65320, 0x65320, 
            0x65321, 0x65321, 0x65400, 0x65400, 0x65410, 0x65410, 0x65420, 0x65420, 0x65421, 0x65421, 
            0x65430, 0x65430, 0x65431, 0x65431, 0x65432, 0x65432, 0x65432, 0x65432, 0x70000, 0x70000, 
            0x71000, 0x71000, 0x72000, 0x72000, 0x72100, 0x72100, 0x73000, 0x73000, 0x73100, 0x73100, 
            0x73200, 0x73200, 0x73210, 0x73210, 0x74000, 0x74000, 0x74100, 0x74100, 0x74200, 0x74200, 
            0x74210, 0x74210, 0x74300, 0x74300, 0x74310, 0x74310, 0x74320, 0x74320, 0x74321, 0x74321, 
            0x75000, 0x75000, 0x75100, 0x75100, 0x75200, 0x75200, 0x75210, 0x75210, 0x75300, 0x75300, 
            0x75310, 0x75310, 0x75320, 0x75320, 0x75321, 0x75321, 0x75400, 0x75400, 0x75410, 0x75410, 
            0x75420, 0x75420, 0x75421, 0x75421, 0x75430, 0x75430, 0x75431, 0x75431, 0x75432, 0x75432, 
            0x75432, 0x75432, 0x76000, 0x76000, 0x76100, 0x76100, 0x76200, 0x76200, 0x76210, 0x76210, 
            0x76300, 0x76300, 0x76310, 0x76310, 0x76320, 0x76320, 0x76321, 0x76321, 0x76400, 0x76400, 
            0x76410, 0x76410, 0x76420, 0x76420, 0x76421, 0x76421, 0x76430, 0x76430, 0x76431, 0x76431, 
            0x76432, 0x76432, 0x76432, 0x76432, 0x76500, 0x76500, 0x76510, 0x76510, 0x76520, 0x76520, 
            0x76521, 0x76521, 0x76530, 0x76530, 0x76531, 0x76531, 0x76532, 0x76532, 0x76532, 0x76532, 
            0x76540, 0x76540, 0x76541, 0x76541, 0x76542, 0x76542, 0x76542, 0x76542, 0x76543, 0x76543, 
            0x76543, 0x76543, 0x76543, 0x76543, 0x76543, 0x76543, 0x80000, 0x80000, 0x81000, 0x81000, 
            0x82000, 0x82000, 0x82100, 0x82100, 0x83000, 0x83000, 0x83100, 0x83100, 0x83200, 0x83200, 
            0x83210, 0x83210, 0x84000, 0x84000, 0x84100, 0x84100, 0x84200, 0x84200, 0x84210, 0x84210, 
            0x84300, 0x84300, 0x84310, 0x84310, 0x84320, 0x84320, 0x84321, 0x84321, 0x85000, 0x85000, 
            0x85100, 0x85100, 0x85200, 0x85200, 0x85210, 0x85210, 0x85300, 0x85300, 0x85310, 0x85310, 
            0x85320, 0x85320, 0x85321, 0x85321, 0x85400, 0x85400, 0x85410, 0x85410, 0x85420, 0x85420, 
            0x85421, 0x85421, 0x85430, 0x85430, 0x85431, 0x85431, 0x85432, 0x85432, 0x85432, 0x85432, 
            0x86000, 0x86000, 0x86100, 0x86100, 0x86200, 0x86200, 0x86210, 0x86210, 0x86300, 0x86300, 
            0x86310, 0x86310, 0x86320, 0x86320, 0x86321, 0x86321, 0x86400, 0x86400, 0x86410, 0x86410, 
            0x86420, 0x86420, 0x86421, 0x86421, 0x86430, 0x86430, 0x86431, 0x86431, 0x86432, 0x86432, 
            0x86432, 0x86432, 0x86500, 0x86500, 0x86510, 0x86510, 0x86520, 0x86520, 0x86521, 0x86521, 
            0x86530, 0x86530, 0x86531, 0x86531, 0x86532, 0x86532, 0x86532, 0x86532, 0x86540, 0x86540, 
            0x86541, 0x86541, 0x86542, 0x86542, 0x86542, 0x86542, 0x86543, 0x86543, 0x86543, 0x86543, 
            0x86543, 0x86543, 0x86543, 0x86543, 0x87000, 0x87000, 0x87100, 0x87100, 0x87200, 0x87200, 
            0x87210, 0x87210, 0x87300, 0x87300, 0x87310, 0x87310, 0x87320, 0x87320, 0x87321, 0x87321, 
            0x87400, 0x87400, 0x87410, 0x87410, 0x87420, 0x87420, 0x87421, 0x87421, 0x87430, 0x87430, 
            0x87431, 0x87431, 0x87432, 0x87432, 0x87432, 0x87432, 0x87500, 0x87500, 0x87510, 0x87510, 
            0x87520, 0x87520, 0x87521, 0x87521, 0x87530, 0x87530, 0x87531, 0x87531, 0x87532, 0x87532, 
            0x87532, 0x87532, 0x87540, 0x87540, 0x87541, 0x87541, 0x87542, 0x87542, 0x87542, 0x87542, 
            0x87543, 0x87543, 0x87543, 0x87543, 0x87543, 0x87543, 0x87543, 0x87543, 0x87600, 0x87600, 
            0x87610, 0x87610, 0x87620, 0x87620, 0x87621, 0x87621, 0x87630, 0x87630, 0x87631, 0x87631, 
            0x87632, 0x87632, 0x87632, 0x87632, 0x87640, 0x87640, 0x87641, 0x87641, 0x87642, 0x87642, 
            0x87642, 0x87642, 0x87643, 0x87643, 0x87643, 0x87643, 0x87643, 0x87643, 0x87643, 0x87643, 
            0x87650, 0x87650, 0x87651, 0x87651, 0x87652, 0x87652, 0x87652, 0x87652, 0x87653, 0x87653, 
            0x87653, 0x87653, 0x87653, 0x87653, 0x87653, 0x87653, 0x87654, 0x87654, 0x87654, 0x87654, 
            0x87654, 0x87654, 0x87654, 0x87654, 0x87654, 0x87654, 0x87654, 0x87654, 0x87654, 0x87654, 
            0x87654, 0x87654, 0x90000, 0x90000, 0x91000, 0x91000, 0x92000, 0x92000, 0x92100, 0x92100, 
            0x93000, 0x93000, 0x93100, 0x93100, 0x93200, 0x93200, 0x93210, 0x93210, 0x94000, 0x94000, 
            0x94100, 0x94100, 0x94200, 0x94200, 0x94210, 0x94210, 0x94300, 0x94300, 0x94310, 0x94310, 
            0x94320, 0x94320, 0x94321, 0x94321, 0x95000, 0x95000, 0x95100, 0x95100, 0x95200, 0x95200, 
            0x95210, 0x95210, 0x95300, 0x95300, 0x95310, 0x95310, 0x95320, 0x95320, 0x95321, 0x95321, 
            0x95400, 0x95400, 0x95410, 0x95410, 0x95420, 0x95420, 0x95421, 0x95421, 0x95430, 0x95430, 
            0x95431, 0x95431, 0x95432, 0x95432, 0x95432, 0x95432, 0x96000, 0x96000, 0x96100, 0x96100, 
            0x96200, 0x96200, 0x96210, 0x96210, 0x96300, 0x96300, 0x96310, 0x96310, 0x96320, 0x96320, 
            0x96321, 0x96321, 0x96400, 0x96400, 0x96410, 0x96410, 0x96420, 0x96420, 0x96421, 0x96421, 
            0x96430, 0x96430, 0x96431, 0x96431, 0x96432, 0x96432, 0x96432, 0x96432, 0x96500, 0x96500, 
            0x96510, 0x96510, 0x96520, 0x96520, 0x96521, 0x96521, 0x96530, 0x96530, 0x96531, 0x96531, 
            0x96532, 0x96532, 0x96532, 0x96532, 0x96540, 0x96540, 0x96541, 0x96541, 0x96542, 0x96542, 
            0x96542, 0x96542, 0x96543, 0x96543, 0x96543, 0x96543, 0x96543, 0x96543, 0x96543, 0x96543, 
            0x97000, 0x97000, 0x97100, 0x97100, 0x97200, 0x97200, 0x97210, 0x97210, 0x97300, 0x97300, 
            0x97310, 0x97310, 0x97320, 0x97320, 0x97321, 0x97321, 0x97400, 0x97400, 0x97410, 0x97410, 
            0x97420, 0x97420, 0x97421, 0x97421, 0x97430, 0x97430, 0x97431, 0x97431, 0x97432, 0x97432, 
            0x97432, 0x97432, 0x97500, 0x97500, 0x97510, 0x97510, 0x97520, 0x97520, 0x97521, 0x97521, 
            0x97530, 0x97530, 0x97531, 0x97531, 0x97532, 0x97532, 0x97532, 0x97532, 0x97540, 0x97540, 
            0x97541, 0x97541, 0x97542, 0x97542, 0x97542, 0x97542, 0x97543, 0x97543, 0x97543, 0x97543, 
            0x97543, 0x97543, 0x97543, 0x97543, 0x97600, 0x97600, 0x97610, 0x97610, 0x97620, 0x97620, 
            0x97621, 0x97621, 0x97630, 0x97630, 0x97631, 0x97631, 0x97632, 0x97632, 0x97632, 0x97632, 
            0x97640, 0x97640, 0x97641, 0x97641, 0x97642, 0x97642, 0x97642, 0x97642, 0x97643, 0x97643, 
            0x97643, 0x97643, 0x97643, 0x97643, 0x97643, 0x97643, 0x97650, 0x97650, 0x97651, 0x97651, 
            0x97652, 0x97652, 0x97652, 0x97652, 0x97653, 0x97653, 0x97653, 0x97653, 0x97653, 0x97653, 
            0x97653, 0x97653, 0x97654, 0x97654, 0x97654, 0x97654, 0x97654, 0x97654, 0x97654, 0x97654, 
            0x97654, 0x97654, 0x97654, 0x97654, 0x97654, 0x97654, 0x97654, 0x97654, 0x98000, 0x98000, 
            0x98100, 0x98100, 0x98200, 0x98200, 0x98210, 0x98210, 0x98300, 0x98300, 0x98310, 0x98310, 
            0x98320, 0x98320, 0x98321, 0x98321, 0x98400, 0x98400, 0x98410, 0x98410, 0x98420, 0x98420, 
            0x98421, 0x98421, 0x98430, 0x98430, 0x98431, 0x98431, 0x98432, 0x98432, 0x98432, 0x98432, 
            0x98500, 0x98500, 0x98510, 0x98510, 0x98520, 0x98520, 0x98521, 0x98521, 0x98530, 0x98530, 
            0x98531, 0x98531, 0x98532, 0x98532, 0x98532, 0x98532, 0x98540, 0x98540, 0x98541, 0x98541, 
            0x98542, 0x98542, 0x98542, 0x98542, 0x98543, 0x98543, 0x98543, 0x98543, 0x98543, 0x98543, 
            0x98543, 0x98543, 0x98600, 0x98600, 0x98610, 0x98610, 0x98620, 0x98620, 0x98621, 0x98621, 
            0x98630, 0x98630, 0x98631, 0x98631, 0x98632, 0x98632, 0x98632, 0x98632, 0x98640, 0x98640, 
            0x98641, 0x98641, 0x98642, 0x98642, 0x98642, 0x98642, 0x98643, 0x98643, 0x98643, 0x98643, 
            0x98643, 0x98643, 0x98643, 0x98643, 0x98650, 0x98650, 0x98651, 0x98651, 0x98652, 0x98652, 
            0x98652, 0x98652, 0x98653, 0x98653, 0x98653, 0x98653, 0x98653, 0x98653, 0x98653, 0x98653, 
            0x98654, 0x98654, 0x98654, 0x98654, 0x98654, 0x98654, 0x98654, 0x98654, 0x98654, 0x98654, 
            0x98654, 0x98654, 0x98654, 0x98654, 0x98654, 0x98654, 0x98700, 0x98700, 0x98710, 0x98710, 
            0x98720, 0x98720, 0x98721, 0x98721, 0x98730, 0x98730, 0x98731, 0x98731, 0x98732, 0x98732, 
            0x98732, 0x98732, 0x98740, 0x98740, 0x98741, 0x98741, 0x98742, 0x98742, 0x98742, 0x98742, 
            0x98743, 0x98743, 0x98743, 0x98743, 0x98743, 0x98743, 0x98743, 0x98743, 0x98750, 0x98750, 
            0x98751, 0x98751, 0x98752, 0x98752, 0x98752, 0x98752, 0x98753, 0x98753, 0x98753, 0x98753, 
            0x98753, 0x98753, 0x98753, 0x98753, 0x98754, 0x98754, 0x98754, 0x98754, 0x98754, 0x98754, 
            0x98754, 0x98754, 0x98754, 0x98754, 0x98754, 0x98754, 0x98754, 0x98754, 0x98754, 0x98754, 
            0x98760, 0x98760, 0x98761, 0x98761, 0x98762, 0x98762, 0x98762, 0x98762, 0x98763, 0x98763, 
            0x98763, 0x98763, 0x98763, 0x98763, 0x98763, 0x98763, 0x98764, 0x98764, 0x98764, 0x98764, 
            0x98764, 0x98764, 0x98764, 0x98764, 0x98764, 0x98764, 0x98764, 0x98764, 0x98764, 0x98764, 
            0x98764, 0x98764, 0x98765, 0x98765, 0x98765, 0x98765, 0x98765, 0x98765, 0x98765, 0x98765, 
            0x98765, 0x98765, 0x98765, 0x98765, 0x98765, 0x98765, 0x98765, 0x98765, 0x98765, 0x98765, 
            0x98765, 0x98765, 0x98765, 0x98765, 0x98765, 0x98765, 0x98765, 0x98765, 0x98765, 0x98765, 
            0x98765, 0x98765, 0x98765, 0x98765, 0xa0000, 0xa0000, 0xa1000, 0xa1000, 0xa2000, 0xa2000, 
            0xa2100, 0xa2100, 0xa3000, 0xa3000, 0xa3100, 0xa3100, 0xa3200, 0xa3200, 0xa3210, 0xa3210, 
            0xa4000, 0xa4000, 0xa4100, 0xa4100, 0xa4200, 0xa4200, 0xa4210, 0xa4210, 0xa4300, 0xa4300, 
            0xa4310, 0xa4310, 0xa4320, 0xa4320, 0xa4321, 0xa4321, 0xa5000, 0xa5000, 0xa5100, 0xa5100, 
            0xa5200, 0xa5200, 0xa5210, 0xa5210, 0xa5300, 0xa5300, 0xa5310, 0xa5310, 0xa5320, 0xa5320, 
            0xa5321, 0xa5321, 0xa5400, 0xa5400, 0xa5410, 0xa5410, 0xa5420, 0xa5420, 0xa5421, 0xa5421, 
            0xa5430, 0xa5430, 0xa5431, 0xa5431, 0xa5432, 0xa5432, 0xa5432, 0xa5432, 0xa6000, 0xa6000, 
            0xa6100, 0xa6100, 0xa6200, 0xa6200, 0xa6210, 0xa6210, 0xa6300, 0xa6300, 0xa6310, 0xa6310, 
            0xa6320, 0xa6320, 0xa6321, 0xa6321, 0xa6400, 0xa6400, 0xa6410, 0xa6410, 0xa6420, 0xa6420, 
            0xa6421, 0xa6421, 0xa6430, 0xa6430, 0xa6431, 0xa6431, 0xa6432, 0xa6432, 0xa6432, 0xa6432, 
            0xa6500, 0xa6500, 0xa6510, 0xa6510, 0xa6520, 0xa6520, 0xa6521, 0xa6521, 0xa6530, 0xa6530, 
            0xa6531, 0xa6531, 0xa6532, 0xa6532, 0xa6532, 0xa6532, 0xa6540, 0xa6540, 0xa6541, 0xa6541, 
            0xa6542, 0xa6542, 0xa6542, 0xa6542, 0xa6543, 0xa6543, 0xa6543, 0xa6543, 0xa6543, 0xa6543, 
            0xa6543, 0xa6543, 0xa7000, 0xa7000, 0xa7100, 0xa7100, 0xa7200, 0xa7200, 0xa7210, 0xa7210, 
            0xa7300, 0xa7300, 0xa7310, 0xa7310, 0xa7320, 0xa7320, 0xa7321, 0xa7321, 0xa7400, 0xa7400, 
            0xa7410, 0xa7410, 0xa7420, 0xa7420, 0xa7421, 0xa7421, 0xa7430, 0xa7430, 0xa7431, 0xa7431, 
            0xa7432, 0xa7432, 0xa7432, 0xa7432, 0xa7500, 0xa7500, 0xa7510, 0xa7510, 0xa7520, 0xa7520, 
            0xa7521, 0xa7521, 0xa7530, 0xa7530, 0xa7531, 0xa7531, 0xa7532, 0xa7532, 0xa7532, 0xa7532, 
            0xa7540, 0xa7540, 0xa7541, 0xa7541, 0xa7542, 0xa7542, 0xa7542, 0xa7542, 0xa7543, 0xa7543, 
            0xa7543, 0xa7543, 0xa7543, 0xa7543, 0xa7543, 0xa7543, 0xa7600, 0xa7600, 0xa7610, 0xa7610, 
            0xa7620, 0xa7620, 0xa7621, 0xa7621, 0xa7630, 0xa7630, 0xa7631, 0xa7631, 0xa7632, 0xa7632, 
            0xa7632, 0xa7632, 0xa7640, 0xa7640, 0xa7641, 0xa7641, 0xa7642, 0xa7642, 0xa7642, 0xa7642, 
            0xa7643, 0xa7643, 0xa7643, 0xa7643, 0xa7643, 0xa7643, 0xa7643, 0xa7643, 0xa7650, 0xa7650, 
            0xa7651, 0xa7651, 0xa7652, 0xa7652, 0xa7652, 0xa7652, 0xa7653, 0xa7653, 0xa7653, 0xa7653, 
            0xa7653, 0xa7653, 0xa7653, 0xa7653, 0xa7654, 0xa7654, 0xa7654, 0xa7654, 0xa7654, 0xa7654, 
            0xa7654, 0xa7654, 0xa7654, 0xa7654, 0xa7654, 0xa7654, 0xa7654, 0xa7654, 0xa7654, 0xa7654, 
            0xa8000, 0xa8000, 0xa8100, 0xa8100, 0xa8200, 0xa8200, 0xa8210, 0xa8210, 0xa8300, 0xa8300, 
            0xa8310, 0xa8310, 0xa8320, 0xa8320, 0xa8321, 0xa8321, 0xa8400, 0xa8400, 0xa8410, 0xa8410, 
            0xa8420, 0xa8420, 0xa8421, 0xa8421, 0xa8430, 0xa8430, 0xa8431, 0xa8431, 0xa8432, 0xa8432, 
            0xa8432, 0xa8432, 0xa8500, 0xa8500, 0xa8510, 0xa8510, 0xa8520, 0xa8520, 0xa8521, 0xa8521, 
            0xa8530, 0xa8530, 0xa8531, 0xa8531, 0xa8532, 0xa8532, 0xa8532, 0xa8532, 0xa8540, 0xa8540, 
            0xa8541, 0xa8541, 0xa8542, 0xa8542, 0xa8542, 0xa8542, 0xa8543, 0xa8543, 0xa8543, 0xa8543, 
            0xa8543, 0xa8543, 0xa8543, 0xa8543, 0xa8600, 0xa8600, 0xa8610, 0xa8610, 0xa8620, 0xa8620, 
            0xa8621, 0xa8621, 0xa8630, 0xa8630, 0xa8631, 0xa8631, 0xa8632, 0xa8632, 0xa8632, 0xa8632, 
            0xa8640, 0xa8640, 0xa8641, 0xa8641, 0xa8642, 0xa8642, 0xa8642, 0xa8642, 0xa8643, 0xa8643, 
            0xa8643, 0xa8643, 0xa8643, 0xa8643, 0xa8643, 0xa8643, 0xa8650, 0xa8650, 0xa8651, 0xa8651, 
            0xa8652, 0xa8652, 0xa8652, 0xa8652, 0xa8653, 0xa8653, 0xa8653, 0xa8653, 0xa8653, 0xa8653, 
            0xa8653, 0xa8653, 0xa8654, 0xa8654, 0xa8654, 0xa8654, 0xa8654, 0xa8654, 0xa8654, 0xa8654, 
            0xa8654, 0xa8654, 0xa8654, 0xa8654, 0xa8654, 0xa8654, 0xa8654, 0xa8654, 0xa8700, 0xa8700, 
            0xa8710, 0xa8710, 0xa8720, 0xa8720, 0xa8721, 0xa8721, 0xa8730, 0xa8730, 0xa8731, 0xa8731, 
            0xa8732, 0xa8732, 0xa8732, 0xa8732, 0xa8740, 0xa8740, 0xa8741, 0xa8741, 0xa8742, 0xa8742, 
            0xa8742, 0xa8742, 0xa8743, 0xa8743, 0xa8743, 0xa8743, 0xa8743, 0xa8743, 0xa8743, 0xa8743, 
            0xa8750, 0xa8750, 0xa8751, 0xa8751, 0xa8752, 0xa8752, 0xa8752, 0xa8752, 0xa8753, 0xa8753, 
            0xa8753, 0xa8753, 0xa8753, 0xa8753, 0xa8753, 0xa8753, 0xa8754, 0xa8754, 0xa8754, 0xa8754, 
            0xa8754, 0xa8754, 0xa8754, 0xa8754, 0xa8754, 0xa8754, 0xa8754, 0xa8754, 0xa8754, 0xa8754, 
            0xa8754, 0xa8754, 0xa8760, 0xa8760, 0xa8761, 0xa8761, 0xa8762, 0xa8762, 0xa8762, 0xa8762, 
            0xa8763, 0xa8763, 0xa8763, 0xa8763, 0xa8763, 0xa8763, 0xa8763, 0xa8763, 0xa8764, 0xa8764, 
            0xa8764, 0xa8764, 0xa8764, 0xa8764, 0xa8764, 0xa8764, 0xa8764, 0xa8764, 0xa8764, 0xa8764, 
            0xa8764, 0xa8764, 0xa8764, 0xa8764, 0xa8765, 0xa8765, 0xa8765, 0xa8765, 0xa8765, 0xa8765, 
            0xa8765, 0xa8765, 0xa8765, 0xa8765, 0xa8765, 0xa8765, 0xa8765, 0xa8765, 0xa8765, 0xa8765, 
            0xa8765, 0xa8765, 0xa8765, 0xa8765, 0xa8765, 0xa8765, 0xa8765, 0xa8765, 0xa8765, 0xa8765, 
            0xa8765, 0xa8765, 0xa8765, 0xa8765, 0xa8765, 0xa8765, 0xa9000, 0xa9000, 0xa9100, 0xa9100, 
            0xa9200, 0xa9200, 0xa9210, 0xa9210, 0xa9300, 0xa9300, 0xa9310, 0xa9310, 0xa9320, 0xa9320, 
            0xa9321, 0xa9321, 0xa9400, 0xa9400, 0xa9410, 0xa9410, 0xa9420, 0xa9420, 0xa9421, 0xa9421, 
            0xa9430, 0xa9430, 0xa9431, 0xa9431, 0xa9432, 0xa9432, 0xa9432, 0xa9432, 0xa9500, 0xa9500, 
            0xa9510, 0xa9510, 0xa9520, 0xa9520, 0xa9521, 0xa9521, 0xa9530, 0xa9530, 0xa9531, 0xa9531, 
            0xa9532, 0xa9532, 0xa9532, 0xa9532, 0xa9540, 0xa9540, 0xa9541, 0xa9541, 0xa9542, 0xa9542, 
            0xa9542, 0xa9542, 0xa9543, 0xa9543, 0xa9543, 0xa9543, 0xa9543, 0xa9543, 0xa9543, 0xa9543, 
            0xa9600, 0xa9600, 0xa9610, 0xa9610, 0xa9620, 0xa9620, 0xa9621, 0xa9621, 0xa9630, 0xa9630, 
            0xa9631, 0xa9631, 0xa9632, 0xa9632, 0xa9632, 0xa9632, 0xa9640, 0xa9640, 0xa9641, 0xa9641, 
            0xa9642, 0xa9642, 0xa9642, 0xa9642, 0xa9643, 0xa9643, 0xa9643, 0xa9643, 0xa9643, 0xa9643, 
            0xa9643, 0xa9643, 0xa9650, 0xa9650, 0xa9651, 0xa9651, 0xa9652, 0xa9652, 0xa9652, 0xa9652, 
            0xa9653, 0xa9653, 0xa9653, 0xa9653, 0xa9653, 0xa9653, 0xa9653, 0xa9653, 0xa9654, 0xa9654, 
            0xa9654, 0xa9654, 0xa9654, 0xa9654, 0xa9654, 0xa9654, 0xa9654, 0xa9654, 0xa9654, 0xa9654, 
            0xa9654, 0xa9654, 0xa9654, 0xa9654, 0xa9700, 0xa9700, 0xa9710, 0xa9710, 0xa9720, 0xa9720, 
            0xa9721, 0xa9721, 0xa9730, 0xa9730, 0xa9731, 0xa9731, 0xa9732, 0xa9732, 0xa9732, 0xa9732, 
            0xa9740, 0xa9740, 0xa9741, 0xa9741, 0xa9742, 0xa9742, 0xa9742, 0xa9742, 0xa9743, 0xa9743, 
            0xa9743, 0xa9743, 0xa9743, 0xa9743, 0xa9743, 0xa9743, 0xa9750, 0xa9750, 0xa9751, 0xa9751, 
            0xa9752, 0xa9752, 0xa9752, 0xa9752, 0xa9753, 0xa9753, 0xa9753, 0xa9753, 0xa9753, 0xa9753, 
            0xa9753, 0xa9753, 0xa9754, 0xa9754, 0xa9754, 0xa9754, 0xa9754, 0xa9754, 0xa9754, 0xa9754, 
            0xa9754, 0xa9754, 0xa9754, 0xa9754, 0xa9754, 0xa9754, 0xa9754, 0xa9754, 0xa9760, 0xa9760, 
            0xa9761, 0xa9761, 0xa9762, 0xa9762, 0xa9762, 0xa9762, 0xa9763, 0xa9763, 0xa9763, 0xa9763, 
            0xa9763, 0xa9763, 0xa9763, 0xa9763, 0xa9764, 0xa9764, 0xa9764, 0xa9764, 0xa9764, 0xa9764, 
            0xa9764, 0xa9764, 0xa9764, 0xa9764, 0xa9764, 0xa9764, 0xa9764, 0xa9764, 0xa9764, 0xa9764, 
            0xa9765, 0xa9765, 0xa9765, 0xa9765, 0xa9765, 0xa9765, 0xa9765, 0xa9765, 0xa9765, 0xa9765, 
            0xa9765, 0xa9765, 0xa9765, 0xa9765, 0xa9765, 0xa9765, 0xa9765, 0xa9765, 0xa9765, 0xa9765, 
            0xa9765, 0xa9765, 0xa9765, 0xa9765, 0xa9765, 0xa9765, 0xa9765, 0xa9765, 0xa9765, 0xa9765, 
            0xa9765, 0xa9765, 0xa9800, 0xa9800, 0xa9810, 0xa9810, 0xa9820, 0xa9820, 0xa9821, 0xa9821, 
            0xa9830, 0xa9830, 0xa9831, 0xa9831, 0xa9832, 0xa9832, 0xa9832, 0xa9832, 0xa9840, 0xa9840, 
            0xa9841, 0xa9841, 0xa9842, 0xa9842, 0xa9842, 0xa9842, 0xa9843, 0xa9843, 0xa9843, 0xa9843, 
            0xa9843, 0xa9843, 0xa9843, 0xa9843, 0xa9850, 0xa9850, 0xa9851, 0xa9851, 0xa9852, 0xa9852, 
            0xa9852, 0xa9852, 0xa9853, 0xa9853, 0xa9853, 0xa9853, 0xa9853, 0xa9853, 0xa9853, 0xa9853, 
            0xa9854, 0xa9854, 0xa9854, 0xa9854, 0xa9854, 0xa9854, 0xa9854, 0xa9854, 0xa9854, 0xa9854, 
            0xa9854, 0xa9854, 0xa9854, 0xa9854, 0xa9854, 0xa9854, 0xa9860, 0xa9860, 0xa9861, 0xa9861, 
            0xa9862, 0xa9862, 0xa9862, 0xa9862, 0xa9863, 0xa9863, 0xa9863, 0xa9863, 0xa9863, 0xa9863, 
            0xa9863, 0xa9863, 0xa9864, 0xa9864, 0xa9864, 0xa9864, 0xa9864, 0xa9864, 0xa9864, 0xa9864, 
            0xa9864, 0xa9864, 0xa9864, 0xa9864, 0xa9864, 0xa9864, 0xa9864, 0xa9864, 0xa9865, 0xa9865, 
            0xa9865, 0xa9865, 0xa9865, 0xa9865, 0xa9865, 0xa9865, 0xa9865, 0xa9865, 0xa9865, 0xa9865, 
            0xa9865, 0xa9865, 0xa9865, 0xa9865, 0xa9865, 0xa9865, 0xa9865, 0xa9865, 0xa9865, 0xa9865, 
            0xa9865, 0xa9865, 0xa9865, 0xa9865, 0xa9865, 0xa9865, 0xa9865, 0xa9865, 0xa9865, 0xa9865, 
            0xa9870, 0xa9870, 0xa9871, 0xa9871, 0xa9872, 0xa9872, 0xa9872, 0xa9872, 0xa9873, 0xa9873, 
            0xa9873, 0xa9873, 0xa9873, 0xa9873, 0xa9873, 0xa9873, 0xa9874, 0xa9874, 0xa9874, 0xa9874, 
            0xa9874, 0xa9874, 0xa9874, 0xa9874, 0xa9874, 0xa9874, 0xa9874, 0xa9874, 0xa9874, 0xa9874, 
            0xa9874, 0xa9874, 0xa9875, 0xa9875, 0xa9875, 0xa9875, 0xa9875, 0xa9875, 0xa9875, 0xa9875, 
            0xa9875, 0xa9875, 0xa9875, 0xa9875, 0xa9875, 0xa9875, 0xa9875, 0xa9875, 0xa9875, 0xa9875, 
            0xa9875, 0xa9875, 0xa9875, 0xa9875, 0xa9875, 0xa9875, 0xa9875, 0xa9875, 0xa9875, 0xa9875, 
            0xa9875, 0xa9875, 0xa9875, 0xa9875, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 
            0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 
            0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 
            0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 
            0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 
            0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 
            0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xa9876, 0xb0000, 0xb0000, 
            0xb1000, 0xb1000, 0xb2000, 0xb2000, 0xb2100, 0xb2100, 0xb3000, 0xb3000, 0xb3100, 0xb3100, 
            0xb3200, 0xb3200, 0xb3210, 0xb3210, 0xb4000, 0xb4000, 0xb4100, 0xb4100, 0xb4200, 0xb4200, 
            0xb4210, 0xb4210, 0xb4300, 0xb4300, 0xb4310, 0xb4310, 0xb4320, 0xb4320, 0xb4321, 0xb4321, 
            0xb5000, 0xb5000, 0xb5100, 0xb5100, 0xb5200, 0xb5200, 0xb5210, 0xb5210, 0xb5300, 0xb5300, 
            0xb5310, 0xb5310, 0xb5320, 0xb5320, 0xb5321, 0xb5321, 0xb5400, 0xb5400, 0xb5410, 0xb5410, 
            0xb5420, 0xb5420, 0xb5421, 0xb5421, 0xb5430, 0xb5430, 0xb5431, 0xb5431, 0xb5432, 0xb5432, 
            0xb5432, 0xb5432, 0xb6000, 0xb6000, 0xb6100, 0xb6100, 0xb6200, 0xb6200, 0xb6210, 0xb6210, 
            0xb6300, 0xb6300, 0xb6310, 0xb6310, 0xb6320, 0xb6320, 0xb6321, 0xb6321, 0xb6400, 0xb6400, 
            0xb6410, 0xb6410, 0xb6420, 0xb6420, 0xb6421, 0xb6421, 0xb6430, 0xb6430, 0xb6431, 0xb6431, 
            0xb6432, 0xb6432, 0xb6432, 0xb6432, 0xb6500, 0xb6500, 0xb6510, 0xb6510, 0xb6520, 0xb6520, 
            0xb6521, 0xb6521, 0xb6530, 0xb6530, 0xb6531, 0xb6531, 0xb6532, 0xb6532, 0xb6532, 0xb6532, 
            0xb6540, 0xb6540, 0xb6541, 0xb6541, 0xb6542, 0xb6542, 0xb6542, 0xb6542, 0xb6543, 0xb6543, 
            0xb6543, 0xb6543, 0xb6543, 0xb6543, 0xb6543, 0xb6543, 0xb7000, 0xb7000, 0xb7100, 0xb7100, 
            0xb7200, 0xb7200, 0xb7210, 0xb7210, 0xb7300, 0xb7300, 0xb7310, 0xb7310, 0xb7320, 0xb7320, 
            0xb7321, 0xb7321, 0xb7400, 0xb7400, 0xb7410, 0xb7410, 0xb7420, 0xb7420, 0xb7421, 0xb7421, 
            0xb7430, 0xb7430, 0xb7431, 0xb7431, 0xb7432, 0xb7432, 0xb7432, 0xb7432, 0xb7500, 0xb7500, 
            0xb7510, 0xb7510, 0xb7520, 0xb7520, 0xb7521, 0xb7521, 0xb7530, 0xb7530, 0xb7531, 0xb7531, 
            0xb7532, 0xb7532, 0xb7532, 0xb7532, 0xb7540, 0xb7540, 0xb7541, 0xb7541, 0xb7542, 0xb7542, 
            0xb7542, 0xb7542, 0xb7543, 0xb7543, 0xb7543, 0xb7543, 0xb7543, 0xb7543, 0xb7543, 0xb7543, 
            0xb7600, 0xb7600, 0xb7610, 0xb7610, 0xb7620, 0xb7620, 0xb7621, 0xb7621, 0xb7630, 0xb7630, 
            0xb7631, 0xb7631, 0xb7632, 0xb7632, 0xb7632, 0xb7632, 0xb7640, 0xb7640, 0xb7641, 0xb7641, 
            0xb7642, 0xb7642, 0xb7642, 0xb7642, 0xb7643, 0xb7643, 0xb7643, 0xb7643, 0xb7643, 0xb7643, 
            0xb7643, 0xb7643, 0xb7650, 0xb7650, 0xb7651, 0xb7651, 0xb7652, 0xb7652, 0xb7652, 0xb7652, 
            0xb7653, 0xb7653, 0xb7653, 0xb7653, 0xb7653, 0xb7653, 0xb7653, 0xb7653, 0xb7654, 0xb7654, 
            0xb7654, 0xb7654, 0xb7654, 0xb7654, 0xb7654, 0xb7654, 0xb7654, 0xb7654, 0xb7654, 0xb7654, 
            0xb7654, 0xb7654, 0xb7654, 0xb7654, 0xb8000, 0xb8000, 0xb8100, 0xb8100, 0xb8200, 0xb8200, 
            0xb8210, 0xb8210, 0xb8300, 0xb8300, 0xb8310, 0xb8310, 0xb8320, 0xb8320, 0xb8321, 0xb8321, 
            0xb8400, 0xb8400, 0xb8410, 0xb8410, 0xb8420, 0xb8420, 0xb8421, 0xb8421, 0xb8430, 0xb8430, 
            0xb8431, 0xb8431, 0xb8432, 0xb8432, 0xb8432, 0xb8432, 0xb8500, 0xb8500, 0xb8510, 0xb8510, 
            0xb8520, 0xb8520, 0xb8521, 0xb8521, 0xb8530, 0xb8530, 0xb8531, 0xb8531, 0xb8532, 0xb8532, 
            0xb8532, 0xb8532, 0xb8540, 0xb8540, 0xb8541, 0xb8541, 0xb8542, 0xb8542, 0xb8542, 0xb8542, 
            0xb8543, 0xb8543, 0xb8543, 0xb8543, 0xb8543, 0xb8543, 0xb8543, 0xb8543, 0xb8600, 0xb8600, 
            0xb8610, 0xb8610, 0xb8620, 0xb8620, 0xb8621, 0xb8621, 0xb8630, 0xb8630, 0xb8631, 0xb8631, 
            0xb8632, 0xb8632, 0xb8632, 0xb8632, 0xb8640, 0xb8640, 0xb8641, 0xb8641, 0xb8642, 0xb8642, 
            0xb8642, 0xb8642, 0xb8643, 0xb8643, 0xb8643, 0xb8643, 0xb8643, 0xb8643, 0xb8643, 0xb8643, 
            0xb8650, 0xb8650, 0xb8651, 0xb8651, 0xb8652, 0xb8652, 0xb8652, 0xb8652, 0xb8653, 0xb8653, 
            0xb8653, 0xb8653, 0xb8653, 0xb8653, 0xb8653, 0xb8653, 0xb8654, 0xb8654, 0xb8654, 0xb8654, 
            0xb8654, 0xb8654, 0xb8654, 0xb8654, 0xb8654, 0xb8654, 0xb8654, 0xb8654, 0xb8654, 0xb8654, 
            0xb8654, 0xb8654, 0xb8700, 0xb8700, 0xb8710, 0xb8710, 0xb8720, 0xb8720, 0xb8721, 0xb8721, 
            0xb8730, 0xb8730, 0xb8731, 0xb8731, 0xb8732, 0xb8732, 0xb8732, 0xb8732, 0xb8740, 0xb8740, 
            0xb8741, 0xb8741, 0xb8742, 0xb8742, 0xb8742, 0xb8742, 0xb8743, 0xb8743, 0xb8743, 0xb8743, 
            0xb8743, 0xb8743, 0xb8743, 0xb8743, 0xb8750, 0xb8750, 0xb8751, 0xb8751, 0xb8752, 0xb8752, 
            0xb8752, 0xb8752, 0xb8753, 0xb8753, 0xb8753, 0xb8753, 0xb8753, 0xb8753, 0xb8753, 0xb8753, 
            0xb8754, 0xb8754, 0xb8754, 0xb8754, 0xb8754, 0xb8754, 0xb8754, 0xb8754, 0xb8754, 0xb8754, 
            0xb8754, 0xb8754, 0xb8754, 0xb8754, 0xb8754, 0xb8754, 0xb8760, 0xb8760, 0xb8761, 0xb8761, 
            0xb8762, 0xb8762, 0xb8762, 0xb8762, 0xb8763, 0xb8763, 0xb8763, 0xb8763, 0xb8763, 0xb8763, 
            0xb8763, 0xb8763, 0xb8764, 0xb8764, 0xb8764, 0xb8764, 0xb8764, 0xb8764, 0xb8764, 0xb8764, 
            0xb8764, 0xb8764, 0xb8764, 0xb8764, 0xb8764, 0xb8764, 0xb8764, 0xb8764, 0xb8765, 0xb8765, 
            0xb8765, 0xb8765, 0xb8765, 0xb8765, 0xb8765, 0xb8765, 0xb8765, 0xb8765, 0xb8765, 0xb8765, 
            0xb8765, 0xb8765, 0xb8765, 0xb8765, 0xb8765, 0xb8765, 0xb8765, 0xb8765, 0xb8765, 0xb8765, 
            0xb8765, 0xb8765, 0xb8765, 0xb8765, 0xb8765, 0xb8765, 0xb8765, 0xb8765, 0xb8765, 0xb8765, 
            0xb9000, 0xb9000, 0xb9100, 0xb9100, 0xb9200, 0xb9200, 0xb9210, 0xb9210, 0xb9300, 0xb9300, 
            0xb9310, 0xb9310, 0xb9320, 0xb9320, 0xb9321, 0xb9321, 0xb9400, 0xb9400, 0xb9410, 0xb9410, 
            0xb9420, 0xb9420, 0xb9421, 0xb9421, 0xb9430, 0xb9430, 0xb9431, 0xb9431, 0xb9432, 0xb9432, 
            0xb9432, 0xb9432, 0xb9500, 0xb9500, 0xb9510, 0xb9510, 0xb9520, 0xb9520, 0xb9521, 0xb9521, 
            0xb9530, 0xb9530, 0xb9531, 0xb9531, 0xb9532, 0xb9532, 0xb9532, 0xb9532, 0xb9540, 0xb9540, 
            0xb9541, 0xb9541, 0xb9542, 0xb9542, 0xb9542, 0xb9542, 0xb9543, 0xb9543, 0xb9543, 0xb9543, 
            0xb9543, 0xb9543, 0xb9543, 0xb9543, 0xb9600, 0xb9600, 0xb9610, 0xb9610, 0xb9620, 0xb9620, 
            0xb9621, 0xb9621, 0xb9630, 0xb9630, 0xb9631, 0xb9631, 0xb9632, 0xb9632, 0xb9632, 0xb9632, 
            0xb9640, 0xb9640, 0xb9641, 0xb9641, 0xb9642, 0xb9642, 0xb9642, 0xb9642, 0xb9643, 0xb9643, 
            0xb9643, 0xb9643, 0xb9643, 0xb9643, 0xb9643, 0xb9643, 0xb9650, 0xb9650, 0xb9651, 0xb9651, 
            0xb9652, 0xb9652, 0xb9652, 0xb9652, 0xb9653, 0xb9653, 0xb9653, 0xb9653, 0xb9653, 0xb9653, 
            0xb9653, 0xb9653, 0xb9654, 0xb9654, 0xb9654, 0xb9654, 0xb9654, 0xb9654, 0xb9654, 0xb9654, 
            0xb9654, 0xb9654, 0xb9654, 0xb9654, 0xb9654, 0xb9654, 0xb9654, 0xb9654, 0xb9700, 0xb9700, 
            0xb9710, 0xb9710, 0xb9720, 0xb9720, 0xb9721, 0xb9721, 0xb9730, 0xb9730, 0xb9731, 0xb9731, 
            0xb9732, 0xb9732, 0xb9732, 0xb9732, 0xb9740, 0xb9740, 0xb9741, 0xb9741, 0xb9742, 0xb9742, 
            0xb9742, 0xb9742, 0xb9743, 0xb9743, 0xb9743, 0xb9743, 0xb9743, 0xb9743, 0xb9743, 0xb9743, 
            0xb9750, 0xb9750, 0xb9751, 0xb9751, 0xb9752, 0xb9752, 0xb9752, 0xb9752, 0xb9753, 0xb9753, 
            0xb9753, 0xb9753, 0xb9753, 0xb9753, 0xb9753, 0xb9753, 0xb9754, 0xb9754, 0xb9754, 0xb9754, 
            0xb9754, 0xb9754, 0xb9754, 0xb9754, 0xb9754, 0xb9754, 0xb9754, 0xb9754, 0xb9754, 0xb9754, 
            0xb9754, 0xb9754, 0xb9760, 0xb9760, 0xb9761, 0xb9761, 0xb9762, 0xb9762, 0xb9762, 0xb9762, 
            0xb9763, 0xb9763, 0xb9763, 0xb9763, 0xb9763, 0xb9763, 0xb9763, 0xb9763, 0xb9764, 0xb9764, 
            0xb9764, 0xb9764, 0xb9764, 0xb9764, 0xb9764, 0xb9764, 0xb9764, 0xb9764, 0xb9764, 0xb9764, 
            0xb9764, 0xb9764, 0xb9764, 0xb9764, 0xb9765, 0xb9765, 0xb9765, 0xb9765, 0xb9765, 0xb9765, 
            0xb9765, 0xb9765, 0xb9765, 0xb9765, 0xb9765, 0xb9765, 0xb9765, 0xb9765, 0xb9765, 0xb9765, 
            0xb9765, 0xb9765, 0xb9765, 0xb9765, 0xb9765, 0xb9765, 0xb9765, 0xb9765, 0xb9765, 0xb9765, 
            0xb9765, 0xb9765, 0xb9765, 0xb9765, 0xb9765, 0xb9765, 0xb9800, 0xb9800, 0xb9810, 0xb9810, 
            0xb9820, 0xb9820, 0xb9821, 0xb9821, 0xb9830, 0xb9830, 0xb9831, 0xb9831, 0xb9832, 0xb9832, 
            0xb9832, 0xb9832, 0xb9840, 0xb9840, 0xb9841, 0xb9841, 0xb9842, 0xb9842, 0xb9842, 0xb9842, 
            0xb9843, 0xb9843, 0xb9843, 0xb9843, 0xb9843, 0xb9843, 0xb9843, 0xb9843, 0xb9850, 0xb9850, 
            0xb9851, 0xb9851, 0xb9852, 0xb9852, 0xb9852, 0xb9852, 0xb9853, 0xb9853, 0xb9853, 0xb9853, 
            0xb9853, 0xb9853, 0xb9853, 0xb9853, 0xb9854, 0xb9854, 0xb9854, 0xb9854, 0xb9854, 0xb9854, 
            0xb9854, 0xb9854, 0xb9854, 0xb9854, 0xb9854, 0xb9854, 0xb9854, 0xb9854, 0xb9854, 0xb9854, 
            0xb9860, 0xb9860, 0xb9861, 0xb9861, 0xb9862, 0xb9862, 0xb9862, 0xb9862, 0xb9863, 0xb9863, 
            0xb9863, 0xb9863, 0xb9863, 0xb9863, 0xb9863, 0xb9863, 0xb9864, 0xb9864, 0xb9864, 0xb9864, 
            0xb9864, 0xb9864, 0xb9864, 0xb9864, 0xb9864, 0xb9864, 0xb9864, 0xb9864, 0xb9864, 0xb9864, 
            0xb9864, 0xb9864, 0xb9865, 0xb9865, 0xb9865, 0xb9865, 0xb9865, 0xb9865, 0xb9865, 0xb9865, 
            0xb9865, 0xb9865, 0xb9865, 0xb9865, 0xb9865, 0xb9865, 0xb9865, 0xb9865, 0xb9865, 0xb9865, 
            0xb9865, 0xb9865, 0xb9865, 0xb9865, 0xb9865, 0xb9865, 0xb9865, 0xb9865, 0xb9865, 0xb9865, 
            0xb9865, 0xb9865, 0xb9865, 0xb9865, 0xb9870, 0xb9870, 0xb9871, 0xb9871, 0xb9872, 0xb9872, 
            0xb9872, 0xb9872, 0xb9873, 0xb9873, 0xb9873, 0xb9873, 0xb9873, 0xb9873, 0xb9873, 0xb9873, 
            0xb9874, 0xb9874, 0xb9874, 0xb9874, 0xb9874, 0xb9874, 0xb9874, 0xb9874, 0xb9874, 0xb9874, 
            0xb9874, 0xb9874, 0xb9874, 0xb9874, 0xb9874, 0xb9874, 0xb9875, 0xb9875, 0xb9875, 0xb9875, 
            0xb9875, 0xb9875, 0xb9875, 0xb9875, 0xb9875, 0xb9875, 0xb9875, 0xb9875, 0xb9875, 0xb9875, 
            0xb9875, 0xb9875, 0xb9875, 0xb9875, 0xb9875, 0xb9875, 0xb9875, 0xb9875, 0xb9875, 0xb9875, 
            0xb9875, 0xb9875, 0xb9875, 0xb9875, 0xb9875, 0xb9875, 0xb9875, 0xb9875, 0xb9876, 0xb9876, 
            0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 
            0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 
            0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 
            0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 
            0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 
            0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 0xb9876, 
            0xb9876, 0xb9876, 0xba000, 0xba000, 0xba100, 0xba100, 0xba200, 0xba200, 0xba210, 0xba210, 
            0xba300, 0xba300, 0xba310, 0xba310, 0xba320, 0xba320, 0xba321, 0xba321, 0xba400, 0xba400, 
            0xba410, 0xba410, 0xba420, 0xba420, 0xba421, 0xba421, 0xba430, 0xba430, 0xba431, 0xba431, 
            0xba432, 0xba432, 0xba432, 0xba432, 0xba500, 0xba500, 0xba510, 0xba510, 0xba520, 0xba520, 
            0xba521, 0xba521, 0xba530, 0xba530, 0xba531, 0xba531, 0xba532, 0xba532, 0xba532, 0xba532, 
            0xba540, 0xba540, 0xba541, 0xba541, 0xba542, 0xba542, 0xba542, 0xba542, 0xba543, 0xba543, 
            0xba543, 0xba543, 0xba543, 0xba543, 0xba543, 0xba543, 0xba600, 0xba600, 0xba610, 0xba610, 
            0xba620, 0xba620, 0xba621, 0xba621, 0xba630, 0xba630, 0xba631, 0xba631, 0xba632, 0xba632, 
            0xba632, 0xba632, 0xba640, 0xba640, 0xba641, 0xba641, 0xba642, 0xba642, 0xba642, 0xba642, 
            0xba643, 0xba643, 0xba643, 0xba643, 0xba643, 0xba643, 0xba643, 0xba643, 0xba650, 0xba650, 
            0xba651, 0xba651, 0xba652, 0xba652, 0xba652, 0xba652, 0xba653, 0xba653, 0xba653, 0xba653, 
            0xba653, 0xba653, 0xba653, 0xba653, 0xba654, 0xba654, 0xba654, 0xba654, 0xba654, 0xba654, 
            0xba654, 0xba654, 0xba654, 0xba654, 0xba654, 0xba654, 0xba654, 0xba654, 0xba654, 0xba654, 
            0xba700, 0xba700, 0xba710, 0xba710, 0xba720, 0xba720, 0xba721, 0xba721, 0xba730, 0xba730, 
            0xba731, 0xba731, 0xba732, 0xba732, 0xba732, 0xba732, 0xba740, 0xba740, 0xba741, 0xba741, 
            0xba742, 0xba742, 0xba742, 0xba742, 0xba743, 0xba743, 0xba743, 0xba743, 0xba743, 0xba743, 
            0xba743, 0xba743, 0xba750, 0xba750, 0xba751, 0xba751, 0xba752, 0xba752, 0xba752, 0xba752, 
            0xba753, 0xba753, 0xba753, 0xba753, 0xba753, 0xba753, 0xba753, 0xba753, 0xba754, 0xba754, 
            0xba754, 0xba754, 0xba754, 0xba754, 0xba754, 0xba754, 0xba754, 0xba754, 0xba754, 0xba754, 
            0xba754, 0xba754, 0xba754, 0xba754, 0xba760, 0xba760, 0xba761, 0xba761, 0xba762, 0xba762, 
            0xba762, 0xba762, 0xba763, 0xba763, 0xba763, 0xba763, 0xba763, 0xba763, 0xba763, 0xba763, 
            0xba764, 0xba764, 0xba764, 0xba764, 0xba764, 0xba764, 0xba764, 0xba764, 0xba764, 0xba764, 
            0xba764, 0xba764, 0xba764, 0xba764, 0xba764, 0xba764, 0xba765, 0xba765, 0xba765, 0xba765, 
            0xba765, 0xba765, 0xba765, 0xba765, 0xba765, 0xba765, 0xba765, 0xba765, 0xba765, 0xba765, 
            0xba765, 0xba765, 0xba765, 0xba765, 0xba765, 0xba765, 0xba765, 0xba765, 0xba765, 0xba765, 
            0xba765, 0xba765, 0xba765, 0xba765, 0xba765, 0xba765, 0xba765, 0xba765, 0xba800, 0xba800, 
            0xba810, 0xba810, 0xba820, 0xba820, 0xba821, 0xba821, 0xba830, 0xba830, 0xba831, 0xba831, 
            0xba832, 0xba832, 0xba832, 0xba832, 0xba840, 0xba840, 0xba841, 0xba841, 0xba842, 0xba842, 
            0xba842, 0xba842, 0xba843, 0xba843, 0xba843, 0xba843, 0xba843, 0xba843, 0xba843, 0xba843, 
            0xba850, 0xba850, 0xba851, 0xba851, 0xba852, 0xba852, 0xba852, 0xba852, 0xba853, 0xba853, 
            0xba853, 0xba853, 0xba853, 0xba853, 0xba853, 0xba853, 0xba854, 0xba854, 0xba854, 0xba854, 
            0xba854, 0xba854, 0xba854, 0xba854, 0xba854, 0xba854, 0xba854, 0xba854, 0xba854, 0xba854, 
            0xba854, 0xba854, 0xba860, 0xba860, 0xba861, 0xba861, 0xba862, 0xba862, 0xba862, 0xba862, 
            0xba863, 0xba863, 0xba863, 0xba863, 0xba863, 0xba863, 0xba863, 0xba863, 0xba864, 0xba864, 
            0xba864, 0xba864, 0xba864, 0xba864, 0xba864, 0xba864, 0xba864, 0xba864, 0xba864, 0xba864, 
            0xba864, 0xba864, 0xba864, 0xba864, 0xba865, 0xba865, 0xba865, 0xba865, 0xba865, 0xba865, 
            0xba865, 0xba865, 0xba865, 0xba865, 0xba865, 0xba865, 0xba865, 0xba865, 0xba865, 0xba865, 
            0xba865, 0xba865, 0xba865, 0xba865, 0xba865, 0xba865, 0xba865, 0xba865, 0xba865, 0xba865, 
            0xba865, 0xba865, 0xba865, 0xba865, 0xba865, 0xba865, 0xba870, 0xba870, 0xba871, 0xba871, 
            0xba872, 0xba872, 0xba872, 0xba872, 0xba873, 0xba873, 0xba873, 0xba873, 0xba873, 0xba873, 
            0xba873, 0xba873, 0xba874, 0xba874, 0xba874, 0xba874, 0xba874, 0xba874, 0xba874, 0xba874, 
            0xba874, 0xba874, 0xba874, 0xba874, 0xba874, 0xba874, 0xba874, 0xba874, 0xba875, 0xba875, 
            0xba875, 0xba875, 0xba875, 0xba875, 0xba875, 0xba875, 0xba875, 0xba875, 0xba875, 0xba875, 
            0xba875, 0xba875, 0xba875, 0xba875, 0xba875, 0xba875, 0xba875, 0xba875, 0xba875, 0xba875, 
            0xba875, 0xba875, 0xba875, 0xba875, 0xba875, 0xba875, 0xba875, 0xba875, 0xba875, 0xba875, 
            0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 
            0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 
            0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 
            0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 
            0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 
            0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 0xba876, 
            0xba876, 0xba876, 0xba876, 0xba876, 0xba900, 0xba900, 0xba910, 0xba910, 0xba920, 0xba920, 
            0xba921, 0xba921, 0xba930, 0xba930, 0xba931, 0xba931, 0xba932, 0xba932, 0xba932, 0xba932, 
            0xba940, 0xba940, 0xba941, 0xba941, 0xba942, 0xba942, 0xba942, 0xba942, 0xba943, 0xba943, 
            0xba943, 0xba943, 0xba943, 0xba943, 0xba943, 0xba943, 0xba950, 0xba950, 0xba951, 0xba951, 
            0xba952, 0xba952, 0xba952, 0xba952, 0xba953, 0xba953, 0xba953, 0xba953, 0xba953, 0xba953, 
            0xba953, 0xba953, 0xba954, 0xba954, 0xba954, 0xba954, 0xba954, 0xba954, 0xba954, 0xba954, 
            0xba954, 0xba954, 0xba954, 0xba954, 0xba954, 0xba954, 0xba954, 0xba954, 0xba960, 0xba960, 
            0xba961, 0xba961, 0xba962, 0xba962, 0xba962, 0xba962, 0xba963, 0xba963, 0xba963, 0xba963, 
            0xba963, 0xba963, 0xba963, 0xba963, 0xba964, 0xba964, 0xba964, 0xba964, 0xba964, 0xba964, 
            0xba964, 0xba964, 0xba964, 0xba964, 0xba964, 0xba964, 0xba964, 0xba964, 0xba964, 0xba964, 
            0xba965, 0xba965, 0xba965, 0xba965, 0xba965, 0xba965, 0xba965, 0xba965, 0xba965, 0xba965, 
            0xba965, 0xba965, 0xba965, 0xba965, 0xba965, 0xba965, 0xba965, 0xba965, 0xba965, 0xba965, 
            0xba965, 0xba965, 0xba965, 0xba965, 0xba965, 0xba965, 0xba965, 0xba965, 0xba965, 0xba965, 
            0xba965, 0xba965, 0xba970, 0xba970, 0xba971, 0xba971, 0xba972, 0xba972, 0xba972, 0xba972, 
            0xba973, 0xba973, 0xba973, 0xba973, 0xba973, 0xba973, 0xba973, 0xba973, 0xba974, 0xba974, 
            0xba974, 0xba974, 0xba974, 0xba974, 0xba974, 0xba974, 0xba974, 0xba974, 0xba974, 0xba974, 
            0xba974, 0xba974, 0xba974, 0xba974, 0xba975, 0xba975, 0xba975, 0xba975, 0xba975, 0xba975, 
            0xba975, 0xba975, 0xba975, 0xba975, 0xba975, 0xba975, 0xba975, 0xba975, 0xba975, 0xba975, 
            0xba975, 0xba975, 0xba975, 0xba975, 0xba975, 0xba975, 0xba975, 0xba975, 0xba975, 0xba975, 
            0xba975, 0xba975, 0xba975, 0xba975, 0xba975, 0xba975, 0xba976, 0xba976, 0xba976, 0xba976, 
            0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 
            0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 
            0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 
            0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 
            0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 
            0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 0xba976, 
            0xba980, 0xba980, 0xba981, 0xba981, 0xba982, 0xba982, 0xba982, 0xba982, 0xba983, 0xba983, 
            0xba983, 0xba983, 0xba983, 0xba983, 0xba983, 0xba983, 0xba984, 0xba984, 0xba984, 0xba984, 
            0xba984, 0xba984, 0xba984, 0xba984, 0xba984, 0xba984, 0xba984, 0xba984, 0xba984, 0xba984, 
            0xba984, 0xba984, 0xba985, 0xba985, 0xba985, 0xba985, 0xba985, 0xba985, 0xba985, 0xba985, 
            0xba985, 0xba985, 0xba985, 0xba985, 0xba985, 0xba985, 0xba985, 0xba985, 0xba985, 0xba985, 
            0xba985, 0xba985, 0xba985, 0xba985, 0xba985, 0xba985, 0xba985, 0xba985, 0xba985, 0xba985, 
            0xba985, 0xba985, 0xba985, 0xba985, 0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 
            0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 
            0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 
            0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 
            0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 
            0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 
            0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 0xba986, 0xba987, 0xba987, 
            0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 
            0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 
            0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 
            0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 
            0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 
            0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 
            0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 
            0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 
            0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 
            0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 
            0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 
            0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 
            0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xba987, 0xc0000, 0xc0000, 0xc1000, 0xc1000, 
            0xc2000, 0xc2000, 0xc2100, 0xc2100, 0xc3000, 0xc3000, 0xc3100, 0xc3100, 0xc3200, 0xc3200, 
            0xc3210, 0xc3210, 0xc4000, 0xc4000, 0xc4100, 0xc4100, 0xc4200, 0xc4200, 0xc4210, 0xc4210, 
            0xc4300, 0xc4300, 0xc4310, 0xc4310, 0xc4320, 0xc4320, 0xc4321, 0xc4321, 0xc5000, 0xc5000, 
            0xc5100, 0xc5100, 0xc5200, 0xc5200, 0xc5210, 0xc5210, 0xc5300, 0xc5300, 0xc5310, 0xc5310, 
            0xc5320, 0xc5320, 0xc5321, 0xc5321, 0xc5400, 0xc5400, 0xc5410, 0xc5410, 0xc5420, 0xc5420, 
            0xc5421, 0xc5421, 0xc5430, 0xc5430, 0xc5431, 0xc5431, 0xc5432, 0xc5432, 0xc5432, 0xc5432, 
            0xc6000, 0xc6000, 0xc6100, 0xc6100, 0xc6200, 0xc6200, 0xc6210, 0xc6210, 0xc6300, 0xc6300, 
            0xc6310, 0xc6310, 0xc6320, 0xc6320, 0xc6321, 0xc6321, 0xc6400, 0xc6400, 0xc6410, 0xc6410, 
            0xc6420, 0xc6420, 0xc6421, 0xc6421, 0xc6430, 0xc6430, 0xc6431, 0xc6431, 0xc6432, 0xc6432, 
            0xc6432, 0xc6432, 0xc6500, 0xc6500, 0xc6510, 0xc6510, 0xc6520, 0xc6520, 0xc6521, 0xc6521, 
            0xc6530, 0xc6530, 0xc6531, 0xc6531, 0xc6532, 0xc6532, 0xc6532, 0xc6532, 0xc6540, 0xc6540, 
            0xc6541, 0xc6541, 0xc6542, 0xc6542, 0xc6542, 0xc6542, 0xc6543, 0xc6543, 0xc6543, 0xc6543, 
            0xc6543, 0xc6543, 0xc6543, 0xc6543, 0xc7000, 0xc7000, 0xc7100, 0xc7100, 0xc7200, 0xc7200, 
            0xc7210, 0xc7210, 0xc7300, 0xc7300, 0xc7310, 0xc7310, 0xc7320, 0xc7320, 0xc7321, 0xc7321, 
            0xc7400, 0xc7400, 0xc7410, 0xc7410, 0xc7420, 0xc7420, 0xc7421, 0xc7421, 0xc7430, 0xc7430, 
            0xc7431, 0xc7431, 0xc7432, 0xc7432, 0xc7432, 0xc7432, 0xc7500, 0xc7500, 0xc7510, 0xc7510, 
            0xc7520, 0xc7520, 0xc7521, 0xc7521, 0xc7530, 0xc7530, 0xc7531, 0xc7531, 0xc7532, 0xc7532, 
            0xc7532, 0xc7532, 0xc7540, 0xc7540, 0xc7541, 0xc7541, 0xc7542, 0xc7542, 0xc7542, 0xc7542, 
            0xc7543, 0xc7543, 0xc7543, 0xc7543, 0xc7543, 0xc7543, 0xc7543, 0xc7543, 0xc7600, 0xc7600, 
            0xc7610, 0xc7610, 0xc7620, 0xc7620, 0xc7621, 0xc7621, 0xc7630, 0xc7630, 0xc7631, 0xc7631, 
            0xc7632, 0xc7632, 0xc7632, 0xc7632, 0xc7640, 0xc7640, 0xc7641, 0xc7641, 0xc7642, 0xc7642, 
            0xc7642, 0xc7642, 0xc7643, 0xc7643, 0xc7643, 0xc7643, 0xc7643, 0xc7643, 0xc7643, 0xc7643, 
            0xc7650, 0xc7650, 0xc7651, 0xc7651, 0xc7652, 0xc7652, 0xc7652, 0xc7652, 0xc7653, 0xc7653, 
            0xc7653, 0xc7653, 0xc7653, 0xc7653, 0xc7653, 0xc7653, 0xc7654, 0xc7654, 0xc7654, 0xc7654, 
            0xc7654, 0xc7654, 0xc7654, 0xc7654, 0xc7654, 0xc7654, 0xc7654, 0xc7654, 0xc7654, 0xc7654, 
            0xc7654, 0xc7654, 0xc8000, 0xc8000, 0xc8100, 0xc8100, 0xc8200, 0xc8200, 0xc8210, 0xc8210, 
            0xc8300, 0xc8300, 0xc8310, 0xc8310, 0xc8320, 0xc8320, 0xc8321, 0xc8321, 0xc8400, 0xc8400, 
            0xc8410, 0xc8410, 0xc8420, 0xc8420, 0xc8421, 0xc8421, 0xc8430, 0xc8430, 0xc8431, 0xc8431, 
            0xc8432, 0xc8432, 0xc8432, 0xc8432, 0xc8500, 0xc8500, 0xc8510, 0xc8510, 0xc8520, 0xc8520, 
            0xc8521, 0xc8521, 0xc8530, 0xc8530, 0xc8531, 0xc8531, 0xc8532, 0xc8532, 0xc8532, 0xc8532, 
            0xc8540, 0xc8540, 0xc8541, 0xc8541, 0xc8542, 0xc8542, 0xc8542, 0xc8542, 0xc8543, 0xc8543, 
            0xc8543, 0xc8543, 0xc8543, 0xc8543, 0xc8543, 0xc8543, 0xc8600, 0xc8600, 0xc8610, 0xc8610, 
            0xc8620, 0xc8620, 0xc8621, 0xc8621, 0xc8630, 0xc8630, 0xc8631, 0xc8631, 0xc8632, 0xc8632, 
            0xc8632, 0xc8632, 0xc8640, 0xc8640, 0xc8641, 0xc8641, 0xc8642, 0xc8642, 0xc8642, 0xc8642, 
            0xc8643, 0xc8643, 0xc8643, 0xc8643, 0xc8643, 0xc8643, 0xc8643, 0xc8643, 0xc8650, 0xc8650, 
            0xc8651, 0xc8651, 0xc8652, 0xc8652, 0xc8652, 0xc8652, 0xc8653, 0xc8653, 0xc8653, 0xc8653, 
            0xc8653, 0xc8653, 0xc8653, 0xc8653, 0xc8654, 0xc8654, 0xc8654, 0xc8654, 0xc8654, 0xc8654, 
            0xc8654, 0xc8654, 0xc8654, 0xc8654, 0xc8654, 0xc8654, 0xc8654, 0xc8654, 0xc8654, 0xc8654, 
            0xc8700, 0xc8700, 0xc8710, 0xc8710, 0xc8720, 0xc8720, 0xc8721, 0xc8721, 0xc8730, 0xc8730, 
            0xc8731, 0xc8731, 0xc8732, 0xc8732, 0xc8732, 0xc8732, 0xc8740, 0xc8740, 0xc8741, 0xc8741, 
            0xc8742, 0xc8742, 0xc8742, 0xc8742, 0xc8743, 0xc8743, 0xc8743, 0xc8743, 0xc8743, 0xc8743, 
            0xc8743, 0xc8743, 0xc8750, 0xc8750, 0xc8751, 0xc8751, 0xc8752, 0xc8752, 0xc8752, 0xc8752, 
            0xc8753, 0xc8753, 0xc8753, 0xc8753, 0xc8753, 0xc8753, 0xc8753, 0xc8753, 0xc8754, 0xc8754, 
            0xc8754, 0xc8754, 0xc8754, 0xc8754, 0xc8754, 0xc8754, 0xc8754, 0xc8754, 0xc8754, 0xc8754, 
            0xc8754, 0xc8754, 0xc8754, 0xc8754, 0xc8760, 0xc8760, 0xc8761, 0xc8761, 0xc8762, 0xc8762, 
            0xc8762, 0xc8762, 0xc8763, 0xc8763, 0xc8763, 0xc8763, 0xc8763, 0xc8763, 0xc8763, 0xc8763, 
            0xc8764, 0xc8764, 0xc8764, 0xc8764, 0xc8764, 0xc8764, 0xc8764, 0xc8764, 0xc8764, 0xc8764, 
            0xc8764, 0xc8764, 0xc8764, 0xc8764, 0xc8764, 0xc8764, 0xc8765, 0xc8765, 0xc8765, 0xc8765, 
            0xc8765, 0xc8765, 0xc8765, 0xc8765, 0xc8765, 0xc8765, 0xc8765, 0xc8765, 0xc8765, 0xc8765, 
            0xc8765, 0xc8765, 0xc8765, 0xc8765, 0xc8765, 0xc8765, 0xc8765, 0xc8765, 0xc8765, 0xc8765, 
            0xc8765, 0xc8765, 0xc8765, 0xc8765, 0xc8765, 0xc8765, 0xc8765, 0xc8765, 0xc9000, 0xc9000, 
            0xc9100, 0xc9100, 0xc9200, 0xc9200, 0xc9210, 0xc9210, 0xc9300, 0xc9300, 0xc9310, 0xc9310, 
            0xc9320, 0xc9320, 0xc9321, 0xc9321, 0xc9400, 0xc9400, 0xc9410, 0xc9410, 0xc9420, 0xc9420, 
            0xc9421, 0xc9421, 0xc9430, 0xc9430, 0xc9431, 0xc9431, 0xc9432, 0xc9432, 0xc9432, 0xc9432, 
            0xc9500, 0xc9500, 0xc9510, 0xc9510, 0xc9520, 0xc9520, 0xc9521, 0xc9521, 0xc9530, 0xc9530, 
            0xc9531, 0xc9531, 0xc9532, 0xc9532, 0xc9532, 0xc9532, 0xc9540, 0xc9540, 0xc9541, 0xc9541, 
            0xc9542, 0xc9542, 0xc9542, 0xc9542, 0xc9543, 0xc9543, 0xc9543, 0xc9543, 0xc9543, 0xc9543, 
            0xc9543, 0xc9543, 0xc9600, 0xc9600, 0xc9610, 0xc9610, 0xc9620, 0xc9620, 0xc9621, 0xc9621, 
            0xc9630, 0xc9630, 0xc9631, 0xc9631, 0xc9632, 0xc9632, 0xc9632, 0xc9632, 0xc9640, 0xc9640, 
            0xc9641, 0xc9641, 0xc9642, 0xc9642, 0xc9642, 0xc9642, 0xc9643, 0xc9643, 0xc9643, 0xc9643, 
            0xc9643, 0xc9643, 0xc9643, 0xc9643, 0xc9650, 0xc9650, 0xc9651, 0xc9651, 0xc9652, 0xc9652, 
            0xc9652, 0xc9652, 0xc9653, 0xc9653, 0xc9653, 0xc9653, 0xc9653, 0xc9653, 0xc9653, 0xc9653, 
            0xc9654, 0xc9654, 0xc9654, 0xc9654, 0xc9654, 0xc9654, 0xc9654, 0xc9654, 0xc9654, 0xc9654, 
            0xc9654, 0xc9654, 0xc9654, 0xc9654, 0xc9654, 0xc9654, 0xc9700, 0xc9700, 0xc9710, 0xc9710, 
            0xc9720, 0xc9720, 0xc9721, 0xc9721, 0xc9730, 0xc9730, 0xc9731, 0xc9731, 0xc9732, 0xc9732, 
            0xc9732, 0xc9732, 0xc9740, 0xc9740, 0xc9741, 0xc9741, 0xc9742, 0xc9742, 0xc9742, 0xc9742, 
            0xc9743, 0xc9743, 0xc9743, 0xc9743, 0xc9743, 0xc9743, 0xc9743, 0xc9743, 0xc9750, 0xc9750, 
            0xc9751, 0xc9751, 0xc9752, 0xc9752, 0xc9752, 0xc9752, 0xc9753, 0xc9753, 0xc9753, 0xc9753, 
            0xc9753, 0xc9753, 0xc9753, 0xc9753, 0xc9754, 0xc9754, 0xc9754, 0xc9754, 0xc9754, 0xc9754, 
            0xc9754, 0xc9754, 0xc9754, 0xc9754, 0xc9754, 0xc9754, 0xc9754, 0xc9754, 0xc9754, 0xc9754, 
            0xc9760, 0xc9760, 0xc9761, 0xc9761, 0xc9762, 0xc9762, 0xc9762, 0xc9762, 0xc9763, 0xc9763, 
            0xc9763, 0xc9763, 0xc9763, 0xc9763, 0xc9763, 0xc9763, 0xc9764, 0xc9764, 0xc9764, 0xc9764, 
            0xc9764, 0xc9764, 0xc9764, 0xc9764, 0xc9764, 0xc9764, 0xc9764, 0xc9764, 0xc9764, 0xc9764, 
            0xc9764, 0xc9764, 0xc9765, 0xc9765, 0xc9765, 0xc9765, 0xc9765, 0xc9765, 0xc9765, 0xc9765, 
            0xc9765, 0xc9765, 0xc9765, 0xc9765, 0xc9765, 0xc9765, 0xc9765, 0xc9765, 0xc9765, 0xc9765, 
            0xc9765, 0xc9765, 0xc9765, 0xc9765, 0xc9765, 0xc9765, 0xc9765, 0xc9765, 0xc9765, 0xc9765, 
            0xc9765, 0xc9765, 0xc9765, 0xc9765, 0xc9800, 0xc9800, 0xc9810, 0xc9810, 0xc9820, 0xc9820, 
            0xc9821, 0xc9821, 0xc9830, 0xc9830, 0xc9831, 0xc9831, 0xc9832, 0xc9832, 0xc9832, 0xc9832, 
            0xc9840, 0xc9840, 0xc9841, 0xc9841, 0xc9842, 0xc9842, 0xc9842, 0xc9842, 0xc9843, 0xc9843, 
            0xc9843, 0xc9843, 0xc9843, 0xc9843, 0xc9843, 0xc9843, 0xc9850, 0xc9850, 0xc9851, 0xc9851, 
            0xc9852, 0xc9852, 0xc9852, 0xc9852, 0xc9853, 0xc9853, 0xc9853, 0xc9853, 0xc9853, 0xc9853, 
            0xc9853, 0xc9853, 0xc9854, 0xc9854, 0xc9854, 0xc9854, 0xc9854, 0xc9854, 0xc9854, 0xc9854, 
            0xc9854, 0xc9854, 0xc9854, 0xc9854, 0xc9854, 0xc9854, 0xc9854, 0xc9854, 0xc9860, 0xc9860, 
            0xc9861, 0xc9861, 0xc9862, 0xc9862, 0xc9862, 0xc9862, 0xc9863, 0xc9863, 0xc9863, 0xc9863, 
            0xc9863, 0xc9863, 0xc9863, 0xc9863, 0xc9864, 0xc9864, 0xc9864, 0xc9864, 0xc9864, 0xc9864, 
            0xc9864, 0xc9864, 0xc9864, 0xc9864, 0xc9864, 0xc9864, 0xc9864, 0xc9864, 0xc9864, 0xc9864, 
            0xc9865, 0xc9865, 0xc9865, 0xc9865, 0xc9865, 0xc9865, 0xc9865, 0xc9865, 0xc9865, 0xc9865, 
            0xc9865, 0xc9865, 0xc9865, 0xc9865, 0xc9865, 0xc9865, 0xc9865, 0xc9865, 0xc9865, 0xc9865, 
            0xc9865, 0xc9865, 0xc9865, 0xc9865, 0xc9865, 0xc9865, 0xc9865, 0xc9865, 0xc9865, 0xc9865, 
            0xc9865, 0xc9865, 0xc9870, 0xc9870, 0xc9871, 0xc9871, 0xc9872, 0xc9872, 0xc9872, 0xc9872, 
            0xc9873, 0xc9873, 0xc9873, 0xc9873, 0xc9873, 0xc9873, 0xc9873, 0xc9873, 0xc9874, 0xc9874, 
            0xc9874, 0xc9874, 0xc9874, 0xc9874, 0xc9874, 0xc9874, 0xc9874, 0xc9874, 0xc9874, 0xc9874, 
            0xc9874, 0xc9874, 0xc9874, 0xc9874, 0xc9875, 0xc9875, 0xc9875, 0xc9875, 0xc9875, 0xc9875, 
            0xc9875, 0xc9875, 0xc9875, 0xc9875, 0xc9875, 0xc9875, 0xc9875, 0xc9875, 0xc9875, 0xc9875, 
            0xc9875, 0xc9875, 0xc9875, 0xc9875, 0xc9875, 0xc9875, 0xc9875, 0xc9875, 0xc9875, 0xc9875, 
            0xc9875, 0xc9875, 0xc9875, 0xc9875, 0xc9875, 0xc9875, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 
            0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 
            0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 
            0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 
            0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 
            0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 
            0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 0xc9876, 
            0xca000, 0xca000, 0xca100, 0xca100, 0xca200, 0xca200, 0xca210, 0xca210, 0xca300, 0xca300, 
            0xca310, 0xca310, 0xca320, 0xca320, 0xca321, 0xca321, 0xca400, 0xca400, 0xca410, 0xca410, 
            0xca420, 0xca420, 0xca421, 0xca421, 0xca430, 0xca430, 0xca431, 0xca431, 0xca432, 0xca432, 
            0xca432, 0xca432, 0xca500, 0xca500, 0xca510, 0xca510, 0xca520, 0xca520, 0xca521, 0xca521, 
            0xca530, 0xca530, 0xca531, 0xca531, 0xca532, 0xca532, 0xca532, 0xca532, 0xca540, 0xca540, 
            0xca541, 0xca541, 0xca542, 0xca542, 0xca542, 0xca542, 0xca543, 0xca543, 0xca543, 0xca543, 
            0xca543, 0xca543, 0xca543, 0xca543, 0xca600, 0xca600, 0xca610, 0xca610, 0xca620, 0xca620, 
            0xca621, 0xca621, 0xca630, 0xca630, 0xca631, 0xca631, 0xca632, 0xca632, 0xca632, 0xca632, 
            0xca640, 0xca640, 0xca641, 0xca641, 0xca642, 0xca642, 0xca642, 0xca642, 0xca643, 0xca643, 
            0xca643, 0xca643, 0xca643, 0xca643, 0xca643, 0xca643, 0xca650, 0xca650, 0xca651, 0xca651, 
            0xca652, 0xca652, 0xca652, 0xca652, 0xca653, 0xca653, 0xca653, 0xca653, 0xca653, 0xca653, 
            0xca653, 0xca653, 0xca654, 0xca654, 0xca654, 0xca654, 0xca654, 0xca654, 0xca654, 0xca654, 
            0xca654, 0xca654, 0xca654, 0xca654, 0xca654, 0xca654, 0xca654, 0xca654, 0xca700, 0xca700, 
            0xca710, 0xca710, 0xca720, 0xca720, 0xca721, 0xca721, 0xca730, 0xca730, 0xca731, 0xca731, 
            0xca732, 0xca732, 0xca732, 0xca732, 0xca740, 0xca740, 0xca741, 0xca741, 0xca742, 0xca742, 
            0xca742, 0xca742, 0xca743, 0xca743, 0xca743, 0xca743, 0xca743, 0xca743, 0xca743, 0xca743, 
            0xca750, 0xca750, 0xca751, 0xca751, 0xca752, 0xca752, 0xca752, 0xca752, 0xca753, 0xca753, 
            0xca753, 0xca753, 0xca753, 0xca753, 0xca753, 0xca753, 0xca754, 0xca754, 0xca754, 0xca754, 
            0xca754, 0xca754, 0xca754, 0xca754, 0xca754, 0xca754, 0xca754, 0xca754, 0xca754, 0xca754, 
            0xca754, 0xca754, 0xca760, 0xca760, 0xca761, 0xca761, 0xca762, 0xca762, 0xca762, 0xca762, 
            0xca763, 0xca763, 0xca763, 0xca763, 0xca763, 0xca763, 0xca763, 0xca763, 0xca764, 0xca764, 
            0xca764, 0xca764, 0xca764, 0xca764, 0xca764, 0xca764, 0xca764, 0xca764, 0xca764, 0xca764, 
            0xca764, 0xca764, 0xca764, 0xca764, 0xca765, 0xca765, 0xca765, 0xca765, 0xca765, 0xca765, 
            0xca765, 0xca765, 0xca765, 0xca765, 0xca765, 0xca765, 0xca765, 0xca765, 0xca765, 0xca765, 
            0xca765, 0xca765, 0xca765, 0xca765, 0xca765, 0xca765, 0xca765, 0xca765, 0xca765, 0xca765, 
            0xca765, 0xca765, 0xca765, 0xca765, 0xca765, 0xca765, 0xca800, 0xca800, 0xca810, 0xca810, 
            0xca820, 0xca820, 0xca821, 0xca821, 0xca830, 0xca830, 0xca831, 0xca831, 0xca832, 0xca832, 
            0xca832, 0xca832, 0xca840, 0xca840, 0xca841, 0xca841, 0xca842, 0xca842, 0xca842, 0xca842, 
            0xca843, 0xca843, 0xca843, 0xca843, 0xca843, 0xca843, 0xca843, 0xca843, 0xca850, 0xca850, 
            0xca851, 0xca851, 0xca852, 0xca852, 0xca852, 0xca852, 0xca853, 0xca853, 0xca853, 0xca853, 
            0xca853, 0xca853, 0xca853, 0xca853, 0xca854, 0xca854, 0xca854, 0xca854, 0xca854, 0xca854, 
            0xca854, 0xca854, 0xca854, 0xca854, 0xca854, 0xca854, 0xca854, 0xca854, 0xca854, 0xca854, 
            0xca860, 0xca860, 0xca861, 0xca861, 0xca862, 0xca862, 0xca862, 0xca862, 0xca863, 0xca863, 
            0xca863, 0xca863, 0xca863, 0xca863, 0xca863, 0xca863, 0xca864, 0xca864, 0xca864, 0xca864, 
            0xca864, 0xca864, 0xca864, 0xca864, 0xca864, 0xca864, 0xca864, 0xca864, 0xca864, 0xca864, 
            0xca864, 0xca864, 0xca865, 0xca865, 0xca865, 0xca865, 0xca865, 0xca865, 0xca865, 0xca865, 
            0xca865, 0xca865, 0xca865, 0xca865, 0xca865, 0xca865, 0xca865, 0xca865, 0xca865, 0xca865, 
            0xca865, 0xca865, 0xca865, 0xca865, 0xca865, 0xca865, 0xca865, 0xca865, 0xca865, 0xca865, 
            0xca865, 0xca865, 0xca865, 0xca865, 0xca870, 0xca870, 0xca871, 0xca871, 0xca872, 0xca872, 
            0xca872, 0xca872, 0xca873, 0xca873, 0xca873, 0xca873, 0xca873, 0xca873, 0xca873, 0xca873, 
            0xca874, 0xca874, 0xca874, 0xca874, 0xca874, 0xca874, 0xca874, 0xca874, 0xca874, 0xca874, 
            0xca874, 0xca874, 0xca874, 0xca874, 0xca874, 0xca874, 0xca875, 0xca875, 0xca875, 0xca875, 
            0xca875, 0xca875, 0xca875, 0xca875, 0xca875, 0xca875, 0xca875, 0xca875, 0xca875, 0xca875, 
            0xca875, 0xca875, 0xca875, 0xca875, 0xca875, 0xca875, 0xca875, 0xca875, 0xca875, 0xca875, 
            0xca875, 0xca875, 0xca875, 0xca875, 0xca875, 0xca875, 0xca875, 0xca875, 0xca876, 0xca876, 
            0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 
            0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 
            0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 
            0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 
            0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 
            0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 0xca876, 
            0xca876, 0xca876, 0xca900, 0xca900, 0xca910, 0xca910, 0xca920, 0xca920, 0xca921, 0xca921, 
            0xca930, 0xca930, 0xca931, 0xca931, 0xca932, 0xca932, 0xca932, 0xca932, 0xca940, 0xca940, 
            0xca941, 0xca941, 0xca942, 0xca942, 0xca942, 0xca942, 0xca943, 0xca943, 0xca943, 0xca943, 
            0xca943, 0xca943, 0xca943, 0xca943, 0xca950, 0xca950, 0xca951, 0xca951, 0xca952, 0xca952, 
            0xca952, 0xca952, 0xca953, 0xca953, 0xca953, 0xca953, 0xca953, 0xca953, 0xca953, 0xca953, 
            0xca954, 0xca954, 0xca954, 0xca954, 0xca954, 0xca954, 0xca954, 0xca954, 0xca954, 0xca954, 
            0xca954, 0xca954, 0xca954, 0xca954, 0xca954, 0xca954, 0xca960, 0xca960, 0xca961, 0xca961, 
            0xca962, 0xca962, 0xca962, 0xca962, 0xca963, 0xca963, 0xca963, 0xca963, 0xca963, 0xca963, 
            0xca963, 0xca963, 0xca964, 0xca964, 0xca964, 0xca964, 0xca964, 0xca964, 0xca964, 0xca964, 
            0xca964, 0xca964, 0xca964, 0xca964, 0xca964, 0xca964, 0xca964, 0xca964, 0xca965, 0xca965, 
            0xca965, 0xca965, 0xca965, 0xca965, 0xca965, 0xca965, 0xca965, 0xca965, 0xca965, 0xca965, 
            0xca965, 0xca965, 0xca965, 0xca965, 0xca965, 0xca965, 0xca965, 0xca965, 0xca965, 0xca965, 
            0xca965, 0xca965, 0xca965, 0xca965, 0xca965, 0xca965, 0xca965, 0xca965, 0xca965, 0xca965, 
            0xca970, 0xca970, 0xca971, 0xca971, 0xca972, 0xca972, 0xca972, 0xca972, 0xca973, 0xca973, 
            0xca973, 0xca973, 0xca973, 0xca973, 0xca973, 0xca973, 0xca974, 0xca974, 0xca974, 0xca974, 
            0xca974, 0xca974, 0xca974, 0xca974, 0xca974, 0xca974, 0xca974, 0xca974, 0xca974, 0xca974, 
            0xca974, 0xca974, 0xca975, 0xca975, 0xca975, 0xca975, 0xca975, 0xca975, 0xca975, 0xca975, 
            0xca975, 0xca975, 0xca975, 0xca975, 0xca975, 0xca975, 0xca975, 0xca975, 0xca975, 0xca975, 
            0xca975, 0xca975, 0xca975, 0xca975, 0xca975, 0xca975, 0xca975, 0xca975, 0xca975, 0xca975, 
            0xca975, 0xca975, 0xca975, 0xca975, 0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 
            0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 
            0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 
            0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 
            0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 
            0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 
            0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 0xca976, 0xca980, 0xca980, 
            0xca981, 0xca981, 0xca982, 0xca982, 0xca982, 0xca982, 0xca983, 0xca983, 0xca983, 0xca983, 
            0xca983, 0xca983, 0xca983, 0xca983, 0xca984, 0xca984, 0xca984, 0xca984, 0xca984, 0xca984, 
            0xca984, 0xca984, 0xca984, 0xca984, 0xca984, 0xca984, 0xca984, 0xca984, 0xca984, 0xca984, 
            0xca985, 0xca985, 0xca985, 0xca985, 0xca985, 0xca985, 0xca985, 0xca985, 0xca985, 0xca985, 
            0xca985, 0xca985, 0xca985, 0xca985, 0xca985, 0xca985, 0xca985, 0xca985, 0xca985, 0xca985, 
            0xca985, 0xca985, 0xca985, 0xca985, 0xca985, 0xca985, 0xca985, 0xca985, 0xca985, 0xca985, 
            0xca985, 0xca985, 0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 
            0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 
            0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 
            0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 
            0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 
            0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 
            0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 0xca986, 0xca987, 0xca987, 0xca987, 0xca987, 
            0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 
            0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 
            0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 
            0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 
            0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 
            0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 
            0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 
            0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 
            0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 
            0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 
            0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 
            0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 0xca987, 
            0xca987, 0xca987, 0xca987, 0xca987, 0xcb000, 0xcb000, 0xcb100, 0xcb100, 0xcb200, 0xcb200, 
            0xcb210, 0xcb210, 0xcb300, 0xcb300, 0xcb310, 0xcb310, 0xcb320, 0xcb320, 0xcb321, 0xcb321, 
            0xcb400, 0xcb400, 0xcb410, 0xcb410, 0xcb420, 0xcb420, 0xcb421, 0xcb421, 0xcb430, 0xcb430, 
            0xcb431, 0xcb431, 0xcb432, 0xcb432, 0xcb432, 0xcb432, 0xcb500, 0xcb500, 0xcb510, 0xcb510, 
            0xcb520, 0xcb520, 0xcb521, 0xcb521, 0xcb530, 0xcb530, 0xcb531, 0xcb531, 0xcb532, 0xcb532, 
            0xcb532, 0xcb532, 0xcb540, 0xcb540, 0xcb541, 0xcb541, 0xcb542, 0xcb542, 0xcb542, 0xcb542, 
            0xcb543, 0xcb543, 0xcb543, 0xcb543, 0xcb543, 0xcb543, 0xcb543, 0xcb543, 0xcb600, 0xcb600, 
            0xcb610, 0xcb610, 0xcb620, 0xcb620, 0xcb621, 0xcb621, 0xcb630, 0xcb630, 0xcb631, 0xcb631, 
            0xcb632, 0xcb632, 0xcb632, 0xcb632, 0xcb640, 0xcb640, 0xcb641, 0xcb641, 0xcb642, 0xcb642, 
            0xcb642, 0xcb642, 0xcb643, 0xcb643, 0xcb643, 0xcb643, 0xcb643, 0xcb643, 0xcb643, 0xcb643, 
            0xcb650, 0xcb650, 0xcb651, 0xcb651, 0xcb652, 0xcb652, 0xcb652, 0xcb652, 0xcb653, 0xcb653, 
            0xcb653, 0xcb653, 0xcb653, 0xcb653, 0xcb653, 0xcb653, 0xcb654, 0xcb654, 0xcb654, 0xcb654, 
            0xcb654, 0xcb654, 0xcb654, 0xcb654, 0xcb654, 0xcb654, 0xcb654, 0xcb654, 0xcb654, 0xcb654, 
            0xcb654, 0xcb654, 0xcb700, 0xcb700, 0xcb710, 0xcb710, 0xcb720, 0xcb720, 0xcb721, 0xcb721, 
            0xcb730, 0xcb730, 0xcb731, 0xcb731, 0xcb732, 0xcb732, 0xcb732, 0xcb732, 0xcb740, 0xcb740, 
            0xcb741, 0xcb741, 0xcb742, 0xcb742, 0xcb742, 0xcb742, 0xcb743, 0xcb743, 0xcb743, 0xcb743, 
            0xcb743, 0xcb743, 0xcb743, 0xcb743, 0xcb750, 0xcb750, 0xcb751, 0xcb751, 0xcb752, 0xcb752, 
            0xcb752, 0xcb752, 0xcb753, 0xcb753, 0xcb753, 0xcb753, 0xcb753, 0xcb753, 0xcb753, 0xcb753, 
            0xcb754, 0xcb754, 0xcb754, 0xcb754, 0xcb754, 0xcb754, 0xcb754, 0xcb754, 0xcb754, 0xcb754, 
            0xcb754, 0xcb754, 0xcb754, 0xcb754, 0xcb754, 0xcb754, 0xcb760, 0xcb760, 0xcb761, 0xcb761, 
            0xcb762, 0xcb762, 0xcb762, 0xcb762, 0xcb763, 0xcb763, 0xcb763, 0xcb763, 0xcb763, 0xcb763, 
            0xcb763, 0xcb763, 0xcb764, 0xcb764, 0xcb764, 0xcb764, 0xcb764, 0xcb764, 0xcb764, 0xcb764, 
            0xcb764, 0xcb764, 0xcb764, 0xcb764, 0xcb764, 0xcb764, 0xcb764, 0xcb764, 0xcb765, 0xcb765, 
            0xcb765, 0xcb765, 0xcb765, 0xcb765, 0xcb765, 0xcb765, 0xcb765, 0xcb765, 0xcb765, 0xcb765, 
            0xcb765, 0xcb765, 0xcb765, 0xcb765, 0xcb765, 0xcb765, 0xcb765, 0xcb765, 0xcb765, 0xcb765, 
            0xcb765, 0xcb765, 0xcb765, 0xcb765, 0xcb765, 0xcb765, 0xcb765, 0xcb765, 0xcb765, 0xcb765, 
            0xcb800, 0xcb800, 0xcb810, 0xcb810, 0xcb820, 0xcb820, 0xcb821, 0xcb821, 0xcb830, 0xcb830, 
            0xcb831, 0xcb831, 0xcb832, 0xcb832, 0xcb832, 0xcb832, 0xcb840, 0xcb840, 0xcb841, 0xcb841, 
            0xcb842, 0xcb842, 0xcb842, 0xcb842, 0xcb843, 0xcb843, 0xcb843, 0xcb843, 0xcb843, 0xcb843, 
            0xcb843, 0xcb843, 0xcb850, 0xcb850, 0xcb851, 0xcb851, 0xcb852, 0xcb852, 0xcb852, 0xcb852, 
            0xcb853, 0xcb853, 0xcb853, 0xcb853, 0xcb853, 0xcb853, 0xcb853, 0xcb853, 0xcb854, 0xcb854, 
            0xcb854, 0xcb854, 0xcb854, 0xcb854, 0xcb854, 0xcb854, 0xcb854, 0xcb854, 0xcb854, 0xcb854, 
            0xcb854, 0xcb854, 0xcb854, 0xcb854, 0xcb860, 0xcb860, 0xcb861, 0xcb861, 0xcb862, 0xcb862, 
            0xcb862, 0xcb862, 0xcb863, 0xcb863, 0xcb863, 0xcb863, 0xcb863, 0xcb863, 0xcb863, 0xcb863, 
            0xcb864, 0xcb864, 0xcb864, 0xcb864, 0xcb864, 0xcb864, 0xcb864, 0xcb864, 0xcb864, 0xcb864, 
            0xcb864, 0xcb864, 0xcb864, 0xcb864, 0xcb864, 0xcb864, 0xcb865, 0xcb865, 0xcb865, 0xcb865, 
            0xcb865, 0xcb865, 0xcb865, 0xcb865, 0xcb865, 0xcb865, 0xcb865, 0xcb865, 0xcb865, 0xcb865, 
            0xcb865, 0xcb865, 0xcb865, 0xcb865, 0xcb865, 0xcb865, 0xcb865, 0xcb865, 0xcb865, 0xcb865, 
            0xcb865, 0xcb865, 0xcb865, 0xcb865, 0xcb865, 0xcb865, 0xcb865, 0xcb865, 0xcb870, 0xcb870, 
            0xcb871, 0xcb871, 0xcb872, 0xcb872, 0xcb872, 0xcb872, 0xcb873, 0xcb873, 0xcb873, 0xcb873, 
            0xcb873, 0xcb873, 0xcb873, 0xcb873, 0xcb874, 0xcb874, 0xcb874, 0xcb874, 0xcb874, 0xcb874, 
            0xcb874, 0xcb874, 0xcb874, 0xcb874, 0xcb874, 0xcb874, 0xcb874, 0xcb874, 0xcb874, 0xcb874, 
            0xcb875, 0xcb875, 0xcb875, 0xcb875, 0xcb875, 0xcb875, 0xcb875, 0xcb875, 0xcb875, 0xcb875, 
            0xcb875, 0xcb875, 0xcb875, 0xcb875, 0xcb875, 0xcb875, 0xcb875, 0xcb875, 0xcb875, 0xcb875, 
            0xcb875, 0xcb875, 0xcb875, 0xcb875, 0xcb875, 0xcb875, 0xcb875, 0xcb875, 0xcb875, 0xcb875, 
            0xcb875, 0xcb875, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 
            0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 
            0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 
            0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 
            0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 
            0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 
            0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb876, 0xcb900, 0xcb900, 0xcb910, 0xcb910, 
            0xcb920, 0xcb920, 0xcb921, 0xcb921, 0xcb930, 0xcb930, 0xcb931, 0xcb931, 0xcb932, 0xcb932, 
            0xcb932, 0xcb932, 0xcb940, 0xcb940, 0xcb941, 0xcb941, 0xcb942, 0xcb942, 0xcb942, 0xcb942, 
            0xcb943, 0xcb943, 0xcb943, 0xcb943, 0xcb943, 0xcb943, 0xcb943, 0xcb943, 0xcb950, 0xcb950, 
            0xcb951, 0xcb951, 0xcb952, 0xcb952, 0xcb952, 0xcb952, 0xcb953, 0xcb953, 0xcb953, 0xcb953, 
            0xcb953, 0xcb953, 0xcb953, 0xcb953, 0xcb954, 0xcb954, 0xcb954, 0xcb954, 0xcb954, 0xcb954, 
            0xcb954, 0xcb954, 0xcb954, 0xcb954, 0xcb954, 0xcb954, 0xcb954, 0xcb954, 0xcb954, 0xcb954, 
            0xcb960, 0xcb960, 0xcb961, 0xcb961, 0xcb962, 0xcb962, 0xcb962, 0xcb962, 0xcb963, 0xcb963, 
            0xcb963, 0xcb963, 0xcb963, 0xcb963, 0xcb963, 0xcb963, 0xcb964, 0xcb964, 0xcb964, 0xcb964, 
            0xcb964, 0xcb964, 0xcb964, 0xcb964, 0xcb964, 0xcb964, 0xcb964, 0xcb964, 0xcb964, 0xcb964, 
            0xcb964, 0xcb964, 0xcb965, 0xcb965, 0xcb965, 0xcb965, 0xcb965, 0xcb965, 0xcb965, 0xcb965, 
            0xcb965, 0xcb965, 0xcb965, 0xcb965, 0xcb965, 0xcb965, 0xcb965, 0xcb965, 0xcb965, 0xcb965, 
            0xcb965, 0xcb965, 0xcb965, 0xcb965, 0xcb965, 0xcb965, 0xcb965, 0xcb965, 0xcb965, 0xcb965, 
            0xcb965, 0xcb965, 0xcb965, 0xcb965, 0xcb970, 0xcb970, 0xcb971, 0xcb971, 0xcb972, 0xcb972, 
            0xcb972, 0xcb972, 0xcb973, 0xcb973, 0xcb973, 0xcb973, 0xcb973, 0xcb973, 0xcb973, 0xcb973, 
            0xcb974, 0xcb974, 0xcb974, 0xcb974, 0xcb974, 0xcb974, 0xcb974, 0xcb974, 0xcb974, 0xcb974, 
            0xcb974, 0xcb974, 0xcb974, 0xcb974, 0xcb974, 0xcb974, 0xcb975, 0xcb975, 0xcb975, 0xcb975, 
            0xcb975, 0xcb975, 0xcb975, 0xcb975, 0xcb975, 0xcb975, 0xcb975, 0xcb975, 0xcb975, 0xcb975, 
            0xcb975, 0xcb975, 0xcb975, 0xcb975, 0xcb975, 0xcb975, 0xcb975, 0xcb975, 0xcb975, 0xcb975, 
            0xcb975, 0xcb975, 0xcb975, 0xcb975, 0xcb975, 0xcb975, 0xcb975, 0xcb975, 0xcb976, 0xcb976, 
            0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 
            0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 
            0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 
            0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 
            0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 
            0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 0xcb976, 
            0xcb976, 0xcb976, 0xcb980, 0xcb980, 0xcb981, 0xcb981, 0xcb982, 0xcb982, 0xcb982, 0xcb982, 
            0xcb983, 0xcb983, 0xcb983, 0xcb983, 0xcb983, 0xcb983, 0xcb983, 0xcb983, 0xcb984, 0xcb984, 
            0xcb984, 0xcb984, 0xcb984, 0xcb984, 0xcb984, 0xcb984, 0xcb984, 0xcb984, 0xcb984, 0xcb984, 
            0xcb984, 0xcb984, 0xcb984, 0xcb984, 0xcb985, 0xcb985, 0xcb985, 0xcb985, 0xcb985, 0xcb985, 
            0xcb985, 0xcb985, 0xcb985, 0xcb985, 0xcb985, 0xcb985, 0xcb985, 0xcb985, 0xcb985, 0xcb985, 
            0xcb985, 0xcb985, 0xcb985, 0xcb985, 0xcb985, 0xcb985, 0xcb985, 0xcb985, 0xcb985, 0xcb985, 
            0xcb985, 0xcb985, 0xcb985, 0xcb985, 0xcb985, 0xcb985, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 
            0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 
            0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 
            0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 
            0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 
            0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 
            0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 0xcb986, 
            0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 
            0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 
            0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 
            0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 
            0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 
            0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 
            0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 
            0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 
            0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 
            0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 
            0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 
            0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 
            0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcb987, 0xcba00, 0xcba00, 
            0xcba10, 0xcba10, 0xcba20, 0xcba20, 0xcba21, 0xcba21, 0xcba30, 0xcba30, 0xcba31, 0xcba31, 
            0xcba32, 0xcba32, 0xcba32, 0xcba32, 0xcba40, 0xcba40, 0xcba41, 0xcba41, 0xcba42, 0xcba42, 
            0xcba42, 0xcba42, 0xcba43, 0xcba43, 0xcba43, 0xcba43, 0xcba43, 0xcba43, 0xcba43, 0xcba43, 
            0xcba50, 0xcba50, 0xcba51, 0xcba51, 0xcba52, 0xcba52, 0xcba52, 0xcba52, 0xcba53, 0xcba53, 
            0xcba53, 0xcba53, 0xcba53, 0xcba53, 0xcba53, 0xcba53, 0xcba54, 0xcba54, 0xcba54, 0xcba54, 
            0xcba54, 0xcba54, 0xcba54, 0xcba54, 0xcba54, 0xcba54, 0xcba54, 0xcba54, 0xcba54, 0xcba54, 
            0xcba54, 0xcba54, 0xcba60, 0xcba60, 0xcba61, 0xcba61, 0xcba62, 0xcba62, 0xcba62, 0xcba62, 
            0xcba63, 0xcba63, 0xcba63, 0xcba63, 0xcba63, 0xcba63, 0xcba63, 0xcba63, 0xcba64, 0xcba64, 
            0xcba64, 0xcba64, 0xcba64, 0xcba64, 0xcba64, 0xcba64, 0xcba64, 0xcba64, 0xcba64, 0xcba64, 
            0xcba64, 0xcba64, 0xcba64, 0xcba64, 0xcba65, 0xcba65, 0xcba65, 0xcba65, 0xcba65, 0xcba65, 
            0xcba65, 0xcba65, 0xcba65, 0xcba65, 0xcba65, 0xcba65, 0xcba65, 0xcba65, 0xcba65, 0xcba65, 
            0xcba65, 0xcba65, 0xcba65, 0xcba65, 0xcba65, 0xcba65, 0xcba65, 0xcba65, 0xcba65, 0xcba65, 
            0xcba65, 0xcba65, 0xcba65, 0xcba65, 0xcba65, 0xcba65, 0xcba70, 0xcba70, 0xcba71, 0xcba71, 
            0xcba72, 0xcba72, 0xcba72, 0xcba72, 0xcba73, 0xcba73, 0xcba73, 0xcba73, 0xcba73, 0xcba73, 
            0xcba73, 0xcba73, 0xcba74, 0xcba74, 0xcba74, 0xcba74, 0xcba74, 0xcba74, 0xcba74, 0xcba74, 
            0xcba74, 0xcba74, 0xcba74, 0xcba74, 0xcba74, 0xcba74, 0xcba74, 0xcba74, 0xcba75, 0xcba75, 
            0xcba75, 0xcba75, 0xcba75, 0xcba75, 0xcba75, 0xcba75, 0xcba75, 0xcba75, 0xcba75, 0xcba75, 
            0xcba75, 0xcba75, 0xcba75, 0xcba75, 0xcba75, 0xcba75, 0xcba75, 0xcba75, 0xcba75, 0xcba75, 
            0xcba75, 0xcba75, 0xcba75, 0xcba75, 0xcba75, 0xcba75, 0xcba75, 0xcba75, 0xcba75, 0xcba75, 
            0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 
            0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 
            0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 
            0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 
            0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 
            0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba76, 
            0xcba76, 0xcba76, 0xcba76, 0xcba76, 0xcba80, 0xcba80, 0xcba81, 0xcba81, 0xcba82, 0xcba82, 
            0xcba82, 0xcba82, 0xcba83, 0xcba83, 0xcba83, 0xcba83, 0xcba83, 0xcba83, 0xcba83, 0xcba83, 
            0xcba84, 0xcba84, 0xcba84, 0xcba84, 0xcba84, 0xcba84, 0xcba84, 0xcba84, 0xcba84, 0xcba84, 
            0xcba84, 0xcba84, 0xcba84, 0xcba84, 0xcba84, 0xcba84, 0xcba85, 0xcba85, 0xcba85, 0xcba85, 
            0xcba85, 0xcba85, 0xcba85, 0xcba85, 0xcba85, 0xcba85, 0xcba85, 0xcba85, 0xcba85, 0xcba85, 
            0xcba85, 0xcba85, 0xcba85, 0xcba85, 0xcba85, 0xcba85, 0xcba85, 0xcba85, 0xcba85, 0xcba85, 
            0xcba85, 0xcba85, 0xcba85, 0xcba85, 0xcba85, 0xcba85, 0xcba85, 0xcba85, 0xcba86, 0xcba86, 
            0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 
            0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 
            0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 
            0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 
            0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 
            0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 0xcba86, 
            0xcba86, 0xcba86, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 
            0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 
            0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 
            0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 
            0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 
            0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 
            0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 
            0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 
            0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 
            0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 
            0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 
            0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 
            0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 0xcba87, 
            0xcba90, 0xcba90, 0xcba91, 0xcba91, 0xcba92, 0xcba92, 0xcba92, 0xcba92, 0xcba93, 0xcba93, 
            0xcba93, 0xcba93, 0xcba93, 0xcba93, 0xcba93, 0xcba93, 0xcba94, 0xcba94, 0xcba94, 0xcba94, 
            0xcba94, 0xcba94, 0xcba94, 0xcba94, 0xcba94, 0xcba94, 0xcba94, 0xcba94, 0xcba94, 0xcba94, 
            0xcba94, 0xcba94, 0xcba95, 0xcba95, 0xcba95, 0xcba95, 0xcba95, 0xcba95, 0xcba95, 0xcba95, 
            0xcba95, 0xcba95, 0xcba95, 0xcba95, 0xcba95, 0xcba95, 0xcba95, 0xcba95, 0xcba95, 0xcba95, 
            0xcba95, 0xcba95, 0xcba95, 0xcba95, 0xcba95, 0xcba95, 0xcba95, 0xcba95, 0xcba95, 0xcba95, 
            0xcba95, 0xcba95, 0xcba95, 0xcba95, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 
            0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 
            0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 
            0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 
            0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 
            0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 
            0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba96, 0xcba97, 0xcba97, 
            0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 
            0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 
            0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 
            0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 
            0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 
            0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 
            0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 
            0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 
            0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 
            0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 
            0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 
            0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 
            0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba97, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 
            0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 
            0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 
            0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 
            0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 
            0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 
            0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 
            0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 
            0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 
            0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 
            0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 
            0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 
            0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 
            0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 
            0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 
            0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 
            0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 
            0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 
            0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 
            0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 
            0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 
            0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 
            0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 
            0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 
            0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 
            0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 0xcba98, 
            0xcba98, 0xcba98
	    };
        #endregion

        #region Top Card Table
        /// <exclude/>
        private static readonly System.UInt16[] topCardTable = 
		{ 
			0x0, 0x0, 0x1, 0x1, 0x2, 0x2, 0x2, 0x2, 0x3, 0x3, 
            0x3, 0x3, 0x3, 0x3, 0x3, 0x3, 0x4, 0x4, 0x4, 0x4, 
            0x4, 0x4, 0x4, 0x4, 0x4, 0x4, 0x4, 0x4, 0x4, 0x4, 
            0x4, 0x4, 0x5, 0x5, 0x5, 0x5, 0x5, 0x5, 0x5, 0x5, 
            0x5, 0x5, 0x5, 0x5, 0x5, 0x5, 0x5, 0x5, 0x5, 0x5, 
            0x5, 0x5, 0x5, 0x5, 0x5, 0x5, 0x5, 0x5, 0x5, 0x5, 
            0x5, 0x5, 0x5, 0x5, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 
            0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 
            0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 
            0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 
            0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 
            0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 
            0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x6, 0x7, 0x7, 
            0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 
            0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 
            0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 
            0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 
            0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 
            0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 
            0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 
            0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 
            0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 
            0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 
            0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 
            0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 
            0x7, 0x7, 0x7, 0x7, 0x7, 0x7, 0x8, 0x8, 0x8, 0x8, 
            0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 
            0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 
            0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 
            0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 
            0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 
            0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 
            0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 
            0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 
            0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 
            0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 
            0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 
            0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 
            0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 
            0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 
            0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 
            0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 
            0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 
            0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 
            0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 
            0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 
            0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 
            0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 
            0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 
            0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 
            0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 
            0x8, 0x8, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 0x9, 
            0x9, 0x9, 0x9, 0x9, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 
            0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xa, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 
            0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 0xc, 
            0xc, 0xc
		};
        #endregion

        #region CardMask
        
        /// <summary>
        /// This table is equivalent to 1UL left shifted by the index.
        /// The lookup is faster than the left shift operator.
        /// </summary>
        public static readonly ulong[] CardMasksTable = 
		{ 
			0x1,
			0x2,
			0x4,
			0x8,
			0x10,
			0x20,
			0x40,
			0x80,
			0x100,
			0x200,
			0x400,
			0x800,
			0x1000,
			0x2000,
			0x4000,
			0x8000,
			0x10000,
			0x20000,
			0x40000,
			0x80000,
			0x100000,
			0x200000,
			0x400000,
			0x800000,
			0x1000000,
			0x2000000,
			0x4000000,
			0x8000000,
			0x10000000,
			0x20000000,
			0x40000000,
			0x80000000,
			0x100000000,
			0x200000000,
			0x400000000,
			0x800000000,
			0x1000000000,
			0x2000000000,
			0x4000000000,
			0x8000000000,
			0x10000000000,
			0x20000000000,
			0x40000000000,
			0x80000000000,
			0x100000000000,
			0x200000000000,
			0x400000000000,
			0x800000000000,
			0x1000000000000,
			0x2000000000000,
			0x4000000000000,
			0x8000000000000,
		};
        /// <exclude/>
        //public static readonly int CardMasksTableSize = 52; //CardMasksTable.Length;
        public const int CardMasksTableSize = 52;
        #endregion

        #region Card Table
        // converts card number into the equivalent text string.
        /// <exclude/>
        public static readonly string[] CardTable =
		{
			"2c", "3c", "4c", "5c", "6c", "7c", "8c", "9c", "Tc", "Jc", "Qc", "Kc", "Ac",
			"2d", "3d", "4d", "5d", "6d", "7d", "8d", "9d", "Td", "Jd", "Qd", "Kd", "Ad",
			"2h", "3h", "4h", "5h", "6h", "7h", "8h", "9h", "Th", "Jh", "Qh", "Kh", "Ah",
			"2s", "3s", "4s", "5s", "6s", "7s", "8s", "9s", "Ts", "Js", "Qs", "Ks", "As",
		};

        // Converts card number into the card rank text string
        /// <exclude/>
        private static readonly string[] ranktbl =
		{
			"Two", "Three", "Four", "Five", "Six", "Seven", "Eight", "Nine", "Ten", "Jack", "Queen", "King", "Ace",
			"Two", "Three", "Four", "Five", "Six", "Seven", "Eight", "Nine", "Ten", "Jack", "Queen", "King", "Ace",
			"Two", "Three", "Four", "Five", "Six", "Seven", "Eight", "Nine", "Ten", "Jack", "Queen", "King", "Ace",
			"Two", "Three", "Four", "Five", "Six", "Seven", "Eight", "Nine", "Ten", "Jack", "Queen", "King", "Ace"
		};

        // Converts card number into the card suit text string
        /// <exclude/>
        private static readonly string[] suittbl =
		{
			"Clubs", "Clubs", "Clubs", "Clubs", "Clubs", "Clubs", "Clubs", "Clubs", "Clubs", "Clubs", "Clubs", "Clubs", "Clubs",
			"Diamonds", "Diamonds", "Diamonds", "Diamonds", "Diamonds", "Diamonds", "Diamonds", "Diamonds", "Diamonds", "Diamonds", "Diamonds", "Diamonds", "Diamonds",
			"Hearts", "Hearts", "Hearts", "Hearts", "Hearts", "Hearts", "Hearts", "Hearts", "Hearts", "Hearts", "Hearts", "Hearts", "Hearts",
			"Spades", "Spades", "Spades", "Spades", "Spades", "Spades", "Spades", "Spades", "Spades", "Spades", "Spades", "Spades", "Spades",
		};

        // Converts card number into the card rank char
        /// <exclude/>
        private static readonly char[] rankchar =
		{
			'2', '3', '4', '5', '6', '7', '8', '9', 'T', 'J', 'Q', 'K', 'A',
			'2', '3', '4', '5', '6', '7', '8', '9', 'T', 'J', 'Q', 'K', 'A',
			'2', '3', '4', '5', '6', '7', '8', '9', 'T', 'J', 'Q', 'K', 'A',
			'2', '3', '4', '5', '6', '7', '8', '9', 'T', 'J', 'Q', 'K', 'A',
		};

        // Converts card number into the card suit text string
        /// <exclude/>
        private static readonly char[] suitchar =
		{
			'c', 'c', 'c', 'c', 'c', 'c', 'c', 'c', 'c', 'c', 'c', 'c', 'c',
			'd', 'd', 'd', 'd', 'd', 'd', 'd', 'd', 'd', 'd', 'd', 'd', 'd',
			'h', 'h', 'h', 'h', 'h', 'h', 'h', 'h', 'h', 'h', 'h', 'h', 'h',
			's', 's', 's', 's', 's', 's', 's', 's', 's', 's', 's', 's', 's',
		};
        #endregion

        #endregion
    }
    #endregion
}
