using System;
using System.Collections;
using System.ComponentModel;
using System.Diagnostics;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

using System.IO;
using System.Reflection;
using System.Runtime.Remoting;
using System.Runtime.Remoting.Channels;
using System.Runtime.Remoting.Messaging;
using System.Threading;

using Shared.Channels;
using Shared.Channels.GenuineTcp;
using Shared.Channels.BroadcastEngine;
using Shared.Channels.DotNetRemotingLayer;

namespace Console.Debugger
{
    public partial class DialogMain : Form, IBroadcastMessenger
    {
        private delegate void ViewBroadcastHeartBeatCallback(Object message, String nickname);
        private static Shared.XML.SnapShot.Objects.Service.Trampoline trampolineConfig;

        private static String Nickname = Assembly.GetExecutingAssembly().GetName().Name;
        private static IBroadcastIt BroadcastIt;
        private static IBroadcastServer iBroadcastServer;
        private static Object IBroadcastItLock = new Object();

        private static DialogProgressBar progress;
        private static Boolean DebugDelegate = false;
        private static Boolean DebugLogic = false;
        private static Boolean DebugSnapShot = false;

        public DialogMain()
        {
            InitializeComponent();
            PopulateTrampolinedProcesses();
            PopulateProcessToolStripMenu();

            try
            {
                // Setup remoting with assembly objects as remote object.
                IDictionary props = new Hashtable();
                props["name"] = "gtcpd";
                props["prefix"] = "gtcpd";
                props["priority"] = "100";
                props["port"] = "0";

                GenuineTcpChannel channel = new GenuineTcpChannel(props, null, null);
                ChannelServices.RegisterChannel(channel, false);
                WellKnownClientTypeEntry remotetype = new WellKnownClientTypeEntry(typeof(Shared.Channels.GenuineTcp.GenuineTcpChannel), "gtcpd://127.0.0.1:48889/Messenger");
                RemotingConfiguration.RegisterWellKnownClientType(remotetype);

                // Bind client's receiver.
                RemotingServices.Marshal(this, "Messenger");

                // Subscribe to the chat event.
                lock (IBroadcastItLock)
                {
                    iBroadcastServer = (IBroadcastServer)Activator.GetObject(typeof(IBroadcastIt), "gtcpd://127.0.0.1:48889/Server");
                    BroadcastIt = iBroadcastServer.JoinDialog(Nickname);
                }
            }
            catch (Exception ex)
            {
                EventLog evtLog = new EventLog("Application", ".", "Ludus Delegate");
                StringBuilder evtEntry = new StringBuilder("Console::Debugger::DialogMain()");
                evtEntry.Append(Environment.NewLine + String.Format("Exception: {0}. Stack trace: {1}.", ex.Message, ex.StackTrace));
                evtLog.WriteEntry(evtEntry.ToString(), EventLogEntryType.Error);
                evtLog.Close();

                DialogNotice notice = new DialogNotice(String.Format("Exception: {0}. Stack trace: {1}.", ex.Message, ex.StackTrace));
                notice.ShowDialog(this);
            }

            DisplayProgressBar(3000, "Synchronizing with Ludus services...");
        }
        public void ViewConsoleHeartBeat(Object message, String nickname)
        {
            try
            {
                if (txtConsole.Text.Length > txtConsole.MaxLength) { txtConsole.Clear(); }

                if (message.GetType().Equals(typeof(Shared.Console.Messages.Client)))
                {
                    Shared.Console.Messages.Client msg = (Shared.Console.Messages.Client)message;
                    txtConsole.Text += Environment.NewLine + msg.action + " " + msg.pid + " " + msg.hwnd
                        + " monitor=\"" + msg.allowmonitor.ToString().ToLower() + "\""
                        + " play=\"" + msg.allowplay.ToString().ToLower() + "\""
                        + " profile=\"" + msg.profile.ToLower() + "\"";
                }

                txtConsole.SelectionStart = txtConsole.Text.Length;
                txtConsole.ScrollToCaret();
            }
            catch (Exception ex)
            {
                EventLog evtLog = new EventLog("Application", ".", "Ludus Delegate");
                StringBuilder evtEntry = new StringBuilder("Console::Debugger::ViewConsoleHeartBeat()");
                evtEntry.Append(Environment.NewLine + String.Format("Exception: {0}. Stack trace: {1}.", ex.Message, ex.StackTrace));
                evtLog.WriteEntry(evtEntry.ToString(), EventLogEntryType.Error);
                evtLog.Close();
            }
        }
        public void ViewDelegateHeartBeat(Object message, String nickname)
        {
            try
            {
                if (txtDelegate.Text.Length > txtDelegate.MaxLength) { txtDelegate.Clear(); }

                if (message.GetType().Equals(typeof(Shared.Delegate.Messages.Element)))
                {
                    Shared.Delegate.Messages.Element msg = (Shared.Delegate.Messages.Element)message;
                    txtDelegate.Text += Environment.NewLine + msg.action + " " + msg.pid + " " + msg.hwnd
                        + " " + msg.chwnd + " "
                        + "[" + msg.xy + "]"
                        + " " + msg.id + " "
                        + "\"" + msg.value + "\"";
                }
                else if (message.GetType().Equals(typeof(Shared.Delegate.Messages.Client)))
                {
                    Shared.Delegate.Messages.Client msg = (Shared.Delegate.Messages.Client)message;
                    txtDelegate.Text += Environment.NewLine + msg.action + " " + msg.pid + " " + msg.hwnd
                        + " monitor=\"" + msg.allowmonitor.ToString().ToLower() + "\""
                        + " play=\"" + msg.allowplay.ToString().ToLower() + "\"";
                }

                txtDelegate.SelectionStart = txtDelegate.Text.Length;
                txtDelegate.ScrollToCaret();
            }
            catch (Exception ex)
            {
                EventLog evtLog = new EventLog("Application", ".", "Ludus Delegate");
                StringBuilder evtEntry = new StringBuilder("Console::Debugger::ViewDelegateHeartBeat()");
                evtEntry.Append(Environment.NewLine + String.Format("Exception: {0}. Stack trace: {1}.", ex.Message, ex.StackTrace));
                evtLog.WriteEntry(evtEntry.ToString(), EventLogEntryType.Error);
                evtLog.Close();
            }
        }
        public void ViewLogicHeartBeat(Object message, String nickname)
        {
            try
            {
                if (txtLogic.Text.Length > txtLogic.MaxLength) { txtLogic.Clear(); }

                if (message.GetType().Equals(typeof(Shared.Logic.Poker.Messages.State)))
                {
                    Shared.Logic.Poker.Messages.State msg = (Shared.Logic.Poker.Messages.State)message;

                    String playercards = String.Empty;
                    foreach (String s in msg.playercards)
                    {
                        playercards += s + " ";
                    }

                    if (msg.tablecards.Count > 0)
                    {
                        String tablecards = String.Empty;
                        foreach (String s in msg.tablecards)
                        {
                            tablecards += s + " ";
                        }

                        txtLogic.Text += Environment.NewLine + msg.action + " " + msg.pid + " " + msg.hwnd
                            + " (" + msg.playersactive + "x)"
                            + " [" + tablecards.Trim() + "]"
                            + " [" + playercards.Trim() + "]"
                            + " " + Math.Round(msg.playerchances, 2) + "%";
                    }
                    else
                    {
                        txtLogic.Text += Environment.NewLine + msg.action + " " + msg.pid + " " + msg.hwnd
                            + " (" + msg.playersactive + "x)"
                            + " [" + playercards.Trim() + "] "
                            + " " + Math.Round(msg.playerchances, 2) + "%";
                    }
                }
                else if (message.GetType().Equals(typeof(Shared.Logic.Messages.Element)))
                {
                    Shared.Logic.Messages.Element msg = (Shared.Logic.Messages.Element)message;

                    if (msg.amount.Equals(0))
                    {
                        txtLogic.Text += Environment.NewLine + msg.action + " " + msg.pid + " " + msg.hwnd
                            + " decision=\"" + msg.value + "\"";
                    }
                    else
                    {
                        txtLogic.Text += Environment.NewLine + msg.action + " " + msg.pid + " " + msg.hwnd
                            + " decision=\"" + msg.value + " " + msg.amount + "\"";
                    }
                }
                else if (message.GetType().Equals(typeof(Shared.Logic.Messages.Client)))
                {
                    Shared.Logic.Messages.Client msg = (Shared.Logic.Messages.Client)message;

                    String tablebuttons = String.Empty;
                    foreach (Object s in msg.layout.Keys)
                    {
                        tablebuttons += "[" + ((Shared.Logic.Messages.Element)msg.layout[s]).value;
                        if (!((Shared.Logic.Messages.Element)msg.layout[s]).amount.Equals(0))
                        {
                            tablebuttons += " " + ((Shared.Logic.Messages.Element)msg.layout[s]).amount;
                        }
                        tablebuttons += "] ";
                    }

                    txtLogic.Text += Environment.NewLine + msg.action + " " + msg.pid + " " + msg.hwnd
                        + " monitor=\"" + msg.allowmonitor.ToString().ToLower() + "\""
                        + " play=\"" + msg.allowplay.ToString().ToLower() + "\""
                        + " profile=\"" + msg.profile.ToLower() + "\""
                        + " " + tablebuttons;
                }

                txtLogic.SelectionStart = txtLogic.Text.Length;
                txtLogic.ScrollToCaret();
            }
            catch (Exception ex)
            {
                EventLog evtLog = new EventLog("Application", ".", "Ludus Delegate");
                StringBuilder evtEntry = new StringBuilder("Console::Debugger::ViewLogicHeartBeat()");
                evtEntry.Append(Environment.NewLine + String.Format("Exception: {0}. Stack trace: {1}.", ex.Message, ex.StackTrace));
                evtLog.WriteEntry(evtEntry.ToString(), EventLogEntryType.Error);
                evtLog.Close();
            }
        }
        public void ViewSnapShotHeartBeat(Object message, String nickname)
        {
            try
            {
                if (txtSnapShot.Text.Length > txtSnapShot.MaxLength) { txtSnapShot.Clear(); }

                if (message.GetType().Equals(typeof(Shared.SnapShot.Messages.Element)))
                {
                    Shared.SnapShot.Messages.Element msg = (Shared.SnapShot.Messages.Element)message;
                    txtSnapShot.Text += Environment.NewLine + msg.action + " " + msg.pid + " " + msg.hwnd 
                        + " " + msg.chwnd + " "
                        + "[" + msg.xy + "]"
                        + " " + msg.id + " "
                        + "\"" + msg.value + "\"";
                }
                else if (message.GetType().Equals(typeof(Shared.SnapShot.Messages.Client)))
                {
                    Shared.SnapShot.Messages.Client msg = (Shared.SnapShot.Messages.Client)message;
                    txtSnapShot.Text += Environment.NewLine + msg.action + " " + msg.pid + " " + msg.hwnd 
                        + " \"" + msg.caption + "\"";
                }
                else if (message.GetType().Equals(typeof(Shared.SnapShot.Messages.Process)))
                {
                    Shared.SnapShot.Messages.Process msg = (Shared.SnapShot.Messages.Process)message;
                    txtSnapShot.Text += Environment.NewLine + msg.action + " " + msg.pid 
                        + " \"" + msg.name + "\"";
                }

                txtSnapShot.SelectionStart = txtSnapShot.Text.Length;
                txtSnapShot.ScrollToCaret();
            }
            catch (Exception ex)
            {
                EventLog evtLog = new EventLog("Application", ".", "Ludus Delegate");
                StringBuilder evtEntry = new StringBuilder("Console::Debugger::ViewSnapShotHeartBeat()");
                evtEntry.Append(Environment.NewLine + String.Format("Exception: {0}. Stack trace: {1}.", ex.Message, ex.StackTrace));
                evtLog.WriteEntry(evtEntry.ToString(), EventLogEntryType.Error);
                evtLog.Close();
            }
        }
        public Object ReceiveMessage(Object message, String nickname)
        {
            if (message.GetType().Equals(typeof(Shared.SnapShot.Messages.Element))
                || message.GetType().Equals(typeof(Shared.SnapShot.Messages.Client))
                || message.GetType().Equals(typeof(Shared.SnapShot.Messages.Process)))
            {
                if (txtSnapShot.IsHandleCreated)
                {
                    try
                    {
                        txtSnapShot.Invoke(new ViewBroadcastHeartBeatCallback(this.ViewSnapShotHeartBeat), new Object[] { message, nickname });
                    }
                    catch (Exception ex)
                    {
                        EventLog evtLog = new EventLog("Application", ".", "Ludus Delegate");
                        StringBuilder evtEntry = new StringBuilder("Console::Debugger::DialogMain::ReceiveMessage().txtSnapShot");
                        evtEntry.Append(Environment.NewLine + String.Format("Exception: {0}. Stack trace: {1}.", ex.Message, ex.StackTrace));
                        evtLog.WriteEntry(evtEntry.ToString(), EventLogEntryType.Error);
                        evtLog.Close();
                    }
                }
            }
            else if (message.GetType().Equals(typeof(Shared.Logic.Poker.Messages.State))
                || message.GetType().Equals(typeof(Shared.Logic.Messages.Element))
                || message.GetType().Equals(typeof(Shared.Logic.Messages.Client)))
            {
                if (txtLogic.IsHandleCreated)
                {
                    try
                    {
                        txtLogic.Invoke(new ViewBroadcastHeartBeatCallback(this.ViewLogicHeartBeat), new Object[] { message, nickname });
                    }
                    catch (Exception ex)
                    {
                        EventLog evtLog = new EventLog("Application", ".", "Ludus Delegate");
                        StringBuilder evtEntry = new StringBuilder("Console::Debugger::DialogMain::ReceiveMessage().txtLogic");
                        evtEntry.Append(Environment.NewLine + String.Format("Exception: {0}. Stack trace: {1}.", ex.Message, ex.StackTrace));
                        evtLog.WriteEntry(evtEntry.ToString(), EventLogEntryType.Error);
                        evtLog.Close();
                    }
                }
            }
            else if (message.GetType().Equals(typeof(Shared.Delegate.Messages.Element))
                || message.GetType().Equals(typeof(Shared.Delegate.Messages.Client))
                || message.GetType().Equals(typeof(Shared.Delegate.Messages.Process)))
            {
                if (txtDelegate.IsHandleCreated)
                {
                    try
                    {
                        txtDelegate.Invoke(new ViewBroadcastHeartBeatCallback(this.ViewDelegateHeartBeat), new Object[] { message, nickname });
                    }
                    catch (Exception ex)
                    {
                        EventLog evtLog = new EventLog("Application", ".", "Ludus Delegate");
                        StringBuilder evtEntry = new StringBuilder("Console::Debugger::DialogMain::ReceiveMessage().txtDelegate");
                        evtEntry.Append(Environment.NewLine + String.Format("Exception: {0}. Stack trace: {1}.", ex.Message, ex.StackTrace));
                        evtLog.WriteEntry(evtEntry.ToString(), EventLogEntryType.Error);
                        evtLog.Close();
                    }
                }
            }
            else if (message.GetType().Equals(typeof(Shared.Console.Messages.Client)))
            {
                if (txtConsole.IsHandleCreated)
                {
                    try
                    {
                        txtConsole.Invoke(new ViewBroadcastHeartBeatCallback(this.ViewConsoleHeartBeat), new Object[] { message, nickname });
                    }
                    catch (Exception ex)
                    {
                        EventLog evtLog = new EventLog("Application", ".", "Ludus Delegate");
                        StringBuilder evtEntry = new StringBuilder("Console::Debugger::DialogMain::ReceiveMessage().txtConsole");
                        evtEntry.Append(Environment.NewLine + String.Format("Exception: {0}. Stack trace: {1}.", ex.Message, ex.StackTrace));
                        evtLog.WriteEntry(evtEntry.ToString(), EventLogEntryType.Error);
                        evtLog.Close();
                    }
                }
            }
            return null;
        }

        public override Object InitializeLifetimeService()
        {
            return null;
        }
        private void PopulateTrampolinedProcesses()
        {
            trampolineConfig = Shared.XML.SnapShot.Reader.TrampolineConfig();
        }
        private void PopulateProcessToolStripMenu()
        {
            foreach (String process in Shared.XML.SnapShot.Reader.ProcessesAvailableConfig())
            {
                ToolStripMenuItem menu = new ToolStripMenuItem(process, null, new EventHandler(processToolStripMenuItem_Click), process);
                processesToolStripMenuItem.DropDownItems.Add(menu);
            }
        }
        private void processToolStripMenuItem_Click(object sender, EventArgs e)
        {
            ToolStripItem menu = (ToolStripItem)sender;
            String cmd = String.Empty;
            String exe = Shared.XML.SnapShot.Reader.ProcessesAvailableLocationConfig(menu.Name);
            String exeargs = Shared.XML.SnapShot.Reader.ProcessesAvailableArgsConfig(menu.Name);

            if (exeargs.Equals(String.Empty)) { cmd = "PEREWRITE_PRELOAD|" + exe; }
            else { cmd = "PEREWRITE_PRELOAD|" + exe + "| " + exeargs; }

            if (Shared.SnapShot.Trampoline.Client(trampolineConfig.path, trampolineConfig.buffersize, cmd).Contains("ERROR"))
            {
                DialogNotice notice = new DialogNotice("Failure to spawn process \"" + exe + "\""
                    + Environment.NewLine + "Make sure all Ludus related services are started.");
                notice.ShowDialog(this);
            }
            else
            {
                DisplayProgressBar(10000, "Spawning \"" + exe + "\"");
            }
        }

        private void txtDelegate_DoubleClick(object sender, EventArgs e)
        {
            if (DebugDelegate)
            {
                timerProgress.Start();
                txtDelegate.ForeColor = Color.Black;
                progress = new DialogProgressBar("Debugger turned off for \"Delegate\" service...");
                Shared.Delegate.Messages.Debugger message = new Shared.Delegate.Messages.Debugger(false);
                BroadcastIt.SendMessage((Object)message);
                DebugDelegate = false;

                progress.ShowDialog(this);
            }
            else
            {
                timerProgress.Start();
                txtDelegate.ForeColor = Color.Blue;
                progress = new DialogProgressBar("Debugger turned on for \"Delegate\" service...");
                Shared.Delegate.Messages.Debugger message = new Shared.Delegate.Messages.Debugger(true);
                BroadcastIt.SendMessage((Object)message);
                DebugDelegate = true;

                progress.ShowDialog(this);
            }
        }

        private void txtLogic_DoubleClick(object sender, EventArgs e)
        {
            if (DebugLogic)
            {
                timerProgress.Start();
                txtLogic.ForeColor = Color.Black;
                progress = new DialogProgressBar("Debugger turned off for \"Logic\" service...");
                Shared.Logic.Messages.Debugger message = new Shared.Logic.Messages.Debugger(false);
                BroadcastIt.SendMessage((Object)message);
                DebugLogic = false;

                progress.ShowDialog(this);
            }
            else
            {
                timerProgress.Start();
                txtLogic.ForeColor = Color.Blue;
                progress = new DialogProgressBar("Debugger turned on for \"Logic\" service...");
                Shared.Logic.Messages.Debugger message = new Shared.Logic.Messages.Debugger(true);
                BroadcastIt.SendMessage((Object)message);
                DebugLogic = true;

                progress.ShowDialog(this);
            }
        }
        private void txtSnapShot_DoubleClick(object sender, EventArgs e)
        {
            if (DebugSnapShot)
            {
                timerProgress.Start();
                txtSnapShot.ForeColor = Color.Black;
                progress = new DialogProgressBar("Debugger turned off for \"SnapShot\" service...");
                Shared.SnapShot.Messages.Debugger message = new Shared.SnapShot.Messages.Debugger(true, true, true);
                BroadcastIt.SendMessage((Object)message);
                DebugSnapShot = false;

                progress.ShowDialog(this);
            }
            else
            {
                timerProgress.Start();
                txtSnapShot.ForeColor = Color.Blue;
                progress = new DialogProgressBar("Debugger turned on for \"SnapShot\" service...");
                Shared.SnapShot.Messages.Debugger message = new Shared.SnapShot.Messages.Debugger(false, false, false);
                BroadcastIt.SendMessage((Object)message);
                DebugSnapShot = true;

                progress.ShowDialog(this);
            }
        }
        private void DisplayProgressBar(Int32 duration, String display)
        {
            // http://social.msdn.microsoft.com/Forums/en-US/csharpgeneral/thread/a5c28385-f5be-46f6-916f-7bbf17efbdc0
            timerProgress.Interval = duration;
            timerProgress.Start();

            progress = new DialogProgressBar(display);
            progress.ShowDialog(this);
        }
        private void timerProgress_Tick(object sender, EventArgs e)
        {
            progress.Close();
            timerProgress.Stop();
        }

        private void exitToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Close();
        }
    }
}