/*
FILE: DungeonMap.java
AUTHOR: Trevor Evans
CREATED: 2-21-06
MODIFIED: 4-24-06
*/

import javax.imageio.*;
import java.awt.*;
import java.io.*;
import java.util.*;

/*
CLASS: DungeonMap
PURPOSE: Creates a randomly generated maze from dungeon wall and dungeon floor tiles.
*/
public class DungeonMap extends Map
{
	private Image floorImage; //image of a dungeon floor (room) tile
	private Image entranceImage; //image of a maze entrance tile
	private Image exitImage; //image of a maze exit tile
	private Image chestImage; //image of a treasure chest tile
	private ArrayList wallList; //a list of wall (impassable) tiles
	private ArrayList roomList; //a list of room (passable) tiles
	private TreasureChest[] chest; //an array of treasure chest objects
	private int random; //used by the random maze generator to create the maze

	public static final int TOP = 0; //designates the wall directly above a given room tile
	public static final int RIGHT = 1; //designates the wall to the right of a given room tile
	public static final int BOTTOM = 2; //designates the wall directly below a given room tile
	public static final int LEFT = 3; //designates the wall to the left of a given room tile
	public static final int TOP_BUFFER = 3;   //Dungeon maps are buffered on top by 3 rows of black tiles,
	public static final int RIGHT_BUFFER = 3; //on the right by 3 columns of black tiles,
	public static final int BOTTOM_BUFFER = 3;//on bottom by 3 rows of black tiles, and
	public static final int LEFT_BUFFER = 4;  //on the left by 4 columns of black tiles.

	/*
	METHOD: DungeonMap(int, int)
	PURPOSE: Constructs a randomly generated maze using a randomized version of Prim's alogrithm:
	1. Pick a cell, mark it as part of the maze. Add the walls of the cell to the wall list.
	2. While there are walls in the list:
	   3. Pick a random wall from the list, and make it a passage.
	   4. Mark the cell on the opposite side as part of the maze.
	   5. Add the neighboring walls of the cell to the wall list.
	PARAMETERS:
		int level -> this map's level
		int levelChange -> designates where the hero came from:
						   -1 indicates the floor below it
						   +1 indicates the floor above it
	*/
	public DungeonMap(int level, int levelChange)
	{
		super();
		//load all images
		try
		{
			floorImage = ImageIO.read(new File("image_files/cave_floor.gif"));
			entranceImage = ImageIO.read(new File("image_files/stairs_up.gif"));
			exitImage = ImageIO.read(new File("image_files/stairs_down.gif"));
			chestImage = ImageIO.read(new File("image_files/chest.gif"));
			if(level >=  1 && level <= 10)
			{
				image = ImageIO.read(new File("image_files/dungeon_floor_10x10.gif"));
			}
			else if(level > 10 && level <= 20)
			{
				image = ImageIO.read(new File("image_files/dungeon_floor_20x20.gif"));
			}
			else if(level > 20 && level <= 30)
			{
				image = ImageIO.read(new File("image_files/dungeon_floor_30x30.gif"));
			}
			else if(level > 30 && level <= 40)
			{
				image = ImageIO.read(new File("image_files/dungeon_floor_40x40.gif"));
			}
			else if(level > 40 && level <= 50)
			{
				image = ImageIO.read(new File("image_files/dungeon_floor_50x50.gif"));
			}
		}
		catch (IOException exception)
		{
			System.out.println("Caught exception: " + exception);
		}
		//get maze height and width in pixels
		width = image.getWidth(null) / Tile.TILE_SIZE;
		height = image.getHeight(null) / Tile.TILE_SIZE;
		//construct arrays
		tile = new MazeTile[width][height];
		wallList = new ArrayList();
		roomList = new ArrayList();
		//initialize tile array
		for(int i = 0; i < width; i++)
		{
			for(int j = 0; j < height; j++)
			{
				tile[i][j] = new MazeTile(i, j);
			}
		}
		//for every column of tiles inside the left and right buffers...
		for(int i = LEFT_BUFFER; i < (width - RIGHT_BUFFER); i++)
		{	//for every row of tiles inside the top and bottom buffers...
			for(int j = TOP_BUFFER; j < (height - BOTTOM_BUFFER); j++)
			{
				if(i % 2 == 0) //every other column is a column of wall tiles
					tile[i][j].setPassable(false);
				else if(j % 2 == 1) //every other row is a row of wall tiles
					tile[i][j].setPassable(false);
				else //all other tiles are rooms
				{
					roomList.add(tile[i][j]);
				}
			}
		}
		//construct an array of treasure chest objects of a size determined by the current level
		//levels 1-10: 1 chest, levels 11-20: 2 chests, levels 21-30: 3 chests, and so on
 		chest = new TreasureChest[((level - 1) / 10) + 1];
		//randomly place the appropriate number of treasure chests
		for(int i = 0; i <= (level - 1) / 10; i++)
		{
			random = (int) (Math.random() * roomList.size());
			chest[i] = new TreasureChest(level, ((MazeTile) roomList.get(random)).getColumn(), ((MazeTile) roomList.get(random)).getRow());
			((MazeTile) tile[((MazeTile) roomList.get(random)).getColumn()][((MazeTile) roomList.get(random)).getRow()]).setTreasureChest(true);
			((MazeTile) tile[((MazeTile) roomList.get(random)).getColumn()][((MazeTile) roomList.get(random)).getRow()]).setPassable(false);
			roomList.remove(random);
		}
		//randomly place exit tile
		random = (int) (Math.random() * roomList.size());
		((MazeTile) tile[((MazeTile) roomList.get(random)).getColumn()][((MazeTile) roomList.get(random)).getRow()]).setExit(true);
		if(levelChange < 0) //if the hero moved up a floor
		{	//set the hero's starting location to the exit
			heroColumn = ((MazeTile) roomList.get(random)).getColumn();
			heroRow  = ((MazeTile) roomList.get(random)).getRow();
		}
		roomList.remove(random);
		//randomly place entrance tile, use as the starting point for the maze.
		random = (int) (Math.random() * roomList.size());
		((MazeTile) tile[((MazeTile) roomList.get(random)).getColumn()][((MazeTile) roomList.get(random)).getRow()]).setVisited(true);
		((MazeTile) tile[((MazeTile) roomList.get(random)).getColumn()][((MazeTile) roomList.get(random)).getRow()]).setEntrance(true);
		if(levelChange > 0) //if the hero moved down a floor
		{	//set the hero's starting location to the entrance
			heroColumn = ((MazeTile) roomList.get(random)).getColumn();
			heroRow  = ((MazeTile) roomList.get(random)).getRow();
		}
		//if the chosen room is not in the top row of rooms...
		if(((MazeTile) roomList.get(random)).getRow() != (TOP_BUFFER + 1))
		{	//...add the wall directly above it to the wall list
			wallList.add(tile[((MazeTile) roomList.get(random)).getColumn()][((MazeTile) roomList.get(random)).getRow() - 1]);
		}
		//if the chosen room is not in the right-most column of rooms...
		if(((MazeTile) roomList.get(random)).getColumn() != (width - (RIGHT_BUFFER + 2)))
		{	//...add the wall to the right of it to the wall list
			wallList.add(tile[((MazeTile) roomList.get(random)).getColumn() + 1][((MazeTile) roomList.get(random)).getRow()]);
		}
		//if the chosen room is not in the bottom row of rooms...
		if(((MazeTile) roomList.get(random)).getRow() != (height - (BOTTOM_BUFFER + 2)))
		{	//...add the wall directly below it to the wall list
			wallList.add(tile[((MazeTile) roomList.get(random)).getColumn()][((MazeTile) roomList.get(random)).getRow() + 1]);
		}
		//if the chosen room is not in the left-most column of rooms...
		if(((MazeTile) roomList.get(random)).getColumn() != (LEFT_BUFFER + 1))
		{	//...add the wall to the left of it to the wall list
			wallList.add(tile[((MazeTile) roomList.get(random)).getColumn() - 1][((MazeTile) roomList.get(random)).getRow()]);
		}
		//while there are walls in the list...
		while(!wallList.isEmpty())
		{	//randomly chose a wall from the list
			random = (int) (Math.random() * wallList.size());
			//if chosen wall is in an even column...
			if(((MazeTile) wallList.get(random)).getColumn() % 2 == 0)
			{	//if the rooms to the left and right of the chosen wall have already been visited...
				if(((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() - 1][((MazeTile) wallList.get(random)).getRow()]).hasBeenVisited() &&
				   ((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() + 1][((MazeTile) wallList.get(random)).getRow()]).hasBeenVisited())
				{
					//do nothing
				}
				//if the room to the left of the chosen wall has not yet been visited...
				else if(!((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() - 1][((MazeTile) wallList.get(random)).getRow()]).hasBeenVisited())
				{	//...mark the room as visited and make the wall a passage
					((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() - 1][((MazeTile) wallList.get(random)).getRow()]).setVisited(true);
					((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn()][((MazeTile) wallList.get(random)).getRow()]).setPassable(true);
					//if the room is not in the top row of rooms...
					if(((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() - 1][((MazeTile) wallList.get(random)).getRow()]).getRow() != (TOP_BUFFER + 1))
					{	//if the wall directly above the room is not already in the wall list...
						if(!((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() - 1][((MazeTile) wallList.get(random)).getRow() - 1]).isInWallList())
						{	//...add the wall to the wall list
							wallList.add(((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() - 1][((MazeTile) wallList.get(random)).getRow() - 1]));
						}
					}
					//if the room is not in the bottom row of rooms...
					if(((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() - 1][((MazeTile) wallList.get(random)).getRow()]).getRow() != (height - (BOTTOM_BUFFER + 2)))
					{	//if the wall directly below the room is not already in the wall list...
						if(!((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn()][((MazeTile) wallList.get(random)).getRow() + 1]).isInWallList())
						{	//...add the wall to the wall list
							wallList.add(((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() - 1][((MazeTile) wallList.get(random)).getRow() + 1]));
						}
					}
					//if the room is not in the left-most column of rooms...
					if(((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() - 1][((MazeTile) wallList.get(random)).getRow()]).getColumn() != (LEFT_BUFFER + 1))
					{	//if the wall to the left of the room is not already in the wall list...
						if(!((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() - 2][((MazeTile) wallList.get(random)).getRow()]).isInWallList())
						{	//...add the wall to the wall list
							wallList.add(((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() - 2][((MazeTile) wallList.get(random)).getRow()]));
						}
					}
				}
				//if the room to the right of the chosen wall has not yet been visited...
				else if(!((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() + 1][((MazeTile) wallList.get(random)).getRow()]).hasBeenVisited())
				{	//...mark the room as visited and make the wall a passage
					((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() + 1][((MazeTile) wallList.get(random)).getRow()]).setVisited(true);
					((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn()][((MazeTile) wallList.get(random)).getRow()]).setPassable(true);
					//if the room is not in the top row of rooms...
					if(((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() + 1][((MazeTile) wallList.get(random)).getRow()]).getRow() != (TOP_BUFFER + 1))
					{	//if the wall directly above the room is not already in the wall list...
						if(!((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() + 1][((MazeTile) wallList.get(random)).getRow() - 1]).isInWallList())
						{	//...add the wall to the wall list
							wallList.add(((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() + 1][((MazeTile) wallList.get(random)).getRow() - 1]));
						}
					}
					//if the room is not in right-most column of rooms...
					if(((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() + 1][((MazeTile) wallList.get(random)).getRow()]).getColumn() != (width - (RIGHT_BUFFER + 2)))
					{	//if the wall to the right of the room is not already in the wall list...
						if(!((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() + 2][((MazeTile) wallList.get(random)).getRow()]).isInWallList())
						{	//...add the wall to the wall list
							wallList.add(((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() + 2][((MazeTile) wallList.get(random)).getRow()]));
						}
					}
					//if the room is not in the bottom row of rooms...
					if(((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() + 1][((MazeTile) wallList.get(random)).getRow()]).getRow() != (height - (BOTTOM_BUFFER + 2)))
					{	//if the wall directly below the room is not already in the wall list...
						if(!((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() + 1][((MazeTile) wallList.get(random)).getRow() + 1]).isInWallList())
						{	//...add the wall to the wall list
							wallList.add(((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() + 1][((MazeTile) wallList.get(random)).getRow() + 1]));
						}
					}
				}
			}
			else //if chosen wall is in an odd column...
			{
				if(((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn()][((MazeTile) wallList.get(random)).getRow() - 1]).hasBeenVisited() &&
				   ((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn()][((MazeTile) wallList.get(random)).getRow() + 1]).hasBeenVisited())
				{
					//do nothing
				}
				//if the room directly above the chosen wall has not yet been visited
				else if(!((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn()][((MazeTile) wallList.get(random)).getRow() - 1]).hasBeenVisited())
				{	//...mark the room as visited and make the wall a passage
					((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn()][((MazeTile) wallList.get(random)).getRow() - 1]).setVisited(true);
					((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn()][((MazeTile) wallList.get(random)).getRow()]).setPassable(true);
					//if the room is not in the top row of rooms...
					if(((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn()][((MazeTile) wallList.get(random)).getRow() - 1]).getRow() != (TOP_BUFFER + 1))
					{	//if the wall directly above the room is not already in the wall list...
						if(!((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn()][((MazeTile) wallList.get(random)).getRow() - 2]).isInWallList())
						{	//...add the wall to the wall list
							wallList.add(((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn()][((MazeTile) wallList.get(random)).getRow() - 2]));
						}
					}
					//if the room is not in right-most column of rooms...
					if(((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn()][((MazeTile) wallList.get(random)).getRow() - 1]).getColumn() != (width - (RIGHT_BUFFER + 2)))
					{	//if the wall to the right of the room is not already in the wall list...
						if(!((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() + 1][((MazeTile) wallList.get(random)).getRow() - 1]).isInWallList())
						{	//...add the wall to the wall list
							wallList.add(((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() + 1][((MazeTile) wallList.get(random)).getRow() - 1]));
						}
					}
					//if the room is not in the left-most column of rooms...
					if(((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn()][((MazeTile) wallList.get(random)).getRow() - 1]).getColumn() != (LEFT_BUFFER + 1))
					{	//if the wall to the left of the room is not already in the wall list...
						if(!((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() - 1][((MazeTile) wallList.get(random)).getRow() - 1]).isInWallList())
						{	//...add the wall to the wall list
							wallList.add(((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() - 1][((MazeTile) wallList.get(random)).getRow() - 1]));
						}
					}
				}
				//if the room directly below the chosen wall has not yet been visited...
				else if(!((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn()][((MazeTile) wallList.get(random)).getRow() + 1]).hasBeenVisited())
				{	//...mark the room as visited and make the wall a passage
					((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn()][((MazeTile) wallList.get(random)).getRow() + 1]).setVisited(true);
					((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn()][((MazeTile) wallList.get(random)).getRow()]).setPassable(true);
					//if the room is not in right-most column of rooms...
					if(((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn()][((MazeTile) wallList.get(random)).getRow() + 1]).getColumn() != (width - (RIGHT_BUFFER + 2)))
					{	//if the wall to the right of the room is not already in the wall list...
						if(!((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() + 1][((MazeTile) wallList.get(random)).getRow() + 1]).isInWallList())
						{	//...add the wall to the wall list
							wallList.add(((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() + 1][((MazeTile) wallList.get(random)).getRow() + 1]));
						}
					}
					//if the room is not in the bottom row of rooms...
					if(((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn()][((MazeTile) wallList.get(random)).getRow() + 1]).getRow() != (height - (BOTTOM_BUFFER + 2)))
					{	//if the wall directly below the room is not already in the wall list...
						if(!((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn()][((MazeTile) wallList.get(random)).getRow() + 2]).isInWallList())
						{	//...add the wall to the wall list
							wallList.add(((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn()][((MazeTile) wallList.get(random)).getRow() + 2]));
						}
					}
					//if the room is not in the left-most column of rooms...
					if(((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn()][((MazeTile) wallList.get(random)).getRow() + 1]).getColumn() != (LEFT_BUFFER + 1))
					{	//if the wall to the left of the room is not already in the wall list...
						if(!((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() - 1][((MazeTile) wallList.get(random)).getRow() + 1]).isInWallList())
						{	//...add the wall to the wall list
							wallList.add(((MazeTile) tile[((MazeTile) wallList.get(random)).getColumn() - 1][((MazeTile) wallList.get(random)).getRow() + 1]));
						}
					}
				}
			}
			//remove the wall from the list
			wallList.remove(random);
		}
	}

	/*
	METHOD: drawMap(Graphics, int, int, int, int)
	PURPOSE: Draws the randomly generated maze.
	PARAMETERS:
		Graphics g -> The graphics object being drawn to
		int upperLeftX -> x-coordinate of top-left corner of the displayed portion of map
		int upperLeftY -> y-coordinate of top-left corner of the displayed portion of map
		int lowerRightX -> x-coordinate of lower-right corner of the displayed portion of map
		int lowerRightY -> y-coordinate of lower-right corner of the displayed portion of map
	*/
	public void drawMap(Graphics g, int upperLeftX, int upperLeftY, int lowerRightX, int lowerRightY)
	{
		super.drawMap(g, upperLeftX, upperLeftY, lowerRightX, lowerRightY);

		int panelX; //the current x-coordinate of the panel being drawn on
		int panelY; //the current y-coordinate of the panel being drawn on

		panelX = 0;
		//for each column of tiles...
		for(int i = (upperLeftX / Tile.TILE_SIZE); i < (lowerRightX / Tile.TILE_SIZE); i++)
		{
			panelY = 0;
			//for each row of tiles...
			for(int j = (upperLeftY / Tile.TILE_SIZE); j < (lowerRightY / Tile.TILE_SIZE); j++)
			{	//if the tile is passable and inside the buffers...
				if(tile[i][j].isPassable() && i >= LEFT_BUFFER && i < (width - RIGHT_BUFFER) && j >= TOP_BUFFER && j < (height - BOTTOM_BUFFER))
				{	//...draw a floor tile
					g.drawImage(floorImage, panelX, panelY, null);
				}
				//if the tile is an entrance...
				if(((MazeTile) tile[i][j]).isEntrance())
				{	//...draw an entrance tile
					g.drawImage(entranceImage, panelX, panelY, null);
				}
				//if the tile is an exit...
				if(((MazeTile) tile[i][j]).isExit())
				{	//...draw an exit tile
					g.drawImage(exitImage, panelX, panelY, null);
				}
				//if the tile is a treasure chest...
				if(((MazeTile) tile[i][j]).isTreasureChest())
				{	//...draw a treasure chest tile
					g.drawImage(chestImage, panelX, panelY, null);
				}

				panelY += Tile.TILE_SIZE;
			}

			panelX += Tile.TILE_SIZE;
		}
	}

	/*
	METHOD: TreasureChest
	PURPOSE: Returns the appropriate treasure chest, specified by its row and column.
	*/
	public TreasureChest getChest(int column, int row)
	{
		for(int i = 0; i < chest.length; i++)
		{
			if(chest[i].getColumn() == column && chest[i].getRow() == row)
			{
				return chest[i];
			}
		}

		return null;
	}
}