#include "StateManager.h"

const std::string StateManager::m_oInvalidState = "invalid state";

StateManager::StateManager()
{
	Clear();
}

StateManager::~StateManager()
{
	Term();
}

void StateManager::Clear()
{
	m_oCurrentState.clear();
	for ( GameStateClassMap::iterator itr = m_oState.begin() ; itr != m_oState.end() ; ++itr )
	{
		delete itr->second;
	}
	m_oState.clear();
	while( !m_oCmdQueue.empty() )
	{
		m_oCmdQueue.pop();
	}
	m_bFirstUpdate = false;
	m_bInitialized = false;
	m_fCurrentTime = 0.0f;
}

void StateManager::Init()
{
	m_bFirstUpdate = true;
	m_bInitialized = true;
}

void StateManager::Term()
{
	if( !m_bInitialized )
	{
		ChaosEngine::Log::WriteMessage( "StateManager - Can't terminate before initialized" , ChaosEngine::Log::Warning );
		return;
	}

    // Make sure we properly exit the currently active state
    for( int i = static_cast< int >( m_oCurrentState.size() ) - 1 ; i >= 0 ; i-- )
	{
		GetStateClass( m_oCurrentState[ i ].c_str() )->OnExit( m_oInvalidState );
	}
	Clear();
}


bool StateManager::RegisterState( const std::string& oStateName , IBaseState* pState )
{
	if( !m_bInitialized )
	{
		ChaosEngine::Log::WriteMessage( "StateManager - Can't register state before initialized" , ChaosEngine::Log::Warning );
		return false;
	}

	// Attempt to find the given string ID
	GameStateClassMap::iterator itr = m_oState.find( oStateName );
	
	// If the string ID already exists in the state map, return an error
	if( itr != m_oState.end() )
	{
		ChaosEngine::Log::WriteMessage( "StateManager - Can't register state, state name is already used" , ChaosEngine::Log::Error );
		return false;
	}

	// Assign the object pointer to the ID location in the map
	m_oState[ oStateName ] = pState;

	// If this is the first state registered, set it as the default state
	if( m_oState.size() == 1 )
	{
		m_oCurrentState.push_back( oStateName );
	}

	return true;
}


bool StateManager::IsStateChangePending() const
{
	return ( m_oCmdQueue.empty() ) ? false : true;
}

const std::string StateManager::GetState() const
{
	if ( m_oCurrentState.size() == 0 )
	{
		ChaosEngine::Log::WriteMessage( "StateManager - Can't get state as there is no current state" , ChaosEngine::Log::Error );
		return NULL;
	}
	return m_oCurrentState[ m_oCurrentState.size() - 1 ];
}

int StateManager::GetStateStackSize( void ) const
{
	return static_cast< int >( m_oCurrentState.size() );
}


IBaseState* StateManager::GetStateClass( const std::string& oState )
{
	return m_oState[ oState ];
}

IBaseState* StateManager::GetCurrentStateClass( void )
{
	if ( m_oCurrentState.size() > 0 )
	{
		return GetStateClass( m_oCurrentState[ m_oCurrentState.size() - 1 ] );
	}
	ChaosEngine::Log::WriteMessage( "StateManager - Can't get current state class as there is no current state" , ChaosEngine::Log::Error );
	return NULL;
}

void StateManager::ChangeState( const std::string& oState , float fDelay , bool bFlush )
{
	// Clear the queue
	if ( bFlush )
	{
		while ( !m_oCmdQueue.empty() )
		{
			m_oCmdQueue.pop();
		}
	}
	// Push a "change state" command onto the queue
	GSCommand oCommand;
	oCommand.eCmd = GS_CMD_CHANGE;
	oCommand.oState = oState;
	oCommand.fDelay = fDelay + m_fCurrentTime;
	m_oCmdQueue.push( oCommand );
}

void StateManager::PushState( const std::string& oState , float fDelay , bool bFlush )
{
	// Clear the queue
	if ( bFlush )
	{
		while ( !m_oCmdQueue.empty() )
		{
			m_oCmdQueue.pop();
		}
	}
	// Push a "push state" command onto the queue
	GSCommand cmd;
	cmd.eCmd = GS_CMD_PUSH;
	cmd.oState = oState;
	cmd.fDelay = fDelay + m_fCurrentTime;
	m_oCmdQueue.push( cmd );
}

void StateManager::PopState( int iStatesToPop , float fDelay , bool bFlush )
{
	// Clear the queue
	if ( bFlush )
	{
		while ( !m_oCmdQueue.empty() )
			m_oCmdQueue.pop();
	}
	// Push a "pop state" command onto the queue
	GSCommand cmd;
	cmd.eCmd = GS_CMD_POP;
	cmd.oState = std::string( "" );
	cmd.fDelay = fDelay + m_fCurrentTime;
	while( iStatesToPop > 0 )
	{
		m_oCmdQueue.push( cmd );
		// Only the first state command can have a non-zero value
		cmd.fDelay = 0.0f;
		iStatesToPop--;
	}
}

void StateManager::PopAllStates( float fDelay , bool bFlush )
{
	// Clear the queue
	if ( bFlush )
	{
		while ( !m_oCmdQueue.empty() )
		{
			m_oCmdQueue.pop();
		}
	}
	// Push a "pop all states" command onto the queue
	GSCommand cmd;
	cmd.eCmd = GS_CMD_POP_ALL;
	cmd.oState = std::string( "" );
	cmd.fDelay = fDelay + m_fCurrentTime;
	m_oCmdQueue.push( cmd );
}

void StateManager::Update( float dt )
{
	// Update the total run time
	m_fCurrentTime += dt;

	// Check for the first update and make the appropriate calls
	if( m_bFirstUpdate )
	{
		m_bFirstUpdate = false;
		GetCurrentStateClass()->OnEnter( m_oInvalidState );
	}
	// Empty our comment queue, consisting of commands to either
	// push new states onto the stack, pop states off the stack, or
	// to switch the states on the top of the stack.  In each case
	// we transmit the new state to the old one, and vice-versa.
	GSCommand oCommand;
	while( !m_oCmdQueue.empty() && ( m_oCmdQueue.front().fDelay <= m_fCurrentTime ) )
	{
		oCommand = m_oCmdQueue.front();
		if( oCommand.eCmd == GS_CMD_PUSH )
		{
			const std::string oPrev = GetState();
			if( oPrev != oCommand.oState )
			{
				GetCurrentStateClass()->OnOverride( oCommand.oState );
				m_oCurrentState.push_back( oCommand.oState );
				GetCurrentStateClass()->OnEnter( oPrev );
			}
			else
			{
				ChaosEngine::Log::WriteMessage( "StateManager - cannot push duplicate state" , ChaosEngine::Log::Error );
			}
		}
		else if( oCommand.eCmd == GS_CMD_POP )
		{
			const std::string oNext = m_oInvalidState;
			const std::string oPrev = GetState();
			// If we only have one state, we shouldn't be popping it off the
			// current state stack.  We should be switching it instead.  This
			// code was hit because someone tried to pop off too many states.
			if( m_oCurrentState.size() > 1 )
			{
				GetCurrentStateClass()->OnExit( m_oCurrentState[ m_oCurrentState.size() - 2 ] );
				m_oCurrentState.pop_back();
				GetCurrentStateClass()->OnResume( oPrev );
			}
			else
			{
				ChaosEngine::Log::WriteMessage( "StateManager - cannot pop the last state" , ChaosEngine::Log::Error );
			}
		}
		else if( oCommand.eCmd == GS_CMD_POP_ALL )
		{
			while( m_oCurrentState.size() > 1 )
			{
				const std::string oNext = m_oInvalidState;
				const std::string oPrev = GetState();
				GetCurrentStateClass()->OnExit( m_oCurrentState[ m_oCurrentState.size() - 2 ] );
				m_oCurrentState.pop_back();
				GetCurrentStateClass()->OnResume( oPrev );
			}
		}
		else
		{
			const std::string oPrev = GetState();
			if( oPrev != oCommand.oState )
			{
				GetCurrentStateClass()->OnExit( oCommand.oState );
				m_oCurrentState.pop_back();
				m_oCurrentState.push_back( oCommand.oState );
				GetCurrentStateClass()->OnEnter( oPrev );
			}
			else
			{
				ChaosEngine::Log::WriteMessage( "StateManager - cannot switch to duplicate state" , ChaosEngine::Log::Error );
			}
		}
		m_oCmdQueue.pop();
	}
	// After all state transitions are finished, do the current state stack updates
	for( int i = static_cast< int >( m_oCurrentState.size() ) - 1 ; i >= 0 ; i-- )
	{
		GetStateClass( m_oCurrentState[ i ] )->Update( dt );
	}
}




