#include "D3DGraphics.h"


ChaosEngine::D3DGraphics::D3DGraphics( System* pSystem ) : m_pSystem( pSystem )
{
	m_pFont = NULL;
	m_pTextSprite = NULL;
	m_pStack = NULL;
	D3DStatics::Startup( this );
};

ChaosEngine::D3DGraphics::~D3DGraphics( void )
{
	OnLostDevice();
	OnDestroyDevice();
	D3DStatics::Shutdown();
};

void ChaosEngine::D3DGraphics::Update( float fDeltaTime )
{
	m_pSystem->Update( fDeltaTime );
};

void ChaosEngine::D3DGraphics::Render( IDirect3DDevice9* pd3dDevice )
{
	pd3dDevice->SetLight( 0 , &m_oLight );
	pd3dDevice->LightEnable( 0 , TRUE );
	pd3dDevice->SetRenderState( D3DRS_LIGHTING , TRUE );
	pd3dDevice->SetRenderState( D3DRS_AMBIENT , 0x00606060 );
	
	D3DXMATRIX oMatrix;
	D3DXMatrixLookAtLH( &oMatrix , &m_oEye , &m_oAt , &m_oUp );
	pd3dDevice->SetTransform( D3DTS_VIEW , &oMatrix );
	D3DXMatrixPerspectiveFovLH( &oMatrix , D3DX_PI / 4.0f , 1.333f , 0.5f , 1000.0f );
	pd3dDevice->SetTransform( D3DTS_PROJECTION , &oMatrix );

	for ( MaterialHash::iterator oMaterialIterator = m_oMaterials.begin() ; oMaterialIterator != m_oMaterials.end() ; oMaterialIterator++ )
	{
		oMaterialIterator->second->PreRender( pd3dDevice );
		for ( std::set< std::string >::iterator oObjectNameIterator = oMaterialIterator->second->m_oObjectSet.begin() ; oObjectNameIterator != oMaterialIterator->second->m_oObjectSet.end() ; oObjectNameIterator++ )
		{
			ObjectHash::iterator oObjectIterator = m_oObjects.find( *oObjectNameIterator );
			if ( oObjectIterator != m_oObjects.end() )
			{
				MeshHash::iterator oMeshIterator = m_oMeshes.find( oObjectIterator->second->m_oMeshName );
				if ( oMeshIterator != m_oMeshes.end() )
				{
					m_pStack->Push();
					m_pStack->Scale( oObjectIterator->second->m_oScale[ 0 ] , oObjectIterator->second->m_oScale[ 1 ] , oObjectIterator->second->m_oScale[ 2 ] );
					m_pStack->MultMatrix( D3DXMatrixRotationQuaternion( &oMatrix , &oObjectIterator->second->m_oOrientation ) );
					m_pStack->Translate( oObjectIterator->second->m_oPosition[ 0 ] , oObjectIterator->second->m_oPosition[ 1 ] , oObjectIterator->second->m_oPosition[ 2 ] );
					// set the object's world matrix
					pd3dDevice->SetTransform( D3DTS_WORLD , m_pStack->GetTop() );
					// render the object's mesh
					oMeshIterator->second->DrawSubset( 0 );
					m_pStack->Pop();
				}
			}
		}
		oMaterialIterator->second->PostRender( pd3dDevice );
	}

	m_pTextHelper = new CDXUTTextHelper( m_pFont , m_pTextSprite , 15 );
	m_pTextHelper->Begin();
	m_pTextHelper->End();
	delete m_pTextHelper;
	m_pTextHelper = NULL;
};

HRESULT ChaosEngine::D3DGraphics::OnCreateDevice( IDirect3DDevice9* pd3dDevice , const D3DSURFACE_DESC* pBackBufferSurfaceDesc )
{
	LPD3DXMESH pNewMesh;
	D3DXCreateSphere( pd3dDevice , 1.0f , 32 , 32 , &pNewMesh , NULL );
	m_oMeshes[ std::string( "sphere" ) ] = pNewMesh;
	D3DXCreateBox( pd3dDevice , 1.0f , 1.0f , 1.0f , &pNewMesh , NULL );
	m_oMeshes[ std::string( "cube" ) ] = pNewMesh;
	D3DXCreateFontA( pd3dDevice , 15 , 0 , FW_BOLD , 1 , FALSE , DEFAULT_CHARSET , OUT_DEFAULT_PRECIS , DEFAULT_QUALITY , DEFAULT_PITCH | FF_DONTCARE , "Verdana" , &m_pFont );
	D3DXCreateMatrixStack( 0 , &m_pStack );
	m_pStack->LoadIdentity();
	return 0;
};

HRESULT ChaosEngine::D3DGraphics::OnResetDevice( IDirect3DDevice9* pd3dDevice , const D3DSURFACE_DESC* pBackBufferSurfaceDesc )
{
	m_pFont->OnResetDevice();
	D3DXCreateSprite( pd3dDevice , &m_pTextSprite );
	return 0;
};

void ChaosEngine::D3DGraphics::OnLostDevice( void )
{
    if( m_pFont != NULL )
	{
        m_pFont->OnLostDevice();
	}
	if ( m_pTextSprite != NULL )
	{
		m_pTextSprite->Release();
		m_pTextSprite = NULL;
	}
};

void ChaosEngine::D3DGraphics::OnDestroyDevice( void )
{
	for ( MeshHash::iterator oIterator = m_oMeshes.begin() ; oIterator != m_oMeshes.end() ; oIterator++ )
	{
		oIterator->second->Release();
	}
	m_oMeshes.clear();
	for ( ObjectHash::iterator oIterator = m_oObjects.begin() ; oIterator != m_oObjects.end() ; oIterator++ )
	{
		delete oIterator->second;
	}
	m_oObjects.clear();
	for ( MaterialHash::iterator oIterator = m_oMaterials.begin() ; oIterator != m_oMaterials.end() ; oIterator++ )
	{
		delete oIterator->second;
	}
	m_oMaterials.clear();
	if ( m_pFont != NULL )
	{
		m_pFont->Release();
		m_pFont = NULL;
	}
	if ( m_pStack != NULL )
	{
		m_pStack->Release();
		m_pStack = NULL;
	}
};

void ChaosEngine::D3DGraphics::WindowActivate( void )
{
	if ( m_pSystem->GetInput() )
	{
		m_pSystem->GetInput()->Aquire();
	}
};

void ChaosEngine::D3DGraphics::WindowDeactivate( void )
{
	if ( m_pSystem->GetInput() )
	{
		m_pSystem->GetInput()->UnAquire();
	}
};


void ChaosEngine::D3DGraphics::CreateSphere( const std::string& oName , const std::string& oMaterialName , float* pPosition , float* pOrientation , float fRadius )
{
	if ( m_oObjects.find( oName ) == m_oObjects.end() )
	{
		ObjectDescription* pObject = new ObjectDescription();
		pObject->m_oMeshName = std::string( "sphere" );
		pObject->m_oMaterialName = oMaterialName;
		pObject->m_oPosition.x = pPosition[ 0 ];
		pObject->m_oPosition.y = pPosition[ 1 ];
		pObject->m_oPosition.z = pPosition[ 2 ];
		pObject->m_oOrientation.x = pOrientation[ 0 ];
		pObject->m_oOrientation.y = pOrientation[ 1 ];
		pObject->m_oOrientation.z = pOrientation[ 2 ];
		pObject->m_oOrientation.w = pOrientation[ 3 ];
		pObject->m_oScale.x = fRadius;
		pObject->m_oScale.y = fRadius;
		pObject->m_oScale.z = fRadius;
		m_oObjects[ oName ] = pObject;
		MaterialHash::iterator oIterator = m_oMaterials.find( oMaterialName );
		if ( oIterator != m_oMaterials.end() )
		{
			std::set< std::string >::iterator oObjectSetIterator = oIterator->second->m_oObjectSet.find( oName );
			if ( oObjectSetIterator == oIterator->second->m_oObjectSet.end() )
			{
				oIterator->second->m_oObjectSet.insert( oName );
			}
			else
			{
				Log::WriteMessage( std::string( "D3DGraphics - CreateBox - Object name already exists in the material's object vector" ) , Log::Warning );
			}
		}
		else
		{
			Log::WriteMessage( std::string( "D3DGraphics - CreateBox - Material doesn't exist" ) , Log::Warning );
		}
	}
	else
	{
		Log::WriteMessage( std::string( "D3DGraphics - CreateBox - Object already exists" ) , Log::Warning );
	}
};


void ChaosEngine::D3DGraphics::CreateBox( const std::string& oName , const std::string& oMaterialName , float* pPosition , float* pOrientation , float* pSideLengths )
{
	if ( m_oObjects.find( oName ) == m_oObjects.end() )
	{
		ObjectDescription* pObject = new ObjectDescription();
		pObject->m_oMeshName = std::string( "cube" );
		pObject->m_oMaterialName = oMaterialName;
		pObject->m_oPosition.x = pPosition[ 0 ];
		pObject->m_oPosition.y = pPosition[ 1 ];
		pObject->m_oPosition.z = pPosition[ 2 ];
		pObject->m_oOrientation.x = pOrientation[ 0 ];
		pObject->m_oOrientation.y = pOrientation[ 1 ];
		pObject->m_oOrientation.z = pOrientation[ 2 ];
		pObject->m_oOrientation.w = pOrientation[ 3 ];
		pObject->m_oScale.x = pSideLengths[ 0 ];
		pObject->m_oScale.y = pSideLengths[ 1 ];
		pObject->m_oScale.z = pSideLengths[ 2 ];
		m_oObjects[ oName ] = pObject;
		MaterialHash::iterator oIterator = m_oMaterials.find( oMaterialName );
		if ( oIterator != m_oMaterials.end() )
		{
			std::set< std::string >::iterator oObjectSetIterator = oIterator->second->m_oObjectSet.find( oName );
			if ( oObjectSetIterator == oIterator->second->m_oObjectSet.end() )
			{
				oIterator->second->m_oObjectSet.insert( oName );
			}
			else
			{
				Log::WriteMessage( std::string( "D3DGraphics - CreateBox - Object name already exists in the material's object vector" ) , Log::Warning );
			}
		}
		else
		{
			Log::WriteMessage( std::string( "D3DGraphics - CreateBox - Material doesn't exist" ) , Log::Warning );
		}
	}
	else
	{
		Log::WriteMessage( std::string( "D3DGraphics - CreateBox - Object already exists" ) , Log::Warning );
	}
};

void ChaosEngine::D3DGraphics::SetObjectPosition( const std::string& oName , float* pPosition )
{
	ObjectHash::iterator oObjectIterator = m_oObjects.find( oName );
	if ( oObjectIterator != m_oObjects.end() )
	{
		oObjectIterator->second->m_oPosition.x = pPosition[ 0 ];
		oObjectIterator->second->m_oPosition.y = pPosition[ 1 ];
		oObjectIterator->second->m_oPosition.z = pPosition[ 2 ];
	}
};

void ChaosEngine::D3DGraphics::SetObjectOrientation( const std::string& oName , float* pOrientation )
{
	ObjectHash::iterator oObjectIterator = m_oObjects.find( oName );
	if ( oObjectIterator != m_oObjects.end() )
	{
		oObjectIterator->second->m_oOrientation.x = pOrientation[ 0 ];
		oObjectIterator->second->m_oOrientation.y = pOrientation[ 1 ];
		oObjectIterator->second->m_oOrientation.z = pOrientation[ 2 ];
		oObjectIterator->second->m_oOrientation.w = pOrientation[ 3 ];
	}
};

void ChaosEngine::D3DGraphics::DestroyObject( const std::string& oName )
{
	ObjectHash::iterator oObjectIterator = m_oObjects.find( oName );
	if ( oObjectIterator != m_oObjects.end() )
	{
		MaterialHash::iterator oMaterialIterator = m_oMaterials.find( oObjectIterator->second->m_oMaterialName );
		if ( oMaterialIterator != m_oMaterials.end() )
		{
			std::set< std::string >::iterator oObjectSetIterator = oMaterialIterator->second->m_oObjectSet.find( oName );
			if ( oObjectSetIterator != oMaterialIterator->second->m_oObjectSet.end() )
			{
				oMaterialIterator->second->m_oObjectSet.erase( oObjectSetIterator );
			}
			else
			{
				Log::WriteMessage( std::string( "D3DGraphics - DestroyObject - Object name doesn't exist in material's object vector" ) , Log::Warning );
			}
			delete oObjectIterator->second;
			m_oObjects.erase( oObjectIterator );
		}
		else
		{
				Log::WriteMessage( std::string( "D3DGraphics - DestroyObject - Material doesn't exist" ) , Log::Warning );
		}
	}
	else
	{
		Log::WriteMessage( std::string( "D3DGraphics - DestroyObject - Object doesn't exist" ) , Log::Warning );
	}
};


void ChaosEngine::D3DGraphics::CreateMaterial( const std::string& oMaterialName , float* pColor )
{
	if ( m_oMaterials.find( oMaterialName ) == m_oMaterials.end() )
	{
		Material* pMaterial = new Material( pColor );
		m_oMaterials[ oMaterialName ] = pMaterial;
	}
	else
	{
		Log::WriteMessage( std::string( "D3DGraphics - CreateMaterial - Material already exists" ) , Log::Warning );
	}
};


void ChaosEngine::D3DGraphics::DestroyMaterial( const std::string& oMaterialName )
{
	MaterialHash::iterator oIterator = m_oMaterials.find( oMaterialName );
	if ( oIterator != m_oMaterials.end() )
	{
		delete oIterator->second;
		m_oMaterials.erase( oIterator );
	}
	else
	{
		Log::WriteMessage( std::string( "D3DGraphics - DestroyMaterial - Material doesn't exist" ) , Log::Warning );
	}
};


void ChaosEngine::D3DGraphics::SetLight( float* pPosition , float* pDirection , float* pColor )
{
	ZeroMemory( &m_oLight , sizeof( m_oLight ) );
	m_oLight.Type = D3DLIGHT_DIRECTIONAL;
	m_oLight.Diffuse.r = pColor[ 0 ];
	m_oLight.Diffuse.g = pColor[ 1 ];
	m_oLight.Diffuse.b = pColor[ 2 ];
	m_oLight.Position.x = pPosition[ 0 ];
	m_oLight.Position.y = pPosition[ 1 ];
	m_oLight.Position.z = pPosition[ 2 ];
	D3DXVECTOR3 vecDir;
	vecDir = D3DXVECTOR3( pDirection[ 0 ] , pDirection[ 1 ] , pDirection[ 2 ] );
	D3DXVec3Normalize( ( D3DXVECTOR3* )&m_oLight.Direction , &vecDir );
	m_oLight.Range = 1000.0f;
};


void ChaosEngine::D3DGraphics::SetCamera( float* pPosition , float* pOrientation )
{
	m_oEye = D3DXVECTOR3( pPosition[ 0 ] , pPosition[ 1 ] , pPosition[ 2 ] );
	D3DXVECTOR4 oBigAt;
	D3DXVECTOR3 oForward( 0.0f , 0.0f , 1.0f );
	D3DXMATRIX oMatrix;
	D3DXMatrixRotationYawPitchRoll( &oMatrix , pOrientation[ 1 ] , pOrientation[ 0 ] , pOrientation[ 2 ] );
	D3DXVec3Transform( &oBigAt , &oForward , &oMatrix );
	oForward.x = oBigAt.x;
	oForward.y = oBigAt.y;
	oForward.z = oBigAt.z;
	D3DXVec3Add( &m_oAt , &m_oEye , &oForward );
	m_oUp = D3DXVECTOR3( 0.0f , 1.0f , 0.0f );
};

void ChaosEngine::D3DGraphics::GetCameraDirection( float* pDirection , float* pUp )
{
	D3DXVECTOR3 oDirection( m_oAt - m_oEye );
	D3DXVec3Normalize( &oDirection , &oDirection );
	pDirection[ 0 ] = oDirection.x;
	pDirection[ 1 ] = oDirection.y;
	pDirection[ 2 ] = oDirection.z;
	D3DXVECTOR3 oUp;
	D3DXVECTOR3 oRight;
	oRight.y = 0.0f;
	oRight.x = oDirection.z;
	oRight.z = -oDirection.x;
	D3DXVec3Cross( &oUp , &oDirection , &oRight );
	pUp[ 0 ] = oUp.x;
	pUp[ 1 ] = oUp.y;
	pUp[ 2 ] = oUp.z;
};

void ChaosEngine::D3DGraphics::DrawString( float* pPosition , const std::string& oString )
{
	std::wstringstream oStream;
	for ( size_t i = 0 ; i < oString.length() ; ++i )
	{
		oStream << static_cast< wchar_t >( oString[ i ] );
	}
    m_pTextHelper->SetInsertionPos( static_cast< int >( pPosition[ 0 ] ) , static_cast< int >( pPosition[ 1 ] ) );
    m_pTextHelper->SetForegroundColor( D3DXCOLOR( 1.0f , 1.0f , 1.0f , 1.0f ) );
	m_pTextHelper->DrawTextLine( oStream.str().c_str() );
};


void D3DStatics::Startup( ChaosEngine::D3DGraphics* pGraphics )
{
	g_pGraphics = pGraphics;
	DXUTSetCallbackDeviceCreated( D3DStatics::OnCreateDevice );
    DXUTSetCallbackDeviceReset( D3DStatics::OnResetDevice );
    DXUTSetCallbackDeviceLost( D3DStatics::OnLostDevice );
    DXUTSetCallbackDeviceDestroyed( D3DStatics::OnDestroyDevice );
    DXUTSetCallbackMsgProc( D3DStatics::MsgProc );
    DXUTSetCallbackFrameRender( D3DStatics::OnFrameRender );
    DXUTSetCallbackFrameMove( D3DStatics::OnFrameMove );

    DXUTInit( false, false, false );
    DXUTSetCursorSettings( true, true );
    DXUTCreateWindow( L"Tug of War" );
    DXUTCreateDevice( D3DADAPTER_DEFAULT, true, 800, 600, D3DStatics::IsDeviceAcceptable, D3DStatics::ModifyDeviceSettings );
};


void D3DStatics::Shutdown( void )
{
	if ( g_pGraphics != NULL )
	{
		g_pGraphics = NULL;
	}
};


bool CALLBACK D3DStatics::IsDeviceAcceptable( D3DCAPS9* pCaps , D3DFORMAT AdapterFormat , D3DFORMAT BackBufferFormat , bool bWindowed , void* pUserContext )
{
    // Typically want to skip backbuffer formats that don't support alpha blending
    IDirect3D9* pD3D = DXUTGetD3DObject(); 
    if( FAILED( pD3D->CheckDeviceFormat( pCaps->AdapterOrdinal , pCaps->DeviceType , AdapterFormat , D3DUSAGE_QUERY_POSTPIXELSHADER_BLENDING , D3DRTYPE_TEXTURE , BackBufferFormat ) ) )
	{
        return false;
	}
    return true;
};


bool CALLBACK D3DStatics::ModifyDeviceSettings( DXUTDeviceSettings* pDeviceSettings , const D3DCAPS9* pCaps , void* pUserContext )
{
    return true;
};


HRESULT CALLBACK D3DStatics::OnCreateDevice( IDirect3DDevice9* pd3dDevice , const D3DSURFACE_DESC* pBackBufferSurfaceDesc , void* pUserContext )
{
	if ( g_pGraphics != NULL )
	{
		return g_pGraphics->OnCreateDevice( pd3dDevice , pBackBufferSurfaceDesc );
	}
	return 0;
};


HRESULT CALLBACK D3DStatics::OnResetDevice( IDirect3DDevice9* pd3dDevice , const D3DSURFACE_DESC* pBackBufferSurfaceDesc , void* pUserContext )
{
	if ( g_pGraphics != NULL )
	{
		return g_pGraphics->OnResetDevice( pd3dDevice , pBackBufferSurfaceDesc );
	}
	return 0;
};


void CALLBACK D3DStatics::OnFrameMove( IDirect3DDevice9* pd3dDevice , double fTime , float fElapsedTime , void* pUserContext )
{
	if ( g_pGraphics != NULL )
	{
		g_pGraphics->Update( fElapsedTime );
	}
};


void CALLBACK D3DStatics::OnFrameRender( IDirect3DDevice9* pd3dDevice , double fTime , float fElapsedTime , void* pUserContext )
{
    HRESULT hr;

    // Clear the render target and the zbuffer 
    V( pd3dDevice->Clear( 0 , NULL , D3DCLEAR_TARGET | D3DCLEAR_ZBUFFER , D3DCOLOR_ARGB( 0 , 0 , 0 , 0 ) , 1.0f , 0 ) );

    // Render the scene
    if( SUCCEEDED( pd3dDevice->BeginScene() ) )
    {
		if ( g_pGraphics != NULL )
		{
			g_pGraphics->Render( pd3dDevice );
		}
        V( pd3dDevice->EndScene() );
    }
};


LRESULT CALLBACK D3DStatics::MsgProc( HWND hWnd , UINT uMsg , WPARAM wParam , LPARAM lParam , bool* pbNoFurtherProcessing , void* pUserContext )
{
	switch ( uMsg )
	{
	case WM_ACTIVATE:
		if ( WA_INACTIVE == wParam )
		{
			if ( g_pGraphics != NULL )
			{
				g_pGraphics->WindowDeactivate();
			}
		}
		else
		{
			if ( g_pGraphics != NULL )
			{
				g_pGraphics->WindowActivate();
			}
		}
	}
    return 0;
};


void CALLBACK D3DStatics::OnLostDevice( void* pUserContext )
{
	if ( g_pGraphics != NULL )
	{
		g_pGraphics->OnLostDevice();
	}
};


void CALLBACK D3DStatics::OnDestroyDevice( void* pUserContext )
{
	if ( g_pGraphics != NULL )
	{
		g_pGraphics->OnDestroyDevice();
	}
};

