#pragma once

#include <string>
#include <vector>
#include <map>
#include <queue>
#include "Log.h"

class IBaseState
{
public:
	virtual ~IBaseState() {}

	virtual std::string GetName( void ) = 0;
	virtual void OnEnter( const std::string& oPrevious ) = 0;
	virtual void Update( float fDeltaTime ) = 0;
	virtual void OnExit( const std::string& oNext ) = 0;
	virtual void OnOverride( const std::string& oNext ) = 0;
	virtual void OnResume( const std::string& oPrevious ) = 0;
};


class StateManager
{
public:
	StateManager();
	~StateManager();

	void Init();
	void Term();

	// Register a state object and associate it with a string identifier
	bool RegisterState(const std::string& oStateName, IBaseState* pState);

	// Checks if the current state will change
	// on the next update cycle
	bool IsStateChangePending() const;

	// Returns the current state
	const std::string GetState() const;

	// Get the state object based on the string ID
	IBaseState* GetStateClass(const std::string& oState);

	// Get the state object on top of the current state stack
	IBaseState* GetCurrentStateClass();

	// Returns the size of the state stack
	int GetStateStackSize() const;

	// Passing bFlush = true will override any previous state changing
	// commands that may be pending.  Otherwise, state commands
	// will queue and be executed in the order of the calls made.

	// Changes the current state on the next update cycle.  
	void ChangeState(const std::string& oState, float fDelay = 0.0f, bool bFlush = false);

	// Pushes a new state on top of the existing one on the next update cycle.
	void PushState(const std::string& oState, float fDelay = 0.0f, bool bFlush = false);

	// Pops off the current state or states to reveal a stored state 
	// underneath.  You may not pop off the last state
	void PopState(int iStatesToPop = 1, float fDelay = 0.0f, bool bFlush = false);

	// Pops all but the last state.
	void PopAllStates(float fDelay = 0.0f, bool bFlush = false);

	// Updates the state machine internal mechanism.  This function is called 
	// once by the main update loop and should not be called by anyone else.
	void Update(float dt);

private:

	// private enums, structs, and typedefs
	enum GS_CMD_TYPE
	{
		GS_CMD_PUSH,
		GS_CMD_CHANGE,
		GS_CMD_POP,
		GS_CMD_POP_ALL
	};

	struct GSCommand
	{
		GS_CMD_TYPE eCmd;
		std::string oState;
		float fDelay;
	};

	typedef std::vector<std::string> GameStateVector;
	typedef std::map<std::string, IBaseState*> GameStateClassMap;
	typedef std::queue<GSCommand> GSCmdQueue;


	// Private functions
	void Clear();

	// Private class data
	GameStateVector         m_oCurrentState;
	GameStateClassMap	    m_oState;
	GSCmdQueue              m_oCmdQueue;
	bool					m_bFirstUpdate;
	bool					m_bInitialized;
	float					m_fCurrentTime;

	static const std::string m_oInvalidState;
};
