#include "ODEPhysics.h"

ChaosEngine::ODEPhysics* g_pPhysics;

ChaosEngine::ODEPhysics::ODEPhysics( System* pSystem ) : m_pSystem( pSystem )
{
	g_pPhysics = this;
	SetPhysicsEventHandler( dynamic_cast< IPhysicsEventHandler* >( pSystem ) );
	CreateWorld();
};

ChaosEngine::ODEPhysics::~ODEPhysics( void )
{
	DestroyWorld();
	// cleanup ode
	dCloseODE();
};

static void CollisionCallback( void* pData , dGeomID l_Geom1 , dGeomID l_Geom2 )
{
	g_pPhysics->Collision( l_Geom1 , l_Geom2 , pData );
};

void ChaosEngine::ODEPhysics::SetPhysicsEventHandler( IPhysicsEventHandler* pHandler )
{
	m_pHandler = pHandler;
};

//void ChaosEngine::ODEPhysics::CreateSpace( const std::string& oSpaceName )
//{
//	//// create and setup the collision space
//	//m_dSpace = dHashSpaceCreate( 0 );
//	//dHashSpaceSetLevels( m_dSpace , -10 , 10 );
//	//dSpaceSetCleanup( m_dSpace , 1 );
//};

void ChaosEngine::ODEPhysics::CreateWorld()
{
	WorldHash::iterator oIterator = m_oWorlds.find( std::string( "world" ) );
	if ( oIterator == m_oWorlds.end() )
	{
		WorldDescriptor* pWorld = new WorldDescriptor();

		pWorld->m_oWorldID = dWorldCreate();
		dWorldSetGravity( pWorld->m_oWorldID , dReal( 0.0 ) , dReal( -9.81 ) , dReal( 0.0 ) );
		dWorldSetERP( pWorld->m_oWorldID , dReal( 0.1 ) );
		dWorldSetCFM( pWorld->m_oWorldID , dReal( 0.001 ) );
		dWorldSetAutoDisableFlag( pWorld->m_oWorldID , 0 );
		dWorldSetAutoDisableLinearThreshold( pWorld->m_oWorldID , dReal( 0.001 ) );
		dWorldSetAutoDisableAngularThreshold( pWorld->m_oWorldID , dReal( 0.001 ) );
		dWorldSetAutoDisableSteps( pWorld->m_oWorldID , 10 );
		dWorldSetAutoDisableTime( pWorld->m_oWorldID , dReal( 0.0 ) );
		//dWorldSetContactMaxCorrectingVel( pWorld->m_oWorldID , 0.1f );
		dWorldSetContactSurfaceLayer( pWorld->m_oWorldID , dReal( 0.001 ) );

		// create the joint group for per step contacts
		pWorld->m_oJointGroupID = dJointGroupCreate( 0 );

		// create and setup the collision space
		pWorld->m_oSpaceID = dHashSpaceCreate( 0 );
		dHashSpaceSetLevels( pWorld->m_oSpaceID , -10 , 10 );
		dSpaceSetCleanup( pWorld->m_oSpaceID , 1 );

		m_oWorlds[ std::string( "world" ) ] = pWorld;
	}
	else
	{
		Log::WriteMessage( std::string( "ODEPhysics - CreateWorld - World already exists" ) , Log::Warning );
	}
};

void ChaosEngine::ODEPhysics::DestroyWorld()
{
	for ( BodyHash::iterator oBodyIterator = m_oBodies.begin() ; oBodyIterator != m_oBodies.end() ; oBodyIterator++ )
	{
		delete oBodyIterator->second;
	}
	m_oBodies.clear();
	WorldHash::iterator oIterator = m_oWorlds.find( std::string( "world" ) );
	if ( oIterator != m_oWorlds.end() )
	{
		dSpaceDestroy( oIterator->second->m_oSpaceID );
		dJointGroupDestroy( oIterator->second->m_oJointGroupID );
		dWorldDestroy( oIterator->second->m_oWorldID );
		delete oIterator->second;
		m_oWorlds.erase( oIterator );
	}
	else
	{
		Log::WriteMessage( std::string( "ODEPhysics - DestroyWorld - World doesn't exist" ) , Log::Warning );
	}
};

void ChaosEngine::ODEPhysics::CreateBoxBody( const std::string& oBodyName , bool bDynamic , float* pPosition , float* pOrientation , float* pLinearVelocity , float* pAngularVelocity , float* pSides , float fMass )
{
	WorldHash::iterator oIterator = m_oWorlds.find( std::string( "world" ) );
	if ( oIterator != m_oWorlds.end() )
	{
		BodyHash::iterator oBodyIterator = m_oBodies.find( oBodyName );
		if ( oBodyIterator == m_oBodies.end() )
		{
			dQuaternion oQuaternion;
			oQuaternion[ 0 ] = pOrientation[ 3 ];
			oQuaternion[ 1 ] = pOrientation[ 0 ];
			oQuaternion[ 2 ] = pOrientation[ 1 ];
			oQuaternion[ 3 ] = pOrientation[ 2 ];
			dGeomID oGeom = NULL;
			dBodyID oBody = NULL;
			dMass dMass;

			if ( bDynamic )
			{
				oBody = dBodyCreate( oIterator->second->m_oWorldID );
				dBodySetPosition( oBody , pPosition[ 0 ] , pPosition[ 1 ] , pPosition[ 2 ] );
				dBodySetQuaternion( oBody , oQuaternion );
				dBodySetLinearVel( oBody , pLinearVelocity[ 0 ] , pLinearVelocity[ 1 ] , pLinearVelocity[ 2 ] );
				dBodySetAngularVel( oBody , pAngularVelocity[ 0 ] , pAngularVelocity[ 1 ] , pAngularVelocity[ 2 ] );
				dMassSetBox( &dMass , 100.0f , pSides[ 0 ] , pSides[ 1 ] , pSides[ 2 ] );
				dMassAdjust( &dMass , fMass );
				oGeom = dCreateBox( oIterator->second->m_oSpaceID , pSides[ 0 ] , pSides[ 1 ] , pSides[ 2 ] );
				dGeomSetBody( oGeom , oBody );
				dBodySetMass( oBody , &dMass );
				Body* pObject = new Body( oBodyName , oBody , oGeom );
				dGeomSetData( oGeom , static_cast< void* >( pObject ) );
				m_oBodies[ oBodyName ] = pObject;
			}
			else
			{
				oGeom = dCreateBox( oIterator->second->m_oSpaceID , pSides[ 0 ] , pSides[ 1 ] , pSides[ 2 ] );
				dGeomSetPosition( oGeom , pPosition[ 0 ] , pPosition[ 1 ] , pPosition[ 2 ] );
				dGeomSetQuaternion( oGeom , oQuaternion );
				Body* pObject = new Body( oBodyName , oBody , oGeom );
				dGeomSetData( oGeom , static_cast< void* >( pObject ) );
				m_oBodies[ oBodyName ] = pObject;
			}
		}
		else
		{
			Log::WriteMessage( std::string( "ODEPhysics - CreateBoxBody - Body already exists" ) , Log::Warning );
		}
	}
	else
	{
		Log::WriteMessage( std::string( "ODEPhysics - CreateBoxBody - World doesn't exist" ) , Log::Warning );
	}
};

void ChaosEngine::ODEPhysics::CreateSphereBody( const std::string& oBodyName , bool bDynamic , float* pPosition , float* pOrientation , float* pLinearVelocity , float* pAngularVelocity , float fRadius , float fMass )
{
	WorldHash::iterator oIterator = m_oWorlds.find( std::string( "world" ) );
	if ( oIterator != m_oWorlds.end() )
	{
		BodyHash::iterator oBodyIterator = m_oBodies.find( oBodyName );
		if ( oBodyIterator == m_oBodies.end() )
		{
			dQuaternion oQuaternion;
			oQuaternion[ 0 ] = pOrientation[ 3 ];
			oQuaternion[ 1 ] = pOrientation[ 0 ];
			oQuaternion[ 2 ] = pOrientation[ 1 ];
			oQuaternion[ 3 ] = pOrientation[ 2 ];
			dGeomID oGeom = NULL;
			dBodyID oBody = NULL;
			dMass dMass;

			if ( bDynamic )
			{
				oBody = dBodyCreate( oIterator->second->m_oWorldID );
				dBodySetPosition( oBody , pPosition[ 0 ] , pPosition[ 1 ] , pPosition[ 2 ] );
				dBodySetQuaternion( oBody , oQuaternion );
				dBodySetLinearVel( oBody , pLinearVelocity[ 0 ] , pLinearVelocity[ 1 ] , pLinearVelocity[ 2 ] );
				dBodySetAngularVel( oBody , pAngularVelocity[ 0 ] , pAngularVelocity[ 1 ] , pAngularVelocity[ 2 ] );
				dMassSetSphere( &dMass , 100.0f , fRadius );
				dMassAdjust( &dMass , fMass );
				oGeom = dCreateSphere( oIterator->second->m_oSpaceID , fRadius );
				dGeomSetBody( oGeom , oBody );
				dBodySetMass( oBody , &dMass );
				Body* pObject = new Body( oBodyName , oBody , oGeom );
				dGeomSetData( oGeom , static_cast< void* >( pObject ) );
				m_oBodies[ oBodyName ] = pObject;
			}
			else
			{
				oGeom = dCreateSphere( oIterator->second->m_oSpaceID , fRadius );
				dGeomSetPosition( oGeom , pPosition[ 0 ] , pPosition[ 1 ] , pPosition[ 2 ] );
				dGeomSetQuaternion( oGeom , oQuaternion );

				Body* pObject = new Body( oBodyName , oBody , oGeom );
				dGeomSetData( oGeom , static_cast< void* >( pObject ) );
				m_oBodies[ oBodyName ] = pObject;
			}
		}
		else
		{
			Log::WriteMessage( std::string( "ODEPhysics - CreateSphereBody - Body already exists" ) , Log::Warning );
		}
	}
	else
	{
		Log::WriteMessage( std::string( "ODEPhysics - CreateSphereBody - World doesn't exist" ) , Log::Warning );
	}
};

void ChaosEngine::ODEPhysics::DestroyBody( const std::string& oBodyName )
{
	BodyHash::iterator oIterator = m_oBodies.find( oBodyName );
	if ( oIterator != m_oBodies.end() )
	{
		if ( oIterator->second->m_oBodyID != 0 )
		{
			dBodyDestroy( oIterator->second->m_oBodyID );
		}
		dGeomDestroy( oIterator->second->m_oGeomID );
		delete oIterator->second;
		m_oBodies.erase( oIterator );
	}
	else
	{
		Log::WriteMessage( std::string( "ODEPhysics - DestroyBody - Body doesn't exist" ) , Log::Warning );
	}
};

void ChaosEngine::ODEPhysics::Step( float fDeltaTime )
{
	WorldHash::iterator oIterator = m_oWorlds.find( std::string( "world" ) );
	if ( oIterator != m_oWorlds.end() )
	{
		// check for collisions, creating contact joints when a collision is detected
		dSpaceCollide( oIterator->second->m_oSpaceID , static_cast< void* >( oIterator->second ) , &CollisionCallback );
		// step the world
		dWorldQuickStep( oIterator->second->m_oWorldID , fDeltaTime );
		// clear contact joints
		dJointGroupEmpty( oIterator->second->m_oJointGroupID );
	}
	else
	{
		Log::WriteMessage( std::string( "ODEPhysics - Step - World doesn't exist" ) , Log::Warning );
	}
};

void ChaosEngine::ODEPhysics::Collision( dGeomID l_Geom1 , dGeomID l_Geom2 , void* pData )
{
	WorldDescriptor* pWorld = static_cast< WorldDescriptor* >( pData );
	Body* pBody1 = static_cast< Body* >( dGeomGetData( l_Geom1 ) );
	Body* pBody2 = static_cast< Body* >( dGeomGetData( l_Geom2 ) );
	if ( m_pHandler->ShouldCreateContact( pBody1->m_oBodyName , pBody2->m_oBodyName ) )
	{
		int iMaxContacts = 4;
		// check that the collision between the two geoms is valid
		dBodyID l_Body1 = pBody1->m_oBodyID;
		dBodyID l_Body2 = pBody2->m_oBodyID;
		// if valid then create a contact joint and add it to the contact joint group
		if ( !( ( l_Body1 != 0 ) && ( l_Body2 != 0 ) && dAreConnectedExcluding( l_Body1 , l_Body2 , dJointTypeContact ) ) )
		{
			// create an array of contact structures to hold the contact information
			dContact* pContacts = new dContact[ iMaxContacts ];
			// initialize the contact structures
			for ( int iContact = 0 ; iContact < iMaxContacts ; ++iContact )
			{
				pContacts[ iContact ].surface.mode = dContactBounce | dContactSoftCFM;
				pContacts[ iContact ].surface.mu = dInfinity;
				pContacts[ iContact ].surface.mu2 = dReal( 1.0 );
				pContacts[ iContact ].surface.bounce = dReal( 0.01 );
				pContacts[ iContact ].surface.bounce_vel = dReal( 0.01 );
				pContacts[ iContact ].surface.soft_cfm = dReal( 0.001 );
			}
			int iNumberOfContacts = dCollide( l_Geom1 , l_Geom2 , iMaxContacts , &pContacts[ 0 ].geom , sizeof( dContact ) );
			for ( int iContact = 0 ; iContact < iNumberOfContacts ; ++iContact )
			{
				dJointID l_dJoint = dJointCreateContact( pWorld->m_oWorldID , pWorld->m_oJointGroupID , pContacts + iContact );
				dJointAttach( l_dJoint , l_Body1 , l_Body2 );
			}
			delete[] pContacts;
		}
	}
};

void ChaosEngine::ODEPhysics::SetBodyPosition( const std::string& oBodyName , float* pPosition )
{
	BodyHash::iterator oIterator = m_oBodies.find( oBodyName );
	if ( oIterator != m_oBodies.end() )
	{
		dGeomSetPosition( oIterator->second->m_oGeomID , pPosition[ 0 ] , pPosition[ 1 ] , pPosition[ 2 ] );
	}
	else
	{
		Log::WriteMessage( std::string( "ODEPhysics - SetBodyPosition - Body doesn't exist" ) , Log::Warning );
	}
};

void ChaosEngine::ODEPhysics::SetBodyOrientation( const std::string& oBodyName , float* pOrientation )
{
	BodyHash::iterator oIterator = m_oBodies.find( oBodyName );
	if ( oIterator != m_oBodies.end() )
	{
		dQuaternion oQuat;
		oQuat[ 0 ] = pOrientation[ 3 ];
		oQuat[ 1 ] = pOrientation[ 0 ];
		oQuat[ 2 ] = pOrientation[ 1 ];
		oQuat[ 3 ] = pOrientation[ 2 ];
		dGeomSetQuaternion( oIterator->second->m_oGeomID , oQuat );
	}
	else
	{
		Log::WriteMessage( std::string( "ODEPhysics - SetBodyOrientation - Body doesn't exist" ) , Log::Warning );
	}
};

void ChaosEngine::ODEPhysics::SetBodyLinearVelocity( const std::string& oBodyName , float* pLinearVelocity )
{
	BodyHash::iterator oIterator = m_oBodies.find( oBodyName );
	if ( oIterator != m_oBodies.end() )
	{
		if ( oIterator->second->m_oBodyID != 0 )
		{
			dBodySetLinearVel( oIterator->second->m_oBodyID , pLinearVelocity[ 0 ] , pLinearVelocity[ 1 ] , pLinearVelocity[ 2 ] );
		}
	}
	else
	{
		Log::WriteMessage( std::string( "ODEPhysics - SetBodyLinearVelocity - Body doesn't exist" ) , Log::Warning );
	}
};

void ChaosEngine::ODEPhysics::SetBodyAngularVelocity( const std::string& oBodyName , float* pAngularVelocity )
{
	BodyHash::iterator oIterator = m_oBodies.find( oBodyName );
	if ( oIterator != m_oBodies.end() )
	{
		if ( oIterator->second->m_oBodyID != 0 )
		{
			dBodySetAngularVel( oIterator->second->m_oBodyID , pAngularVelocity[ 0 ] , pAngularVelocity[ 1 ] , pAngularVelocity[ 2 ] );
		}
	}
	else
	{
		Log::WriteMessage( std::string( "ODEPhysics - SetBodyAngularVelocity - Body doesn't exist" ) , Log::Warning );
	}
};

void ChaosEngine::ODEPhysics::GetBodyPosition( const std::string& oBodyName , float* pPosition )
{
	BodyHash::iterator oIterator = m_oBodies.find( oBodyName );
	if ( oIterator != m_oBodies.end() )
	{
		const float* pPos = dGeomGetPosition( oIterator->second->m_oGeomID );
		pPosition[ 0 ] = pPos[ 0 ];
		pPosition[ 1 ] = pPos[ 1 ];
		pPosition[ 2 ] = pPos[ 2 ];
	}
	else
	{
		Log::WriteMessage( std::string( "ODEPhysics - GetBodyPosition - Body doesn't exist" ) , Log::Warning );
	}
};

void ChaosEngine::ODEPhysics::GetBodyOrientation( const std::string& oBodyName , float* pOrientation )
{
	BodyHash::iterator oIterator = m_oBodies.find( oBodyName );
	if ( oIterator != m_oBodies.end() )
	{
		dQuaternion oQuaternion;
		dGeomGetQuaternion( oIterator->second->m_oGeomID , oQuaternion );
		pOrientation[ 0 ] = oQuaternion[ 1 ];
		pOrientation[ 1 ] = oQuaternion[ 2 ];
		pOrientation[ 2 ] = oQuaternion[ 3 ];
		pOrientation[ 3 ] = oQuaternion[ 0 ];
	}
	else
	{
		Log::WriteMessage( std::string( "ODEPhysics - GetBodyOrientation - Body doesn't exist" ) , Log::Warning );
	}
};

void ChaosEngine::ODEPhysics::GetBodyLinearVelocity( const std::string& oBodyName , float* pLinearVelocity )
{
	BodyHash::iterator oIterator = m_oBodies.find( oBodyName );
	if ( oIterator != m_oBodies.end() )
	{
		if ( oIterator->second->m_oBodyID != 0 )
		{
			const float* pVel = dBodyGetLinearVel( oIterator->second->m_oBodyID );
			pLinearVelocity[ 0 ] = pVel[ 0 ];
			pLinearVelocity[ 1 ] = pVel[ 1 ];
			pLinearVelocity[ 2 ] = pVel[ 2 ];
		}
		else
		{
			pLinearVelocity[ 0 ] = 0.0f;
			pLinearVelocity[ 1 ] = 0.0f;
			pLinearVelocity[ 2 ] = 0.0f;
		}
	}
	else
	{
		Log::WriteMessage( std::string( "ODEPhysics - GetBodyLinearVelocity - Body doesn't exist" ) , Log::Warning );
	}
};

void ChaosEngine::ODEPhysics::GetBodyAngularVelocity( const std::string& oBodyName , float* pAngularVelocity )
{
	BodyHash::iterator oIterator = m_oBodies.find( oBodyName );
	if ( oIterator != m_oBodies.end() )
	{
		if ( oIterator->second->m_oBodyID != 0 )
		{
			const float* pVel = dBodyGetAngularVel( oIterator->second->m_oBodyID );
			pAngularVelocity[ 0 ] = pVel[ 0 ];
			pAngularVelocity[ 1 ] = pVel[ 1 ];
			pAngularVelocity[ 2 ] = pVel[ 2 ];
		}
		else
		{
			pAngularVelocity[ 0 ] = 0.0f;
			pAngularVelocity[ 1 ] = 0.0f;
			pAngularVelocity[ 2 ] = 0.0f;
		}
	}
	else
	{
		Log::WriteMessage( std::string( "ODEPhysics - GetBodyAngularVelocity - Body doesn't exist" ) , Log::Warning );
	}
};

void ChaosEngine::ODEPhysics::CreateSpace( const std::string& oSpaceName )
{
};

void ChaosEngine::ODEPhysics::DestroySpace( const std::string& oSpaceName )
{
};

void ChaosEngine::ODEPhysics::CreateGeometry( const std::string& oGeometry , const std::string& oDescription )
{
};

void ChaosEngine::ODEPhysics::DestroyGeometry( const std::string& oGeometry )
{
};

void ChaosEngine::ODEPhysics::AddGeometryToSpace( const std::string& oSpaceName , const std::string& oGeometry )
{
};

void ChaosEngine::ODEPhysics::RemoveGeometryFromSpace( const std::string& oSpaceName , const std::string& oGeometry )
{
};

void ChaosEngine::ODEPhysics::SetGeometryPosition( const std::string& oGeometryName , float* pPosition )
{
};

void ChaosEngine::ODEPhysics::SetGeometryOrientation( const std::string& oGeometryName , float* pOrientation )
{
};

void ChaosEngine::ODEPhysics::GetGeometryPosition( const std::string& oGeometryName , float* pPosition )
{
};

void ChaosEngine::ODEPhysics::GetGeometryOrientation( const std::string& oGeometryName , float* pOrientation )
{
};

void ChaosEngine::ODEPhysics::EnumerateCollisions( const std::string& oSpaceName )
{
};

void ChaosEngine::ODEPhysics::EnumerateCollisions( const std::string& oSpaceName , const std::string& oGeometryName )
{
};

