package fortytwo.testserver;

/**
 * <p>Title: Forty Two Unlimited</p>
 * <p>Description: Texas Forty Two Game</p>
 * <p>Copyright: Copyright (c) 2004</p>
 * <p>Company: 42Unlimited</p>
 * @author Nathaniel Normandin
 * @version 0.00.01
 */

import java.net.Socket;
import fortytwo.network.*;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Vector;

public class ServerMessageManager {

  // GAME STATES
  private static final int STATE_DISCONNECTED = 0;
  private static final int STATE_CONNECTED = 1;
  private static final int STATE_LOGGEDIN = 2;
  // GAME STATES END

  // the current state of the game
  private int currentState;

  // ProtocolHandler lets us send and received application layer PDUs ( Message )
  private ProtocolHandler protocolHandler;
  private Socket socket;

  // Thread for handling the processing of messages
  private Thread messageProcessingThread;
  private MessageListener messageListener;

  private ArrayList tableListeners, playerListeners, lobbyListeners;

  public void addTableListener( TableListener olistener ) {
    synchronized( tableListeners ) {
      if ( !tableListeners.contains(olistener) ) {
        tableListeners.add(olistener);
      }
    }
  }

  public void addPlayerListener( PlayerListener olistener ) {
    synchronized( playerListeners ) {
      if ( !playerListeners.contains(olistener) ) {
        playerListeners.add(olistener);
      }
    }
  }


  public void addLobbyListener( LobbyListener olistener ) {
    synchronized( lobbyListeners ) {
      if ( !lobbyListeners.contains(olistener) ) {
        lobbyListeners.add(olistener);
      }
    }
  }

  public void removeTableListener( TableListener oListener ) {
    synchronized( tableListeners ) {
      if ( tableListeners.contains( oListener ) )
        tableListeners.remove( oListener );
    }
  }

  public void removePlayerListener( PlayerListener oListener ) {
    synchronized( playerListeners ) {
      if ( playerListeners.contains( oListener ) )
        playerListeners.remove( oListener );
    }
  }

  public void removeLobbyListener( LobbyListener oListener ) {
    synchronized( lobbyListeners ) {
      if ( lobbyListeners.contains( oListener ) )
        lobbyListeners.remove( oListener );
    }
  }

  public ServerMessageManager() {
    lobbyListeners = new ArrayList();
  }

  public ServerMessageManager ( Socket tempSocket, MessageListener tempMessageListener, LobbyListener lobbyListener ) {
    messageListener = tempMessageListener;
    socket = tempSocket;
    tableListeners = new ArrayList();
    playerListeners = new ArrayList();
    lobbyListeners = new ArrayList();
    addLobbyListener( lobbyListener );
    protocolHandler = new ProtocolHandler( socket );
    currentState = STATE_CONNECTED;
    // since we are connected, start the thread to process incomming messages
    messageProcessingThread = new Thread(new MessageProcessor());
    messageProcessingThread.start();
    return;
  }

  public void disconnect() {
    try {
      protocolHandler.sendMessage( new Disconnect() );
      socket.wait( 1000 );
      socket.close();
    }
    catch (IOException ioException ) {
      messageListener.messageReceived( "Error closing Client Socket" );
    }
    catch ( InterruptedException iException ) {
      messageListener.messageReceived( "Error closing Client Socket" );
    }
    catch ( IllegalMonitorStateException IMSException ) {

    }
  }

  public boolean isConnected() {
    return ( currentState == STATE_CONNECTED || currentState == STATE_LOGGEDIN );
  }

  public void errorMessage( ErrorMessage oError ) {
    protocolHandler.sendMessage( oError );
  }

  public void welcomePlayer(String username) {
    currentState = STATE_LOGGEDIN;
    if (currentState == STATE_LOGGEDIN) {
      protocolHandler.sendMessage(new WelcomePlayer( "Welcome " + username ) );
    }
    else {
      if (currentState == STATE_CONNECTED) {
        // Not connected, please connect before attempting to log in
        // TODO: throw a NotConnectedException
      }
      else if (currentState == STATE_DISCONNECTED) {
        // Already logged in
        // TODO: throw an AlreadyLoggedInException
      }
    }
  }

  public void lobbyChatList( LobbyChatList lobbyChatList ) {
    if ( currentState == STATE_LOGGEDIN ) {
      protocolHandler.sendMessage( lobbyChatList );
    }
  }

  public void lobbyChat( LobbyChat lobbyChat ) {
    if ( currentState == STATE_LOGGEDIN ) {
      protocolHandler.sendMessage( lobbyChat );
    }
  }

  public void tableChat( TableChat oTableChat ) {
    if ( currentState == STATE_LOGGEDIN ) {
      protocolHandler.sendMessage( oTableChat );
    }
  }

  public void lobbyTableList( LobbyTableList lobbyTableList ) {
    if ( currentState == STATE_LOGGEDIN ) {
      protocolHandler.sendMessage( lobbyTableList );
    }
  }

  public void loadTable( Vector players, Rules rules, Vector dominos, int dealer ) {
    if ( currentState == STATE_LOGGEDIN )
      protocolHandler.sendMessage( new LoadTable( players, rules, dominos, dealer ) );
  }

  class MessageProcessor implements Runnable {
    public void run() {
      while( protocolHandler.isConnected() ) {
        if ( protocolHandler.hasMessage() ) {
          Message oMessage = protocolHandler.nextMessage();
          switch( currentState ) {
            case STATE_CONNECTED:
            {
              if ( oMessage.getHeader().equals( PlayerLogin.type ) ) {
                PlayerLogin oCommand = new PlayerLogin( oMessage );
                // call method to verify login
                for (int i = 0; i < playerListeners.size(); i++ ) {
                  PlayerListener oListener = ( PlayerListener )playerListeners.get( i );
                  // call the corresponding method
                  oListener.playerLoginReceived( oCommand );
                }
              }
              else if ( oMessage.getHeader().equals( Disconnect.type ) ) {
                Disconnect oCommand = new Disconnect( oMessage );
                for ( int i = 0; i < playerListeners.size(); i ++ ) {
                  PlayerListener oListener = (PlayerListener) playerListeners.get(i);
                  oListener.disconnectReceived();
                }
              }
            }// case end

            case STATE_LOGGEDIN:
            {
              if ( oMessage.getHeader().equals( LobbyChat.type ) ) {
                LobbyChat oCommand = new LobbyChat( oMessage );
                for( int i=0; i < lobbyListeners.size(); i ++ ) {
                  LobbyListener oListener = (LobbyListener) lobbyListeners.get( i );
                  oListener.lobbyChatReceived( oCommand );
                }
              }
              else if ( oMessage.getHeader().equals( TableChat.type ) ) {
                TableChat oCommand = new TableChat( oMessage );
                for ( int i = 0; i < playerListeners.size(); i ++ ) {
                  PlayerListener oListener = (PlayerListener) playerListeners.get(i);
                  oListener.tableChatReceived( oCommand );
                }
              }
              else if ( oMessage.getHeader().equals( Disconnect.type ) ) {
                Disconnect oCommand = new Disconnect( oMessage );
                for ( int i = 0; i < playerListeners.size(); i ++ ) {
                  PlayerListener oListener = (PlayerListener) playerListeners.get(i);
                  oListener.disconnectReceived();
                }
              }
              else if ( oMessage.getHeader().equals( CreateTable.type ) ) {
                CreateTable oCommand = new CreateTable( oMessage );
                for ( int i = 0; i < playerListeners.size(); i ++ ) {
                  PlayerListener oListener  = ( PlayerListener ) playerListeners.get(i);
                  oListener.createTableReceived( oCommand );
                }
              }
              else if ( oMessage.getHeader().equals( JoinTable.type  ) ) {
                JoinTable oCommand = new JoinTable( oMessage );
                for ( int i = 0; i < playerListeners.size(); i ++ ) {
                  PlayerListener oListener = (PlayerListener) playerListeners.get(i);
                  oListener.joinTableReceived( oCommand );
                }

              }
            }
            case STATE_DISCONNECTED:
            {

            }
          }// switch end
        }// if end
        else
        {
          // Sleep some so as to not hog the CPU
          try {
            Thread.sleep(50);
          }
          catch (InterruptedException oException) {}
        }

      }// while end
    }// run() end
  }


}
