// Read in *.ppm, *.bmp, and *.bin files
#include "stdafx.h"
#include "Readfile.h"
#include "DIB.h"

unsigned char *ImageBuffer;

// Read a line in a file
int fReadLine(FILE * fp, char buff[])
{
  char ch;
  unsigned int num = 0;
  ch = getc(fp);
  while((ch!=0x0A) && (!feof(fp)) && (num<256)) {
    buff[num++] = ch;
    ch = getc(fp);
  }
  buff[num] = 0;
  return(num);
}
/*  PPMFileRead : read the color image from a .ppm file */
unsigned char *PPMFileRead(FILE *fh, int *imgwidth, int *imgheight)
{
  char buff[258];
  int row, col, index;
  int width, height;
  int imgsize, size, maxval;
  int header = 100;
  unsigned char u_ch;

  /* verify the I.D. bytes of the .ppm file */
  fReadLine(fh, buff);
  if(buff[0] != 'P' || buff[1] != '6')  return(0);
  /* read the width & height of image */
  fReadLine(fh, buff);
  while(buff[0] == '#')  fReadLine(fh, buff);
  sscanf(buff,"%d%d", &width, &height);
  imgsize = width * height;
  size = imgsize * 3 + header;
  /* read the maximum color index value */
  fReadLine(fh, buff);
  while(buff[0] == '#')  fReadLine(fh, buff);
  sscanf(buff,"%d", &maxval);
  /* read the color image in .ppm file */
  ImageBuffer = (unsigned char *) malloc(size);
  if(ImageBuffer == NULL)
  {
  	MessageBox(NULL,"insufficient memory available","Memory Allocation ERROR",MB_OK);
  	return NULL;
  }

  size = fread(ImageBuffer, size, 1, fh);

  index = 0;
  for(row=0;row<height;row++)
  {
			for(col=0;col<width;col++)
			{
				index=row*width*3+col*3;
				u_ch=ImageBuffer[index];
				ImageBuffer[index]=ImageBuffer[index+2];
				ImageBuffer[index+2]=u_ch;
			}
  }
  *imgwidth = width;
  *imgheight = height;
  return ImageBuffer;
}

/*  BMPFileRead : read the color image from a .BMP file
*/
unsigned char *BMPFileRead(FILE *fh, int *imgwidth, int *imgheight)
{
  char *buff, *tempbuff;
  int row, col, bmpindex, ppmindex;
  int offset, compression;
  char b,m;
  short pixel, bitcount;
  int bytesread, size, imgsize;
  int width, height;
  int pad;

  buff = (char *)malloc(51*sizeof(char));
  if(buff == NULL)
  {
  	MessageBox(NULL,"insufficient memory available","Memory Allocation ERROR",MB_OK);
  	return NULL;
  }
  /* verify the I.D. bytes of the .bmp file */
  bytesread = fread(buff, 51, 1, fh);
  if(bytesread != 1)
	MessageBox(NULL,"Unable to read file 26","File Read ERROR",MB_OK);

  sscanf(buff,"%c%c",&b,&m);
  if(buff[0] != 'B' || buff[1] != 'M')  return(0);
  /* read the width & height of image */
  memcpy(&size,buff + 2,sizeof(int));
  memcpy(&offset,buff + 10,sizeof(int));
  memcpy(&width,buff + 18,sizeof(int));
  memcpy(&height,buff + 22,sizeof(int));
  memcpy(&bitcount,buff + 28,sizeof(short));
  pad = (width*(bitcount/8))%4;
  if((bitcount != 16) && (bitcount != 24) && (bitcount != 8))
  {
	MessageBox(NULL,"Format Not Supported","Error",MB_OK);
	return NULL;
  }
  memcpy(&compression,buff + 30,sizeof(int));
  if(compression != 0)
  {
	MessageBox(NULL,"Compressed Format Not Supported","Error",MB_OK);
	return NULL;
  }
  free(buff);

  imgsize = (width+pad) * height;
  size    = imgsize * 3 + offset;

  /* read the color image in .ppm file */
  tempbuff = (char *) malloc(size*sizeof(char));
  ImageBuffer = (unsigned char *) malloc(size*sizeof(char));
  if((tempbuff == NULL)||(ImageBuffer == NULL))
  {
  	MessageBox(NULL,"insufficient memory available","Memory Allocation ERROR",MB_OK);
  	return NULL;
  }
  size = fread(tempbuff, 1, size, fh);
  if(size < (bitcount*width*height)/8)
	MessageBox(NULL,"Unable to read file 27","File Read ERROR",MB_OK);


  bmpindex = 0;
  ppmindex = 0;

  if(bitcount == 8)
  {
	  for(row=0;row<height;row++)
	  {

		for(col=0;col<width;col++)
		{
			ppmindex=(height-row-1)*width*3+col*3;
			bmpindex=row*(width+pad)+col;
				
			ImageBuffer[ppmindex]=tempbuff[bmpindex+offset-51];
			ImageBuffer[ppmindex+1]=tempbuff[bmpindex+offset-51];
			ImageBuffer[ppmindex+2]=tempbuff[bmpindex+offset-51];
		}
	  }
  }
  else if(bitcount == 16)
  {
	  for(row=0;row<height;row++)
	  {

		for(col=0;col<width;col++)
		{
			ppmindex=(height-row-1)*width*3+col*3;

			memcpy(&pixel,tempbuff+offset-51+bmpindex,2);
				
			ImageBuffer[ppmindex]=((pixel)&0x1F)*8;
			ImageBuffer[ppmindex+1]=((pixel>>5)&0x1F)*8;
			ImageBuffer[ppmindex+2]=((pixel>>10)&0x3F)*8;

			bmpindex+=2;
		}
		bmpindex+=pad;
	  }
  }
  else if(bitcount == 24)
  {
	  for(row=0;row<height;row++)
	  {

		for(col=0;col<width;col++)
		{
			ppmindex=(height-row-1)*width*3+col*3;
			bmpindex=row*(width*3+pad)+col*3;
				
			ImageBuffer[ppmindex]=tempbuff[bmpindex+offset-51];
			ImageBuffer[ppmindex+1]=tempbuff[bmpindex+offset-51+1];
			ImageBuffer[ppmindex+2]=tempbuff[bmpindex+offset-51+2];
		}
	  }
  }

  *imgwidth = width;
  *imgheight = height;
  free(tempbuff);
  return ImageBuffer;
}

void PPMFileWrite(FILE * fh, unsigned char * buffer, int width, int height)
{
  char buff[258];
  char *temp;
  unsigned int i;
  int size;

  size = 3*width*height;
  temp = (char*)malloc(size*sizeof(char));
  /* write the I.D. bytes of the .ppm file */
  sprintf(buff, "P6\n");
  for(i=0; i<strlen(buff); i++)  putc(buff[i], fh);
  /* write the width & height of image */
  sprintf(buff, "%d %d\n", width, height);
  for(i=0; i<strlen(buff); i++)  putc(buff[i], fh);
  /* write the maximum color index value */
  sprintf(buff, "%d\n", 255);
  for(i=0; i<strlen(buff); i++)  putc(buff[i], fh);
  /* write the grey image to .ppm file */
  if(buffer != NULL)
  {
	for(i=0;i<(unsigned)size;i+=3)
	{
		temp[i] = buffer[i+2];
		temp[i+1] = buffer[i+1];
		temp[i+2] = buffer[i];
	}
    fwrite(temp, size, 1, fh);
  }
  free(temp);
  return;
}

/*  BMPFileWrite : Write the color image to a .BMP file
*/
void BMPFileWrite(FILE * fh, unsigned char * buffer, int width, int height)
{
  int offset = 54, compression = 0;
  short pixel = 1, bitcount = 24;
  int bmpindex, ppmindex, h, w;
  int size, filesize;
  int headersize;
  char buff[8];
  char *tempbuff;
  int i, pad;

  for(i=0;i<8;i++)
	  buff[i] = 0;

  fwrite("B",sizeof(char),1,fh);
  fwrite("M",sizeof(char),1,fh);

  size = 3*width*height;
  filesize = size + offset;
  headersize = offset - 3*sizeof(int) - sizeof(short);

  fwrite(&filesize,sizeof(int),1,fh);
//why sizeof(short) instead of 4*sizeof(char)??
  fwrite(buff,2*sizeof(short),1,fh);
  fwrite(&offset,sizeof(int),1,fh);

  fwrite(&headersize,sizeof(int),1,fh);
  fwrite(&width,sizeof(int),1,fh);
  fwrite(&height,sizeof(int),1,fh);
  fwrite(&pixel,sizeof(short),1,fh);
  fwrite(&bitcount,sizeof(short),1,fh);
  fwrite(&compression,sizeof(int),1,fh);
  // without compression, can set this to 0
  fwrite(&size,sizeof(int),1,fh);
  fwrite(buff,sizeof(int),1,fh);
  // deviates from his own format here by combining two fields into one fwrite
  // also, lenna.bmp naturally has these set to 2858
  fwrite(buff,2*sizeof(int),1,fh);
  // here is the error, methinks
  fwrite(buff,sizeof(int)+sizeof(short),1,fh);
//  fwrite(buff,sizeof(int),1,fh);

  pad = (width*(bitcount/8))%4;

  bmpindex = 0;
  ppmindex = 0;
  tempbuff = (char*)malloc((size+pad*height)*sizeof(char));

  for(h=0;h<height;h++) 
 	for(w=0;w<width;w++) 
	{
		ppmindex=(height-h-1)*(width*3+pad)+w*3;
		bmpindex=h*width*3+w*3;
		
		if(ppmindex>1)
		{
			tempbuff[ppmindex]=buffer[bmpindex+2];
			tempbuff[ppmindex-2]=buffer[bmpindex];
			tempbuff[ppmindex-1]=buffer[bmpindex+1];
		}
		else if(ppmindex==1)
		{
			tempbuff[ppmindex]=buffer[bmpindex+2];
			tempbuff[ppmindex+width-1]=buffer[bmpindex];
			tempbuff[ppmindex-1]=buffer[bmpindex+1];
		}
		else if(ppmindex==0)
		{
			tempbuff[ppmindex]=buffer[bmpindex+2];
			tempbuff[ppmindex+width-2]=buffer[bmpindex];
			tempbuff[ppmindex+width-1]=buffer[bmpindex+1];
		}
	}

  fwrite(tempbuff,size+pad*height,1,fh);
  free(tempbuff);
  return;
}


int WVFFileWrite(CWaveletSettings *wavelet, FILE *fh, char *comp, int ylength, int ulength, int vlength, int width, int height)
{
	int i, size;

	fwrite(&wavelet->hlen,sizeof(short),1,fh);
	for(i=0;i<wavelet->hlen;i++)
		fwrite(&wavelet->h2[i],sizeof(double),1,fh);
	fwrite(&wavelet->glen,sizeof(short),1,fh);
	for(i=0;i<wavelet->glen;i++)
		fwrite(&wavelet->g2[i],sizeof(double),1,fh);

	size = 2*sizeof(short)+(wavelet->hlen+wavelet->glen)*sizeof(double);

	fwrite(&wavelet->m_mrlevel,sizeof(int),1,fh);
	fwrite(&wavelet->m_stepsize,sizeof(int),1,fh);
	fwrite(&width,sizeof(int),1,fh);
	fwrite(&height,sizeof(int),1,fh);
	size += 4*sizeof(int);

	fwrite(&ylength,sizeof(int),1,fh);
	fwrite(comp,ylength,1,fh);
	fwrite(&ulength,sizeof(int),1,fh);
	fwrite(comp+ylength,ulength,1,fh);
	fwrite(&vlength,sizeof(int),1,fh);
	fwrite(comp+ylength+ulength,vlength,1,fh);

	size += ylength+ulength+vlength+3*sizeof(int);

	return size;
}

char *WVFFileRead(CWaveletSettings *wavelet, FILE * fh, int *ylength, int *ulength, int *vlength, int *width, int *height)
{
	int i;
	char *comp;
	char *ycomp, *ucomp, *vcomp;

	fread(&wavelet->hlen,sizeof(short),1,fh);
	for(i=0;i<wavelet->hlen;i++)
		fread(&wavelet->h2[i],sizeof(double),1,fh);
	fread(&wavelet->glen,sizeof(short),1,fh);
	for(i=0;i<wavelet->glen;i++)
		fread(&wavelet->g2[i],sizeof(double),1,fh);

	fread(&wavelet->m_mrlevel,sizeof(int),1,fh);
	fread(&wavelet->m_stepsize,sizeof(int),1,fh);
	fread(width,sizeof(int),1,fh);
	fread(height,sizeof(int),1,fh);

	fread(ylength,sizeof(int),1,fh);
	ycomp = new char[ylength[0]];
	fread(ycomp,ylength[0],1,fh);
	fread(ulength,sizeof(int),1,fh);
	ucomp = new char[ulength[0]];
	fread(ucomp,ulength[0],1,fh);
	fread(vlength,sizeof(int),1,fh);
	vcomp = new char[vlength[0]];
	fread(vcomp,vlength[0],1,fh);

	comp = new char[ylength[0]+ulength[0]+vlength[0]];
	memcpy(comp,ycomp,ylength[0]);
	memcpy(comp+ylength[0],ucomp,ulength[0]);
	memcpy(comp+ylength[0]+ulength[0],vcomp,vlength[0]);


	delete[] ycomp;
	delete[] ucomp;
	delete[] vcomp;

	return comp;
}

void rgb_to_yuv(unsigned char *hpRGBbuf, double *hpYbuf, double *hpUbuf, double *hpVbuf, int width, int height)
{
	int i, indx1, indx2;
	short *U, *V;
	int w,h;

	U = new short[width*height];
	V = new short[width*height];

	// prepare Y component
	// indx1: index of hpSrcBuf
	// indx2: index of hpYSrcBuf
	indx1=0;
	indx2=0;
	for(i=0;i<width*height;i++)
	{
		hpYbuf[indx2]=(hpRGBbuf[indx1]+(hpRGBbuf[indx1+1]*2)+hpRGBbuf[indx1+2])/4;
		U[indx2]=hpRGBbuf[indx1]-hpRGBbuf[indx1+1];
		V[indx2]=hpRGBbuf[indx1+2]-hpRGBbuf[indx1+1];
		indx1+=3;
		indx2++;
	}

	for(w=0;w<(width>>1);w++)
		for(h=0;h<(height>>1);h++)
		{
			indx1 = (w<<1)+(h<<1)*width;
			indx2 = w+h*(width>>1);
			hpUbuf[indx2] = (U[indx1]+U[indx1+1]+U[indx1+width]+U[indx1+width+1])>>2;
			hpVbuf[indx2] = (V[indx1]+V[indx1+1]+V[indx1+width]+V[indx1+width+1])>>2;
		}

	delete[] U;
	delete[] V;

	return;
}

void yuv_to_rgb(double *hpYbuf, double *hpUbuf, double *hpVbuf, unsigned char *hpRGBbuf, int width, int height)
{
	int row, col;
	int rgb, y, uv;
	double sR, sG, sB;

	for(row=0;row<height;row++)
	{
		for(col=0;col<width;col++)
		{
			y = col+row*width;
			rgb = 3*y;
			uv = (col>>1)+(row>>1)*(width>>1);

			sG=hpYbuf[y]-((hpUbuf[uv]+hpVbuf[uv])/4);
			sR=hpUbuf[uv]+sG;
			sB=hpVbuf[uv]+sG;

			if(sR > 255)
				hpRGBbuf[rgb] = 255;
			else if(sR < 0)
				hpRGBbuf[rgb] = 0;
			else
				hpRGBbuf[rgb]=unsigned char(sR);

			if(sG > 255)
				hpRGBbuf[rgb+1] = 255;
			else if(sG < 0)
				hpRGBbuf[rgb+1] = 0;
			else
				hpRGBbuf[rgb+1]=unsigned char(sG);

			if(sB > 255)
				hpRGBbuf[rgb+2] = 255;
			else if(sB < 0)
				hpRGBbuf[rgb+2] = 0;
			else
				hpRGBbuf[rgb+2]=unsigned char(sB);
		}
	}
	return;
}

void displayimage(unsigned char *image, int width, int height, HWND hWnd, double stretch)
{
	int pad, j;
	unsigned char *display;

	// Display the Image
	pad = (width*3)%4;
	if(pad != 0)
	{
		display = (unsigned char *) malloc((width+pad)*height*3*sizeof(char));

		for(j=0;j<height;j++)
			memcpy(display+3*j*(width+pad),image+3*j*width,3*width*sizeof(char));

			CreateDIBDisplay(hWnd, display,
				height, width+pad, 3, 3*(width+pad)*height+pad*height*3, stretch);

				free(display);
	}
	else
	{
		CreateDIBDisplay(hWnd, image,
			height, width, 3, 3*width*height, stretch);
	}
}
