// Encode.cpp
// Stack-run encoding to compress 3D 
// Wavelet transformed video
#include "stdafx.h"
#include "Encode.h"

char *buf;
unsigned int zerorun;
int buflength;

// Stack run symbols
short plus = 3;
short minus = 2;
short one = 1;
short zero = 0;


char bitstack;
short bitcount, bitpos;
int coeffval;
BOOL firstdecode;
BOOL run;

// Encode the background in the VO compression mode
char* EncodeFrame(double *GOF, int width, int height, int level2D, int *codelength)
{
	char *retbuf;
	int startw,endw,starth,endh;
	int w,h,k;

	// Allocate temporary buffer for the encoded frame
	buf = (char *)malloc(width*height*sizeof(short));
	if(buf == NULL)
	{
		MessageBox(NULL,"insufficient memory available 46","Memory Allocation ERROR",MB_OK);
		return NULL;
	}
	// initialize various parameters
	buflength = 0;
	bitstack = 0;
	bitcount = 0;
	zerorun = 0;


	// the end vertical and horizontal
	// positions of the subband
	endw = width;
	endh = height;
	for(k=0;k<level2D;k++)
	{
		// break if the subband is 2 pixels
		// or less in either direction
		if((endw < 2) || (endh < 2))
			break;

		// Find starting end ending values of the subbands
		startw = width>>(k+1);
		if(width%((int)pow(2,k+1)) != 0)
			startw++;
		starth = height>>(k+1);
		if(height%((int)pow(2,k+1)) != 0)
			starth++;

		// Encode the Subbands -- 

		//HH subband
		for(h=starth;h<endh;h++)
			for(w=startw;w<endw;w++)
					writevalue(GOF[w+width*h]);
		//HL subband
		for(w=startw;w<endw;w++)
			for(h=0;h<starth;h++)
					writevalue(GOF[w+width*h]);
		//LH subband
		for(h=starth;h<endh;h++)
			for(w=0;w<startw;w++)
					writevalue(GOF[w+width*h]);

		// keep track of the LL band
		endw = startw;
		endh = starth;
	}

	// Encode the LL band
	for(h=0;h<endh;h++)
		for(w=0;w<endw;w++)
			writevalue(GOF[w+width*h]);

	// Encode the remaining run length
	if(zerorun != 0)
	{
		writestack(zerorun, TRUE);
		zerorun = 0;
	}
	// Write the remaining bit-stack
	if(bitstack != 0)
	{
		memcpy(buf,&bitstack,sizeof(char));
		buf+=sizeof(char);
		buflength+=sizeof(char);
		bitstack = 0;
		bitcount = 0;
	}

	// return buffer pointer
	buf = buf - buflength;
	// generate a new buffer with length of the encoded video
	retbuf = (char *)malloc(buflength*sizeof(char));
	if(retbuf == NULL)
	{
		MessageBox(NULL,"insufficient memory available 47","Memory Allocation ERROR",MB_OK);
		return NULL;
	}
	// copy the encoded data
	memcpy(retbuf,buf,buflength);
	// return buffer length
	*codelength = buflength;
	// free temporary buffer
	free(buf);

	return retbuf;
}

double* DecodeFrame(char *decodebuf, int width, int height, int level2D, int codelength)
{
	double *GOF;
	int startw,endw,starth,endh;
	int w,h,k;

	// Allocate space for the decoded frame
	GOF = (double*)malloc(width*height*sizeof(double));
	// Allocate space for the encoded frame
	buf = (char *)malloc(codelength*sizeof(char));
	if((GOF == NULL)||(buf == NULL))
	{
		MessageBox(NULL,"insufficient memory available 48","Memory Allocation ERROR",MB_OK);
		return NULL;
	}

	// copy the encoded data
	memcpy(buf,decodebuf,codelength*sizeof(char));

	// initialize various parameters
	buflength = 0;
	bitcount = 0;
	bitpos = 0;
	coeffval =0;
	firstdecode = TRUE;
	zerorun = 0;

	// the end vertical and horizontal
	// positions of the subband
	endw = width;
	endh = height;
	for(k=0;k<level2D;k++)
	{
		// break if the subband is 2 pixels
		// or less in either direction
		if((endw < 2) || (endh < 2))
			break;

		// Find starting values of the subbands
		startw = width>>(k+1);
		if(width%((int)pow(2,k+1)) != 0)
			startw++;
		starth = height>>(k+1);
		if(height%((int)pow(2,k+1)) != 0)
			starth++;

		// Decode the subbands --

		//HH subband
		for(h=starth;h<endh;h++)
			for(w=startw;w<endw;w++)
					GOF[w+width*h] = readvalue();
		//HL subband
		for(w=startw;w<endw;w++)
			for(h=0;h<starth;h++)
					GOF[w+width*h] = readvalue();
		//LH subband
		for(h=starth;h<endh;h++)
			for(w=0;w<startw;w++)
					GOF[w+width*h] = readvalue();

		// keep track of the LL band
		endw = startw;
		endh = starth;
	}

	// decode the LL band
	for(h=0;h<endh;h++)
		for(w=0;w<endw;w++)
			GOF[w+width*h] = readvalue();

	// return the buffer pointer
	buf = buf-buflength;
	// free temporary buffer
	free(buf);

	return GOF;
}

// Stack-run Encoding
// Write a wavelet coefficient value
// into the stack-run encoder
void writevalue(double value)
{
	// If there is no zero run, and
	// the next coefficient is not zero,
	// just write the coefficient
	if((zerorun == 0) && (value != 0))
	{
		writestack(value, FALSE);
	}
	// If there is a zero run, and
	// the next coefficient is not zero,
	// then we have to write the run value
	// and the coefficient
	else if(value != 0)
	{
		writestack(zerorun, TRUE);
		zerorun = 0;
		writestack(value, FALSE);
	}
	// If the value is zero, then we just 
	// add to the zero run
	else //if(value == 0)
		zerorun++;
}

// Stack-run Decoding
// Read a wavelet coefficient value
// from the encoded buffer
double readvalue(void)
{
	double value;
	BOOL retrun;

	// If we have a run of zeros,
	// just return a zero and decrement
	// the zero-run counter
	if(zerorun > 0)
	{
		zerorun--;
		return 0;
	}
	// if we do not have a run of zeros,
	// then we must read a value from the 
	// encoded bitstream
	else
	{
		// read a value from the encoded file
		value =  readstack(&retrun);
		// if it is a zero-run value, return 0
		// and decrement the zero-run counter
		if(retrun == TRUE)
		{
			zerorun = ((unsigned)value)-1;
			return 0;
		}
		// otherwise, return the coefficient value
		else
			return value;
	}
}
		

void writestack(double dvalue, BOOL writerun)
{
	short bitval;
	int value;
	BOOL negative;
	BOOL writemsb;

	// find if a negative coefficient has
	// been written
	if((dvalue < 0) && (writerun == FALSE))
	{
		value = -(int)dvalue;
		// we must add one to the coefficent value
		// so coefficients cannot be confused with
		// run values
		value++;
		negative = TRUE;
	}
	// find if a positive coefficent has
	// been written
	else if(writerun == FALSE)
	{
		// we must add one to the coefficent value
		// so coefficients cannot be confused with
		// run values
		value = (int)dvalue+1;
		negative = FALSE;
	}
	else
	{
		value = (int)dvalue;
		negative = FALSE;
	}

	// we assume first a coefficient value, or
	// a small run value.  Large run values and coefficient values do 
	// not need to have the msb written to the file
	writemsb = TRUE;
	for(;;)
	{
		// if the bit-stack is full (8 bits)
		// it needs to be written to the encoded buffer
		if(bitcount == 4)
		{
			memcpy(buf,&bitstack,sizeof(char));
			buf+=sizeof(char);
			buflength+=sizeof(char);
			bitstack = 0;
			bitcount = 0;
		}
		// extract the bit value
		bitval = 0x0001&value;
		value = value>>1;

		if(bitval == 0)
		{
			// if the bitvalue is zero, then this must
			// be either a large run value or a coefficient value.
			// either way, we do not have to write the msb
			writemsb = FALSE;
			// Write a run symbol for 0
			if(writerun == FALSE)
			{
				bitstack = (zero<<(2*bitcount))|bitstack;
			}
			// write a coefficient symbol for 0
			else
			{
				bitstack = (minus<<(2*bitcount))|bitstack;
			}
			bitcount++;
		}
		else
		{
			// Write a run symbol for 1
			if(writerun == FALSE)
			{
				bitstack = (one<<(2*bitcount))|bitstack;
				writemsb = FALSE;
			}
			// write a coefficient symbol for 1
			else
			{
				bitstack = (plus<<(2*bitcount))|bitstack;
			}
			bitcount++;
		}

		// if the bit-stack is full (8 bits)
		// it needs to be written to the encoded buffer
		if(bitcount == 4)
		{
			memcpy(buf,&bitstack,sizeof(char));
			buf+=sizeof(char);
			buflength+=sizeof(char);
			bitstack = 0;
			bitcount = 0;
		}

		// if the last bit (msb) is left, and we don't have 
		// to write the msb, then break
		if((value == 1)&&(negative == TRUE)&&(writemsb==FALSE))
		{
			// write the sign of the coefficient value
			if(writerun == FALSE)
			{
				bitstack = (minus<<(2*bitcount))|bitstack;
				bitcount++;
			}
			break;
		}
		else if((value == 1)&&(negative == FALSE)&&(writemsb==FALSE))
		{
			// write the sign of the coefficient value
			if(writerun == FALSE)
			{
				bitstack = (plus<<(2*bitcount))|bitstack;
				bitcount++;
			}
			break;
		}
		// break if the value has been completely written
		else if(value == 0)
		{
			break;
		}
	}
}

double readstack(BOOL *retrun)
{
	double retvalue;
	int symbolval, value;
	BOOL writemsb;

	// For large run-length values and coefficient
	// values, we do not have to write the msb
	writemsb = TRUE;
	for(;;)
	{
		// if the byte-stack is empty, we need to read
		// another full byte (8-bits) into the buffer
		if(bitcount == 0)
		{
			memcpy(&bitstack,buf,sizeof(char));
			buf+=sizeof(char);
			buflength+=sizeof(char);
		}

		// get the symbol value
		symbolval = (bitstack>>(2*bitcount))&0x0003;
		bitcount=(bitcount+1)%4;

		// we know that it is a run if the first symbol read 
		// is a plus or minus
		if((bitpos==0)&&((symbolval==plus)||(symbolval==minus)))
			run = TRUE;
		// we know that it is a coefficient if the first symbol read 
		// is a zero or a one
		else if((bitpos==0)&&((symbolval==zero)||(symbolval==one)))
			run = FALSE;

		// if we are at the end of a significant coefficient, return 
		// the coefficient
		// positive coefficient
		if((symbolval == plus)&&(run == FALSE))
		{
			coeffval = coeffval|(1<<bitpos);
			value = coeffval;
			coeffval = 0;
			bitpos = 0;
			retrun[0] = run;
			retvalue = value-1;
			return retvalue;
		}
		// negative coefficient
		else if((symbolval == minus)&&(run == FALSE))
		{
			coeffval = coeffval|(1<<bitpos);
			value = coeffval;
			coeffval = 0;
			bitpos = 0;
			retrun[0] = run;
			retvalue = -(value-1);
			return retvalue;
		}

		// if we are at the end of a run, return the run
		else if(((symbolval == zero)||(symbolval==one))&&(run==TRUE))
		{
			// add the msb if it was not written into the encoded data
			if(writemsb == FALSE)
				coeffval = coeffval|(1<<bitpos);
			// reset parameters for next coefficient
			value = coeffval;
			coeffval = 0;
			if(symbolval == one)
				coeffval = 1;
			bitpos = 1;
			// return the run boolean
			retrun[0] = run;
			// the next is a coefficient value
			run = FALSE;
			// return the run value
			retvalue = value;
			return retvalue;
		}

		// if we have a run, add a bit
		else if((symbolval == plus)&&(run == TRUE))
		{
			coeffval = coeffval|(1<<bitpos);
			bitpos++;
		}
		else if((symbolval == minus)&&(run == TRUE))
		{
			writemsb = FALSE;
			bitpos++;
		}

		// if we have a coefficient, add a bit
		else if(symbolval == one)
		{
			coeffval = coeffval|(1<<bitpos);
			bitpos++;
		}
		else if(symbolval == zero)
		{
			bitpos++;
		}
		else
			MessageBox(NULL,"Error in Stack-Run Decoding","ERROR",MB_OK);

	}
}