import java.sql.*;

/**
 *  Helper class with functions for querying database.
 *  Functions are made static so they can be called
 *  without creating an instance object.
 *
 * @author     Brandon Douthit-Wood
 * @created    March 31, 2004
 */
public class Query {

	private final static String URL = "jdbc:mysql://137.229.143.72:3306/movie";
	private final static String USER = "root";
	private final static String PASSWORD = "herber12";
	private static Connection dbConnection;

	/**
	 *  Tests for the presence of JDBC driver.
	 *
	 * @return    Returns true if JDBC driver found, false otherwise.
	 */
	public static boolean testDriver() {
		String driverName = "com.mysql.jdbc.Driver";

		try {
			Class.forName( driverName ).newInstance();
			System.out.println( "MySQL driver found" );
			return true;
		}
		catch ( ClassNotFoundException e ) {
			System.err.println( "Could not find MySQL JDBC driver: " + driverName );
			e.printStackTrace();
			return false;
		}
		catch ( InstantiationException e ) {
			System.err.println( "Error creating new instace of MySQL JDBC driver: " + driverName );
			e.printStackTrace();
			return false;
		}
		catch ( IllegalAccessException e ) {
			System.err.println( "Error accessing MySQL JDBC driver: " + driverName );
			e.printStackTrace();
			return false;
		}
	}

	/**
	 *  Connects to the MySQL database.
	 *
	 * @return    Returns true if connection is successful, false otherwise.
	 */
	public static boolean connectToDB() {
		if ( !testDriver() ) {
			return false;
		}

		try {
			dbConnection = DriverManager.getConnection( URL, USER, PASSWORD );
			System.out.println( "Connection established to " + URL + "\n" );
			return true;
		}
		catch ( SQLException e ) {
			System.err.println( "Connection couldn't be established to " + URL + "\n" );
			e.printStackTrace();
			return false;
		}
	}

	/**
	 *  Executes an SQL UPDATE or INSERT query.
	 *
	 * @param  query  The query to be performed.
	 * @return        Returns true if query executed succesfully, false otherwise.
	 */
	public static boolean executeUpdate( String query ) {
		try {
			Statement sqlStatement = dbConnection.createStatement();
			sqlStatement.execute( query );
			sqlStatement.close();
			return true;
		}
		catch ( SQLException e ) {
			System.err.println( "Error executing sql statement" );
			e.printStackTrace();
			return false;
		}
	}

	/**
	 *  Executes an SQL SELECT query.
	 *
	 * @param  query  The query to be performed.
	 * @return        Returns the result set of the query, null if the query is invalid.
	 */
	public static ResultSet executeQuery( String query ) {
		try {
			Statement sqlStatement = dbConnection.createStatement();
			ResultSet sqlResult = sqlStatement.executeQuery( query );
			return sqlResult;
		}
		catch ( SQLException e ) {
			System.err.println( "Error executing sql statement" );
			e.printStackTrace();
			return null;
		}
	}

	/**
	 *  Gets the number of results in the given result set.
	 *
	 * @param  sqlResult  Description of the Parameter
	 * @return            Returns the number of results in the result set, -1 if the result set is null.
	 */
	public static int getNumResults( ResultSet sqlResult ) {
		if ( sqlResult == null ) {
			return -1;
		}

		try {
			sqlResult.last();
			int num = sqlResult.getRow();
			sqlResult.beforeFirst();
			return num;
		}
		catch ( SQLException e ) {
			System.err.println( "Could not get number of results from result set." );
			e.printStackTrace();
			return -1;
		}
	}

	/**
	 *  Cleans the title of a movie, so that they are consitent.  Gets rid of quotations in the title, extra information
	 *  at the end of the title that IMDB provides (including the year, that is stored separately).  Takes the 'A', 'An' and 'The'
	 *  from the end of a title, and puts it at the beginning where it should be.  This is to make the titles match the EachMovie data set.
	 *
	 * @param  movieTitle  The original movie title.
	 * @return             Returns the 'clean' movie title.
	 */
	public static String cleanMovieTitle( String movieTitle ) {
		int pos1 = movieTitle.indexOf( "(TV)" );
		int pos2 = movieTitle.indexOf( "(V)" );
		int pos3 = movieTitle.indexOf( "(VG)" );
		int pos4 = movieTitle.indexOf( "(mini)" );

		// if any of these strings presesent, we don't want that title
		if ( ( pos1 >= 0 ) || ( pos2 >= 0 ) || ( pos3 >= 0 ) || ( pos4 >= 0 ) ) {
			System.out.println( "\t*** Throwing out " + movieTitle );
			return "";
		}
		else {
			// get rid of quotations
			String wordsInTitle[] = movieTitle.split( "'" );
			String newMovieTitle = "";
			for ( int i = 0; i < wordsInTitle.length; i++ ) {
				newMovieTitle += wordsInTitle[i].trim();
			}

			// get rid of garbage at end of title
			int pos = newMovieTitle.indexOf( "(" );
			if ( pos > 0 ) {
				newMovieTitle = newMovieTitle.substring( 0, pos - 1 ).trim();
			}
			// put 'An' at beginning to match EachMovie data
			if ( newMovieTitle.indexOf( ", An" ) >= 0 ) {
				newMovieTitle = newMovieTitle.substring( 0, newMovieTitle.length() - 4 );
				newMovieTitle = "An " + newMovieTitle;
			}
			// put 'A' at beginning to match EachMovie data
			if ( newMovieTitle.indexOf( ", A" ) >= 0 ) {
				newMovieTitle = newMovieTitle.substring( 0, newMovieTitle.length() - 3 );
				newMovieTitle = "A " + newMovieTitle;
			}
			// put 'The' at beginning to match EachMovie data
			if ( newMovieTitle.indexOf( ", The" ) >= 0 ) {
				newMovieTitle = newMovieTitle.substring( 0, newMovieTitle.length() - 5 );
				newMovieTitle = "The " + newMovieTitle;
			}
			// replace all '&' with 'and'
			return newMovieTitle.replaceAll( "&", "and" );
		}
	}

	/**
	 *  Cleans the actor's name.  This is to make sure they are consistent for searching later on.
	 *Names are given as last, first.  This function takes out the comma and puts the name in
	 *proper order.  It also takes Jr., Sr., etc. and puts it in the right place at the end.
	 *
	 * @param  actorName  The original actor's name.
	 * @return            Returns the 'clean' version of the actor's name.
	 */
	public static String cleanActorName( String actorName ) {
		String newActorName = "";
		String wordsInName[] = actorName.split( "," );

		// actor has more than one word in name (John Wayne)
		if ( wordsInName.length > 1 ) {
			String first = wordsInName[1].trim();
			String last = wordsInName[0].trim();
			// check for Jr, I, II, etc. after name
			int pos = first.indexOf( "(" );
			if ( pos > 0 ) {
				last += " " + first.substring( pos, first.length() ).trim();
				first = first.substring( 0, pos - 1 ).trim();
			}

			// put first & last name in correct order
			newActorName = ( new String( first + " " + last ) ).trim();
		}
		// actor has only one word in name (Madonna)
		else {
			newActorName = wordsInName[0].trim();
		}

		// get rid of quotations
		String words[] = newActorName.split( "'" );
		newActorName = "";
		for ( int i = 0; i < words.length; i++ ) {
			newActorName += words[i].trim();
		}

		return newActorName;
	}

	/**
	 *  Cleans the plot description of the movie.  Removes all quotations.
	 *
	 * @param  moviePlot  The original plot of the movie.
	 * @return            Returns the 'clean' version of the movie's plot.
	 */
	public static String cleanPlot( String moviePlot ) {
		// get rid of quotations
		String wordsInPlot[] = moviePlot.split( "'" );
		String newPlot = "";
		for ( int i = 0; i < wordsInPlot.length; i++ ) {
			newPlot += wordsInPlot[i].trim();
		}
		return newPlot;
	}
}

