import java.awt.*;
import java.awt.event.*;
import java.math.BigInteger;
import javax.swing.*;
import javax.swing.border.*;
import java.awt.Graphics2D;
import java.awt.geom.Rectangle2D;

/**
Date: 1/23/04
Author: Anastasia Mironova

Descroption of the Class

The purpose of this class is to represent the landscape model for the PMPSim system
*/

public class Landscape extends JPanel implements MouseListener, MouseMotionListener{
	//set to true if the landscape is in the stage of being modified
	private boolean modificationsEnabled = false;
	private boolean mousePressed = false;
	//there is no color initially
	private Color currentColor;
	
	public static final Color defaultColor = new Color(204, 204, 204);
	public static final Color elevationContourColor = new Color(204, 153, 0);
	public static final Color waterPatchColor = new Color(0, 204, 153);
	public static final Color specificPatchColor = new Color(102, 255, 153);
	public static final Color riparianHabitatColor = new Color(51, 204, 0);
	public static final Color forestHabitatColor = new Color(0, 102, 0);
	
	private LandscapeContainerPanel parent = null;
	private Dimension gridDimensions = new Dimension(1, 1);
	private LandscapeSegment[][] segmentsArray;
	
	/**
	Constructor that creates a Landscape object with a specific number of rows and columns
	*/
	public Landscape(Dimension landscapeGridDimensions){
		super();
		
		gridDimensions = landscapeGridDimensions;
		
		segmentsArray = new LandscapeSegment[(int)(gridDimensions.getHeight())][(int)(gridDimensions.getWidth())];
		
		setLayout(new GridLayout((int)(gridDimensions.getHeight()), (int)(gridDimensions.getWidth())));
		setBorder(new BevelBorder(BevelBorder.LOWERED));
		
		//populate the landscape panel with a set of landscape segment panels
		for (int i = 1; i <= gridDimensions.getHeight(); i++){
			for (int j = 1; j <= gridDimensions.getWidth(); j++){
				segmentsArray[i-1][j-1] = new LandscapeSegment(this, new Location(j, i), null, null, null);
				this.add(segmentsArray[i-1][j-1]);
				segmentsArray[i-1][j-1].setPreferredSize(LandscapeSegment.getDimensions());
			}
		}
		
		/**
		This is a mouse listener to allow the user to swap panels from the main landscape to the active part of landscape
		*/
		addMouseListener(this);
		addMouseMotionListener(this);
	}
	
	
	
	/*
	sets the parent panel in order to enable the panel swap
	*/
	public void setParentPanel(LandscapeContainerPanel p){
		parent = p;
	}
	
	/*
	Returns the dimensions of the landscape
	*/
	public Dimension getGridDimensions(){
		return gridDimensions;
	}
	/*
	Sets the dimensions of the landscape
	*/
	public void setGridDimensions(Dimension d){
		gridDimensions = d;
	}
	/*
	This method sets the LandscapeSegment object at the specified location
	*/
	public void setSegment(Location location, LandscapeSegment s){
		(segmentsArray[location.getYCoordinate()-1][location.getXCoordinate()-1]) = s;
	}
	/*
	This method returns the LandscapeSegment object at the specified location
	*/
	public LandscapeSegment getSegment(Location location){
		return (segmentsArray[location.getYCoordinate()-1][location.getXCoordinate()-1]);
	}
	
	/*
	This method returns a copy of segment at the specified location
	*/
	public LandscapeSegment getCopyOfSegment(Location location){
		return (segmentsArray[location.getYCoordinate()-1][location.getXCoordinate()-1]).copy();
	}
	
	/*
	Enables/disables customization of the main landscape
	*/
	public void enableModifications(boolean b){
		modificationsEnabled = b;
	}
	
	//checks if the modifications are enabled/disabled
	public boolean areModificationsEnabled(){
		return modificationsEnabled;
	}
	
	public void mousePressed(MouseEvent event) {
	}
	
	public void mouseReleased(MouseEvent evt) {
	}
	
	public void mouseDragged(MouseEvent evt) {
	}
	
	public void mouseEntered(MouseEvent evt) {
	}
	
	public void mouseExited(MouseEvent evt) { 
	}
	
	//handles single click mouse events when the landscape is in locked mode, in which case
	//this action results in swaping of the segment selected to the magnified setup
	public void mouseClicked(MouseEvent event){
		System.out.println("Performing panel swap...");
		if ((parent != null)&& (!modificationsEnabled)){
			parent.performPanelSwap(new Location( 1 + (int)(event.getX()/((getSize().getWidth())/(gridDimensions.getWidth()))),
			                                      1 + (int)(event.getY()/((getSize().getHeight())/(gridDimensions.getHeight())))));
		}
	}
	
	public void mouseMoved(MouseEvent evt) {  
	}
	
	//depending upon the currently selected color, the distribution of 
	// appropriate characteristics within the landscape model occurs
	public void setMouseToPressed(boolean b){
		if (!b){
			if (currentColor == Landscape.elevationContourColor){
				try{
					//distributes the specified value of elevation to the specific segments
					int temp = Integer.parseInt(new JOptionPane().showInputDialog("Specify elevation in meters: "));
					distributeElevation(temp);
					System.out.println("Elevation distributed!");
				} catch(NumberFormatException e){
					System.out.println("Invalid entry for elevation!");
				}
			} else if (currentColor == Landscape.waterPatchColor){
				distributeWaterPatch();
				System.out.println("Wetland patch distributed!");
			} else if (currentColor == Landscape.defaultColor){
				viewDefault();
				System.out.println("View has been reset!");
			}else if (currentColor == Landscape.riparianHabitatColor){
				distributeRiparianPatch();
				System.out.println("Riparian Patch Distributed!");
			}else if (currentColor == Landscape.forestHabitatColor){
				distributeForestPatch();
				System.out.println("Forest patch distributed!");
			}/* else if (currentColor == Landscape.specificPatchColor){
				try{
					//distributes the specified value of elevation to the specific segments
					int temp = Integer.parseInt(new JOptionPane().showInputDialog("Specify elevation in meters: "));
					distributeElevation(temp);
					System.out.println("Elevation distributed!");
				} catch(NumberFormatException e){
					System.out.println("Invalid entry for elevation!");
				}
			}*/
			enableModifications(false);
		}
		mousePressed = b;
	}
	
	public boolean isMousePressed(){
		return mousePressed;
	}
	
	public void setCurrentColor(Color c){
		currentColor = c;
	}
	
	public Color getCurrentColor(){
		return currentColor;
	}
	
	//distributing elevation contour within the landscape model
	public void distributeElevation(int elevationValue){
		for (int i = 1; i <= gridDimensions.getHeight(); i++){
			for (int j = 1; j <= gridDimensions.getWidth(); j++){
				if (segmentsArray[i-1][j-1].getBackground() == currentColor){
					
					System.out.println("Distributing elevation...");
					
					(segmentsArray[i-1][j-1]).setElevation(elevationValue);
					//fade the selected contour
					(segmentsArray[i-1][j-1]).setBackground(new java.awt.Color(204, 204, 204));
				}
			}
		}
	}
	
	//viewing elevation contours within the landscape model
	public void viewElevationContours(){
		for (int i = 1; i <= gridDimensions.getHeight(); i++){
			for (int j = 1; j <= gridDimensions.getWidth(); j++){
				if (segmentsArray[i-1][j-1].getElevation() != 0){
					
					System.out.println("Extracting elevation contours...");
					
					(segmentsArray[i-1][j-1]).setBackground(elevationContourColor);
				}
			}
		}
	}
	
	//distributing wetland areas within the landscape model
	public void distributeWaterPatch(){
		for (int i = 1; i <= gridDimensions.getHeight(); i++){
			for (int j = 1; j <= gridDimensions.getWidth(); j++){
				if (segmentsArray[i-1][j-1].getBackground() == currentColor){
					
					System.out.println("Distributing wetalnd patch...");
					
					(segmentsArray[i-1][j-1]).makeWaterPatch();
					//fade the selected contour
					(segmentsArray[i-1][j-1]).setBackground(new java.awt.Color(204, 204, 204));
				}
			}
		}
	}
	
	//viewing the wetland areas specified for the landscape model so far
	public void viewWaterPatches(){
		for (int i = 1; i <= gridDimensions.getHeight(); i++){
			for (int j = 1; j <= gridDimensions.getWidth(); j++){
				if (segmentsArray[i-1][j-1].isWaterPatch() == true){
					
					System.out.println("Extracting water patch...");
					
					(segmentsArray[i-1][j-1]).setBackground(waterPatchColor);
				}
			}
		}
	}
	
	//distributing a riparian patch within the landscape model
	public void distributeRiparianPatch(){
		for (int i = 1; i <= gridDimensions.getHeight(); i++){
			for (int j = 1; j <= gridDimensions.getWidth(); j++){
				if (segmentsArray[i-1][j-1].getBackground() == currentColor){
					
					System.out.println("Distributing riparian patch...");
					
					(segmentsArray[i-1][j-1]).makeRiparianPatch();
					//fade the selected contour
					(segmentsArray[i-1][j-1]).setBackground(new java.awt.Color(204, 204, 204));
				}
			}
		}
	}
	
	//viewing riparian patches specified within the landscape model
	public void viewRiparianPatches(){
		for (int i = 1; i <= gridDimensions.getHeight(); i++){
			for (int j = 1; j <= gridDimensions.getWidth(); j++){
				if (segmentsArray[i-1][j-1].isRiparianPatch() == true){
					
					System.out.println("Extracting riparian patch...");
					
					(segmentsArray[i-1][j-1]).setBackground(riparianHabitatColor);
				}
			}
		}
	}
	
	//distributing a forest patch within the landscape model
	public void distributeForestPatch(){
		for (int i = 1; i <= gridDimensions.getHeight(); i++){
			for (int j = 1; j <= gridDimensions.getWidth(); j++){
				if (segmentsArray[i-1][j-1].getBackground() == currentColor){
					
					System.out.println("Distributing forest patch...");
					
					(segmentsArray[i-1][j-1]).makeForestPatch();
					//fade the selected contour
					(segmentsArray[i-1][j-1]).setBackground(new java.awt.Color(204, 204, 204));
				}
			}
		}
	}
	
	//viewing the forest patches within the landscape model
	public void viewForestPatches(){
		for (int i = 1; i <= gridDimensions.getHeight(); i++){
			for (int j = 1; j <= gridDimensions.getWidth(); j++){
				if (segmentsArray[i-1][j-1].isForestPatch() == true){
					
					System.out.println("Extracting forest patch...");
					
					(segmentsArray[i-1][j-1]).setBackground(forestHabitatColor);
				}
			}
		}
	}
	
	//reset all the segments and hide all the associated properties
	public void viewDefault(){
		for (int i = 1; i <= gridDimensions.getHeight(); i++){
			for (int j = 1; j <= gridDimensions.getWidth(); j++){
				(segmentsArray[i-1][j-1]).setBackground(defaultColor);
			}
		}
	}
}
